/* SPDX-License-Identifier: GPL-2.0+ */
/*
 * Internal header file for scenes
 *
 * Copyright 2022 Google LLC
 * Written by Simon Glass <sjg@chromium.org>
 */

#ifndef __SCENE_INTERNAL_H
#define __SCENE_INTERNAL_H

/**
 * expo_lookup_scene_id() - Look up a scene ID
 *
 * @exp: Expo to use
 * @id: scene ID to look up
 * Returns: Scene for that ID, or NULL if none
 */
struct scene *expo_lookup_scene_id(struct expo *exp, uint scene_id);

/**
 * resolve_id() - Automatically allocate an ID if needed
 *
 * @exp: Expo to use
 * @id: ID to use, or 0 to auto-allocate one
 * @return: Either @id, or the auto-allocated ID
 */
uint resolve_id(struct expo *exp, uint id);

/**
 * scene_obj_find() - Find an object in a scene
 *
 * Note that @type is used to restrict the search when the object type is known.
 * If any type is acceptable, set @type to SCENEOBJT_NONE
 *
 * @scn: Scene to search
 * @id: ID of object to find
 * @type: Type of the object, or SCENEOBJT_NONE to match any type
 */
void *scene_obj_find(struct scene *scn, uint id, enum scene_obj_t type);

/**
 * scene_obj_add() - Add a new object to a scene
 *
 * @scn: Scene to update
 * @name: Name to use (this is allocated by this call)
 * @id: ID to use for the new object (0 to allocate one)
 * @type: Type of object to add
 * @size: Size to allocate for the object, in bytes
 * @objp: Returns a pointer to the new object (must not be NULL)
 * Returns: ID number for the object (generally @id), or -ve on error
 */
int scene_obj_add(struct scene *scn, const char *name, uint id,
		  enum scene_obj_t type, uint size, struct scene_obj **objp);

/**
 * scene_menu_arrange() - Set the position of things in the menu
 *
 * This updates any items associated with a menu to make sure they are
 * positioned correctly relative to the menu. It also selects the first item
 * if not already done
 *
 * @scn: Scene to update
 * @menu: Menu to process
 */
int scene_menu_arrange(struct scene *scn, struct scene_obj_menu *menu);

/**
 * scene_menu_send_key() - Send a key to a menu for processing
 *
 * @scn: Scene to use
 * @menu: Menu to use
 * @key: Key code to send (KEY_...)
 * @event: Place to put any event which is generated by the key
 * @return 0 if OK, -ENOTTY if there is no current menu item, other -ve on other
 *	error
 */
int scene_menu_send_key(struct scene *scn, struct scene_obj_menu *menu, int key,
			struct expo_action *event);

/**
 * scene_menu_destroy() - Destroy a menu in a scene
 *
 * @scn: Scene to destroy
 */
void scene_menu_destroy(struct scene_obj_menu *menu);

/**
 * scene_menu_display() - Display a menu as text
 *
 * @menu: Menu to display
 * @return 0 if OK, -ENOENT if @id is invalid
 */
int scene_menu_display(struct scene_obj_menu *menu);

/**
 * scene_destroy() - Destroy a scene and all its memory
 *
 * @scn: Scene to destroy
 */
void scene_destroy(struct scene *scn);

/**
 * scene_render() - Render a scene
 *
 * This is called from expo_render()
 *
 * @scn: Scene to render
 * Returns: 0 if OK, -ve on error
 */
int scene_render(struct scene *scn);

/**
 * scene_send_key() - set a keypress to a scene
 *
 * @scn: Scene to receive the key
 * @key: Key to send (KEYCODE_UP)
 * @event: Returns resulting event from this keypress
 * Returns: 0 if OK, -ve on error
 */
int scene_send_key(struct scene *scn, int key, struct expo_action *event);

#endif /* __SCENE_INTERNAL_H */
