/*
 * Print out node ID and node string for each node in the inventory
 *
 * The first argument is the torus dimension. This information is not contained
 * in the BASIL XML.
 *
 * usage: $0 2 | 3
 */
#include "basil_alps.h"
#include "basil_torus.h"

extern struct torus_info *torus_from_inventory(const struct basil_inventory *);

int main(int ac, char **av)
{
	struct basil_inventory *inv;
	const struct basil_node *node;
	struct torus_info *ti;
	enum basil_version version = get_basil_version();

	if (ac == 1 || (strcmp(av[1], "2") && strcmp(av[1], "3")))
		errx(1, "use with argument of '2' or '3' to specifiy torus dimension");

	inv = get_full_inventory(version);
	if (inv == NULL)
		err(1, "allocation of inventory data failed");

	ti = torus_from_inventory(inv);
	if (ti == NULL)
		errx(1, "can not determine Torus information");

	if (*av[1]  == '2') {
		/* XT5m/XE6m systems have no connections in the X dimension */
		ti->x_max = 0;
		ti->dimension = TORUS_2D;
	} else if (*av[1]  == '3') {
		ti->dimension = TORUS_3D;
	} else {
		errx(1, "unsupported argument '%s'", *av);
	}

	for (node = inv->f->node_head; node; node = node->next) {
		struct nodecoord *nc = nid2node(node->node_id, ti);
		struct torus_coord tc;
		const char *nstr;

		if (node->arch != BNA_XT)
			errx(1, "nid%05u (%s) has unsupported architecture %s",
			      node->node_id, node->name, nam_arch[node->arch]);
		/*
		 * When the node is not 'UP', its coordinates may be set to
		 * NULL in the SDB database.
		 */
		if (node->state != BNS_UP)
			continue;

		if (!is_valid_nodecoord(nc, ti))
			errx(1, "nid%05u (%s) is invalid for this system",
			      node->node_id, node->name);

		nstr = nodecoord2string(nc);
		if (strcmp(node->name, nstr))
			errx(1, "Decoding of nid%05u failed: %s does not match %s",
				node->node_id, nstr, node->name);

		tc = nid2toruscoord(nc, ti);
		printf("%6u %s  %3d %3d %3d\n", node->node_id, nstr,
			tc.x, tc.y, tc.z);
	}
	free(ti);
	free_inv(inv);

	return EXIT_SUCCESS;
}
