% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesfactor_models.R
\name{bayesfactor_models}
\alias{bayesfactor_models}
\alias{bf_models}
\alias{bayesfactor_models.default}
\alias{update.bayesfactor_models}
\alias{as.matrix.bayesfactor_models}
\title{Bayes Factors (BF) for model comparison}
\usage{
bayesfactor_models(..., denominator = 1, verbose = TRUE)

bf_models(..., denominator = 1, verbose = TRUE)

\method{bayesfactor_models}{default}(..., denominator = 1, verbose = TRUE)

\method{update}{bayesfactor_models}(object, subset = NULL, reference = NULL, ...)

\method{as.matrix}{bayesfactor_models}(x, ...)
}
\arguments{
\item{...}{Fitted models (see details), all fit on the same data, or a single
\code{BFBayesFactor} object (see 'Details'). Ignored in \code{as.matrix()},
\code{update()}. If the following named arguments are present, they are passed
to \link[insight:get_loglikelihood]{insight::get_loglikelihood} (see details):
\itemize{
\item \code{estimator} (defaults to \code{"ML"})
\item \code{check_response}  (defaults to \code{FALSE})
}}

\item{denominator}{Either an integer indicating which of the models to use as
the denominator, or a model to be used as a denominator. Ignored for
\code{BFBayesFactor}.}

\item{verbose}{Toggle off warnings.}

\item{object, x}{A \code{\link[=bayesfactor_models]{bayesfactor_models()}} object.}

\item{subset}{Vector of model indices to keep or remove.}

\item{reference}{Index of model to reference to, or \code{"top"} to
reference to the best model, or \code{"bottom"} to reference to the worst
model.}
}
\value{
A data frame containing the models' formulas (reconstructed fixed and
random effects) and their \code{log(BF)}s  (Use \code{as.numeric()} to extract the
non-log Bayes factors; see examples), that prints nicely.
}
\description{
This function computes or extracts Bayes factors from fitted models.
\cr \cr
The \verb{bf_*} function is an alias of the main function.
}
\details{
If the passed models are supported by \pkg{insight} the DV of all models will be tested for equality
(else this is assumed to be true), and the models' terms will be extracted (allowing for follow-up
analysis with \code{bayesfactor_inclusion}).
\itemize{
\item For \code{brmsfit} or \code{stanreg} models, Bayes factors are computed using the \CRANpkg{bridgesampling} package.
\itemize{
\item \code{brmsfit} models must have been fitted with \code{save_pars = save_pars(all = TRUE)}.
\item \code{stanreg} models must have been fitted with a defined \code{diagnostic_file}.
}
\item For \code{BFBayesFactor}, \code{bayesfactor_models()} is mostly a wraparound \code{BayesFactor::extractBF()}.
\item For all other model types, Bayes factors are computed using the BIC approximation.
Note that BICs are extracted from using \link[insight:get_loglikelihood]{insight::get_loglikelihood}, see documentation
there for options for dealing with transformed responses and REML estimation.
}

In order to correctly and precisely estimate Bayes factors, a rule of thumb
are the 4 P's: \strong{P}roper \strong{P}riors and \strong{P}lentiful
\strong{P}osteriors. How many? The number of posterior samples needed for
testing is substantially larger than for estimation (the default of 4000
samples may not be enough in many cases). A conservative rule of thumb is to
obtain 10 times more samples than would be required for estimation
(\cite{Gronau, Singmann, & Wagenmakers, 2017}). If less than 40,000 samples
are detected, \code{bayesfactor_models()} gives a warning.
\cr \cr
See also \href{https://easystats.github.io/bayestestR/articles/bayes_factors.html}{the Bayes factors vignette}.
}
\note{
There is also a \href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method}
implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{Interpreting Bayes Factors}{

A Bayes factor greater than 1 can be interpreted as evidence against the
null, at which one convention is that a Bayes factor greater than 3 can be
considered as "substantial" evidence against the null (and vice versa, a
Bayes factor smaller than 1/3 indicates substantial evidence in favor of the
null-model) (\cite{Wetzels et al. 2011}).
}

\examples{
# With lm objects:
# ----------------
lm1 <- lm(mpg ~ 1, data = mtcars)
lm2 <- lm(mpg ~ hp, data = mtcars)
lm3 <- lm(mpg ~ hp + drat, data = mtcars)
lm4 <- lm(mpg ~ hp * drat, data = mtcars)
(BFM <- bayesfactor_models(lm1, lm2, lm3, lm4, denominator = 1))
# bayesfactor_models(lm2, lm3, lm4, denominator = lm1) # same result
# bayesfactor_models(lm1, lm2, lm3, lm4, denominator = lm1) # same result


update(BFM, reference = "bottom")
as.matrix(BFM)
as.numeric(BFM)


lm2b <- lm(sqrt(mpg) ~ hp, data = mtcars)
# Set check_response = TRUE for transformed responses
bayesfactor_models(lm2b, denominator = lm2, check_response = TRUE)

\dontrun{
# With lmerMod objects:
# ---------------------
if (require("lme4")) {
  lmer1 <- lmer(Sepal.Length ~ Petal.Length + (1 | Species), data = iris)
  lmer2 <- lmer(Sepal.Length ~ Petal.Length + (Petal.Length | Species), data = iris)
  lmer3 <- lmer(Sepal.Length ~ Petal.Length + (Petal.Length | Species) + (1 | Petal.Width),
    data = iris
  )
  bayesfactor_models(lmer1, lmer2, lmer3,
    denominator = 1,
    estimator = "REML"
  )
}

# rstanarm models
# ---------------------
# (note that a unique diagnostic_file MUST be specified in order to work)
if (require("rstanarm")) {
  stan_m0 <- stan_glm(Sepal.Length ~ 1,
    data = iris,
    family = gaussian(),
    diagnostic_file = file.path(tempdir(), "df0.csv")
  )
  stan_m1 <- stan_glm(Sepal.Length ~ Species,
    data = iris,
    family = gaussian(),
    diagnostic_file = file.path(tempdir(), "df1.csv")
  )
  stan_m2 <- stan_glm(Sepal.Length ~ Species + Petal.Length,
    data = iris,
    family = gaussian(),
    diagnostic_file = file.path(tempdir(), "df2.csv")
  )
  bayesfactor_models(stan_m1, stan_m2, denominator = stan_m0, verbose = FALSE)
}


# brms models
# --------------------
# (note the save_pars MUST be set to save_pars(all = TRUE) in order to work)
if (require("brms")) {
  brm1 <- brm(Sepal.Length ~ 1, data = iris, save_pars = save_pars(all = TRUE))
  brm2 <- brm(Sepal.Length ~ Species, data = iris, save_pars = save_pars(all = TRUE))
  brm3 <- brm(
    Sepal.Length ~ Species + Petal.Length,
    data = iris,
    save_pars = save_pars(all = TRUE)
  )

  bayesfactor_models(brm1, brm2, brm3, denominator = 1, verbose = FALSE)
}


# BayesFactor
# ---------------------------
if (require("BayesFactor")) {
  data(puzzles)
  BF <- anovaBF(RT ~ shape * color + ID,
    data = puzzles,
    whichRandom = "ID", progress = FALSE
  )
  BF
  bayesfactor_models(BF) # basically the same
}
}

}
\references{
\itemize{
\item Gronau, Q. F., Singmann, H., & Wagenmakers, E. J. (2017). Bridgesampling: An R package for estimating
normalizing constants. arXiv preprint arXiv:1710.08162.
\item Kass, R. E., and Raftery, A. E. (1995). Bayes Factors. Journal of the American Statistical Association,
90(430), 773-795.
\item Robert, C. P. (2016). The expected demise of the Bayes factor. Journal of Mathematical Psychology,
72, 33–37.
\item Wagenmakers, E. J. (2007). A practical solution to the pervasive problems of p values.
Psychonomic bulletin & review, 14(5), 779-804.
\item Wetzels, R., Matzke, D., Lee, M. D., Rouder, J. N., Iverson, G. J., and Wagenmakers, E.-J. (2011).
Statistical Evidence in Experimental Psychology: An Empirical Comparison Using 855 t Tests.
Perspectives on Psychological Science, 6(3), 291–298. \doi{10.1177/1745691611406923}
}
}
\author{
Mattan S. Ben-Shachar
}
