/*
 * Copyright 2022-2023 Soren Stoutner <soren@stoutner.com>.
 *
 * This file is part of Privacy Browser PC <https://www.stoutner.com/privacy-browser-pc>.
 *
 * Privacy Browser PC is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Privacy Browser PC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Privacy Browser PC.  If not, see <http://www.gnu.org/licenses/>.
 */

// Application headers.
#include "UrlRequestInterceptor.h"

// KDE Framework headers.
#include <KLocalizedString>

// Construct the class.
UrlRequestInterceptor::UrlRequestInterceptor(QObject *parentObjectPointer) : QWebEngineUrlRequestInterceptor(parentObjectPointer) {}

void UrlRequestInterceptor::interceptRequest(QWebEngineUrlRequestInfo &urlRequestInfo)
{
    // Handle the request according to the resource type.
    switch (urlRequestInfo.resourceType())
    {
        // A naughty HTTP ping request.
        case QWebEngineUrlRequestInfo::ResourceTypePing:
        {
            // Block the HTTP ping request.
            urlRequestInfo.block(true);

            // Display the HTTP Ping blocked dialog.
            emit displayHttpPingDialog(urlRequestInfo.requestUrl().toString());

            break;
        }

        default:
        {
            // Do nothing.
            break;
        }
    }

    // Handle the request according to the navigation type.
    switch (urlRequestInfo.navigationType())
    {
        case QWebEngineUrlRequestInfo::NavigationTypeLink:
        case QWebEngineUrlRequestInfo::NavigationTypeTyped:
        case QWebEngineUrlRequestInfo::NavigationTypeBackForward:
        // case QWebEngineUrlRequestInfo::NavigationTypeReload:  This can be uncommented once https://redmine.stoutner.com/issues/821 has been fixed.
        case QWebEngineUrlRequestInfo::NavigationTypeRedirect:
        {
            // Only check the hosts if the main URL is changing.
            if (urlRequestInfo.resourceType() == QWebEngineUrlRequestInfo::ResourceTypeMainFrame)
            {
                // Get the hosts.
                QString requestingHost = urlRequestInfo.initiator().host();
                QString requestedHost = urlRequestInfo.requestUrl().host();

                // Reapply the domain settings if the host is changing.
                if (requestingHost != requestedHost)
                    emit applyDomainSettings(requestedHost);
            }

            break;
        }

        default:
            // Do nothing.
            break;
    }
}
