///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Window3D.h>
#include <mesh/tri/TriMesh.h>

using namespace Mesh;

namespace Core {

#define CLIP_RIGHT_BIT   0x01
#define CLIP_LEFT_BIT    0x02
#define CLIP_TOP_BIT     0x04
#define CLIP_BOTTOM_BIT  0x08
#define CLIP_NEAR_BIT    0x10
#define CLIP_FAR_BIT     0x20
#define CLIP_USER_BIT    0x40
#define CLIP_ALL_BITS    0x3F

enum {
	X = 0, Y = 1, Z = 2, W = 3
};

/******************************************************************************
* Clips a line segment againts the view volume.
******************************************************************************/
inline void ClipLine(unsigned int clipMask, Vector4& o, Vector4& i)
{
	FloatType dX, dY, dZ, dW, t;

	// Clip against +X side
	if((clipMask & CLIP_RIGHT_BIT) != 0) {
		dX = o[X] - i[X];
		dW = o[W] - i[W];
		t = (i[X] - i[W]) / (dW-dX);
		o[X] = i[X] + t * dX;
		o[Y] = i[Y] + t * (o[Y] - i[Y]);
		o[Z] = i[Z] + t * (o[Z] - i[Z]);
		o[W] = i[W] + t * dW;
	}

	// Clip against -X side
	if((clipMask & CLIP_LEFT_BIT) != 0) {
		dX = o[X] - i[X];
		dW = o[W] - i[W];
		t = -(i[X] + i[W]) / (dW+dX);
		o[X] = i[X] + t * dX;
		o[Y] = i[Y] + t * (o[Y] - i[Y]);
		o[Z] = i[Z] + t * (o[Z] - i[Z]);
		o[W] = i[W] + t * dW;
	}

	// Clip against +Y side
	if((clipMask & CLIP_TOP_BIT) != 0) {
		dY = o[Y] - i[Y];
		dW = o[W] - i[W];
		t = (i[Y] - i[W]) / (dW-dY);
		o[X] = i[X] + t * (o[X] - i[X]);
		o[Y] = i[Y] + t * dY;
		o[Z] = i[Z] + t * (o[Z] - i[Z]);
		o[W] = i[W] + t * dW;
	}

	// Clip against -Y side
	if((clipMask & CLIP_BOTTOM_BIT) != 0) {
		dY = o[Y] - i[Y];
		dW = o[W] - i[W];
		t = -(i[Y] + i[W]) / (dW+dY);
		o[X] = i[X] + t * (o[X] - i[X]);
		o[Y] = i[Y] + t * dY;
		o[Z] = i[Z] + t * (o[Z] - i[Z]);
		o[W] = i[W] + t * dW;
	}

	// Clip against +Z side
	if((clipMask & CLIP_FAR_BIT) != 0) {
		dZ = o[Z] - i[Z];
		dW = o[W] - i[W];
		t = (i[Z] - i[W]) / (dW-dZ);
		o[X] = i[X] + t * (o[X] - i[X]);
		o[Y] = i[Y] + t * (o[Y] - i[Y]);
		o[Z] = i[Z] + t * dZ;
		o[W] = i[W] + t * dW;
	}

	// Clip against -Z side
	if((clipMask & CLIP_NEAR_BIT) != 0) {
		dZ = o[Z] - i[Z];
		dW = o[W] - i[W];
		t = -(i[Z] + i[W]) / (dW+dZ);
		o[X] = i[X] + t * (o[X] - i[X]);
		o[Y] = i[Y] + t * (o[Y] - i[Y]);
		o[Z] = i[Z] + t * dZ;
		o[W] = i[W] + t * dW;
	}
}

/******************************************************************************
* Computes the distance between a point and a line segment.
******************************************************************************/
inline int SquaredDistToLine(const Point2I& pt, const Point2I& linePoint1, const Point2I& linePoint2)
{
	Vector2I v = linePoint1 - pt;
	if(linePoint1 == linePoint2)
		return LengthSquared(v);
	Vector2I delta = linePoint1 - linePoint2;
	int d = LengthSquared(delta);
	int r = DotProduct(v, delta);
	if(r<=0)
		return LengthSquared(v);
	else if(r>=d)
		return DistanceSquared(linePoint2, pt);
	else {
		return (int)(square((uint64_t)(v.X*delta.Y - v.Y*delta.X)) / d);
	}
}

/******************************************************************************
* Tests if the line segment crosses the rectangle region.
******************************************************************************/
inline bool LineCrossesRect(const QRect& rect, const Point2I& linePoint1, const Point2I& linePoint2)
{
	if(rect.contains(linePoint1.X, linePoint1.Y)) return true;
	if(rect.contains(linePoint2.X, linePoint2.Y)) return true;
	int left = rect.x();
	int top = rect.y();
	int right = rect.right();
	int bottom = rect.bottom();
	if(linePoint1.Y != linePoint2.Y) {
		if((linePoint1.Y<bottom)^(linePoint2.Y<bottom)) {
			int x = linePoint1.X + (linePoint2.X - linePoint1.X) * (bottom - linePoint1.Y) / (linePoint2.Y - linePoint1.Y);
			if(x>=left && x<=right) return true;
		}
		if((linePoint1.Y<top)^(linePoint2.Y<top)) {
			int x = linePoint1.X + (linePoint2.X - linePoint1.X) * (top - linePoint1.Y) / (linePoint2.Y - linePoint1.Y);
			if(x>=left && x<=right) return true;
		}
	}
	if(linePoint1.X != linePoint2.X) {
		if((linePoint1.X<left)^(linePoint2.X<left)) {
			int y = linePoint1.Y + (linePoint2.Y - linePoint1.Y) * (left - linePoint1.X) / (linePoint2.X - linePoint1.X);
			if(y>=top && y<=bottom) return true;
		}
		if((linePoint1.X<right)^(linePoint2.X<right)) {
			int y = linePoint1.Y + (linePoint2.Y - linePoint1.Y) * (right - linePoint1.X) / (linePoint2.X - linePoint1.X);
			if(y>=top && y<=bottom) return true;
		}
	}
	return false;
}

/******************************************************************************
* Performs a hit test on a single line segment.
******************************************************************************/
void Window3D::hitTestLineSegment(const Point3& v1, const Point3& v2)
{
    CHECK_POINTER(pickingRegion());

	// Project vertices into screen space.
	Vector4 clipPoint1 = objectToScreenMatrix() * Vector4(v1.X, v1.Y, v1.Z, 1.0);
	Vector4 clipPoint2 = objectToScreenMatrix() * Vector4(v2.X, v2.Y, v2.Z, 1.0);

	unsigned int clipMask1 = 0, clipMask2 = 0;

	// Clip the two endpoints.
	if(clipPoint1[X] > clipPoint1[W])		clipMask1 |= CLIP_RIGHT_BIT;
	else if(clipPoint1[X] < -clipPoint1[W])	clipMask1 |= CLIP_LEFT_BIT;
	if(clipPoint1[Y] > clipPoint1[W])		clipMask1 |= CLIP_TOP_BIT;
	else if(clipPoint1[Y] < -clipPoint1[W])	clipMask1 |= CLIP_BOTTOM_BIT;
	if(clipPoint1[Z] > clipPoint1[W])		clipMask1 |= CLIP_FAR_BIT;
	else if(clipPoint1[Z] < -clipPoint1[W])	clipMask1 |= CLIP_NEAR_BIT;

	if(clipPoint2[X] > clipPoint2[W])		clipMask2 |= CLIP_RIGHT_BIT;
	else if(clipPoint2[X] < -clipPoint2[W])	clipMask2 |= CLIP_LEFT_BIT;
	if(clipPoint2[Y] > clipPoint2[W])		clipMask2 |= CLIP_TOP_BIT;
	else if(clipPoint2[Y] < -clipPoint2[W])	clipMask2 |= CLIP_BOTTOM_BIT;
	if(clipPoint2[Z] > clipPoint2[W])		clipMask2 |= CLIP_FAR_BIT;
	else if(clipPoint2[Z] < -clipPoint2[W])	clipMask2 |= CLIP_NEAR_BIT;

	// Are both points clipped by common plane?
	if((clipMask1 & clipMask2) != 0) return;

	if(clipMask1 != 0) ClipLine(clipMask1, clipPoint1, clipPoint2);
	if(clipMask2 != 0) ClipLine(clipMask2, clipPoint2, clipPoint1);

	FloatType wInv;
	wInv = 1.0 / clipPoint1[W];
	Point2I winPoint1 = viewportToScreen(Point2(clipPoint1[X] * wInv, clipPoint1[Y] * wInv));
	wInv = 1.0 / clipPoint2[W];
	Point2I winPoint2 = viewportToScreen(Point2(clipPoint2[X] * wInv, clipPoint2[Y] * wInv));

	switch(pickingRegion()->type()) {
		case PickRegion::POINT: {
			const PointPickRegion* pointRegion = static_cast<const PointPickRegion*>(pickingRegion());
			if(SquaredDistToLine(pointRegion->center(), winPoint1, winPoint2) <= square(pointRegion->radius())) {
				logHit(clipPoint2[Z] * wInv);
			}
		}
		break;
		case PickRegion::RECTANGLE: {
			const RectanglePickRegion* rectRegion = static_cast<const RectanglePickRegion*>(pickingRegion());
			if(rectRegion->crossing()) {
				if(LineCrossesRect(rectRegion->rect(), winPoint1, winPoint2))
					logHit();
			}
			else {
				if(rectRegion->rect().contains(winPoint1.X, winPoint1.Y) && rectRegion->rect().contains(winPoint2.X, winPoint2.Y))
					logHit();
			}
		}
		break;
		default: OVITO_ASSERT_MSG(false, "Window3D::hitTestLineSegment()", "Hit testing is not implemented yet for this picking region type.");
	};
}

/******************************************************************************
* Performs a hit test on a set of line segments.
******************************************************************************/
void Window3D::hitTestLines(size_t numberOfVertices, const Point3* vertices, const RenderEdgeFlag* edgeFlags)
{
	const Point3* endPointer = vertices + numberOfVertices;
	for(; vertices != endPointer; vertices++) {
		if(edgeFlags) {
			if(*edgeFlags++ != RENDER_EDGE_VISIBLE) continue;
		}
		hitTestLineSegment(vertices[0], vertices[1]);
	}
}

/******************************************************************************
* Performs a hit test on a polyline.
******************************************************************************/
void Window3D::hitTestPolyLine(size_t numberOfVertices, bool close, const Point3* vertices, const RenderEdgeFlag* edgeFlags)
{
	size_t v1, v2, i;
	if(close) { v2 = numberOfVertices - 1; i = 0; }
	else { v2 = 0; i = 1; }
	for(; i<numberOfVertices; i++) {
		v1 = v2; v2 = i;
		if(edgeFlags && edgeFlags[v1] != RENDER_EDGE_VISIBLE) continue;
		hitTestLineSegment(vertices[v1], vertices[v2]);
	}
}

/******************************************************************************
* Performs a hit test on a triangle mesh in wireframe mode.
******************************************************************************/
void Window3D::hitTestMeshWireframe(const TriMesh& mesh)
{
	// Hit test each edge.
	QVector<TriMesh::RenderEdge>::const_iterator edge = mesh.renderEdges.begin();
	for(; edge != mesh.renderEdges.end(); ++edge) {
		hitTestLineSegment(
			mesh.vertex(edge->v[0]),
			mesh.vertex(edge->v[1]));
	}
}

/******************************************************************************
* Performs a hit test on a triangle mesh in shaded mode.
******************************************************************************/
void Window3D::hitTestMeshShaded(const TriMesh& mesh)
{
	QVector<TriMeshFace>::const_iterator face = mesh.faces().begin();
	QVector<TriMeshFace>::const_iterator faceend = mesh.faces().end();
	for(; face != faceend; ++face) {
		hitTestFace(mesh.vertex(face->vertex(0)),
			mesh.vertex(face->vertex(1)),
			mesh.vertex(face->vertex(2)),
			face->normal);
	}
}

/******************************************************************************
* Finds the barycentric coordinates of a point in 2D.
******************************************************************************/
inline bool ComputeBarycentric(const Point2I& x1, const Point2I& x2, const Point2I& x3, const Point2I& p0, Point3& s)
{
	int cross = abs((x2.Y - x1.Y) * (x3.X - x1.X) - (x2.X - x1.X) * (x3.Y - x1.Y));
	if(cross == 0) return false;
	int a1 = abs(p0.X*(x2.Y-x3.Y) +
             x2.X*(x3.Y-p0.Y) +
             x3.X*(p0.Y-x2.Y));
	int a2 = abs(x1.X*(p0.Y-x3.Y) +
             p0.X*(x3.Y-x1.Y) +
             x3.X*(x1.Y-p0.Y));
	int a3 = abs(x1.X*(x2.Y-p0.Y) +
             x2.X*(p0.Y-x1.Y) +
             p0.X*(x1.Y-x2.Y));
	if(a1+a2+a3 != cross) return false;
	s[0] = (FloatType)a1 / cross;
	s[1] = (FloatType)a2 / cross;
	s[2] = (FloatType)a3 / cross;
	return true;
}

/******************************************************************************
* Compares two numbers.
******************************************************************************/
inline int compare(FloatType n1, FloatType n2) {
	return (n1 < n2) ? -1 : (n2 < n1) ? 1 : 0;
}

/******************************************************************************
* Tests if a point is inside an 2D-triangle.
******************************************************************************/
inline bool InsideTriangle(const Point2I& p1, const Point2I& p2, const Point2I& p3, const Point2I& q)
{
	int c = compare((p2.X - p3.X) * (q.Y - p3.Y), (p2.Y - p3.Y) * (q.X - p3.X));
	c    |= compare((p1.X - p2.X) * (q.Y - p2.Y), (p1.Y - p2.Y) * (q.X - p2.X));
	c    |= compare((p3.X - p1.X) * (q.Y - p1.Y), (p3.Y - p1.Y) * (q.X - p1.X));
	return c != (1|(-1));
}

/******************************************************************************
* Tests if a point is inside an 2D-triangle.
******************************************************************************/
inline bool InsideTriangle(const Point2I tri[3], const Point2I& q)
{
	return InsideTriangle(tri[0], tri[1], tri[2], q);
}

/******************************************************************************
* Performs a hit test on a single triangle face.
******************************************************************************/
void Window3D::hitTestFace(const Point3& v1, const Point3& v2, const Point3& v3, const Vector3& normal)
{
	CHECK_POINTER(pickingRegion());

	// Do back culling.
	if(backfaceCulling()) {
		/// Get camera position in object space.
		if(isPerspectiveProjection()) {
			Point3 viewPos = ORIGIN + _objToViewMatrixInv.getTranslation();
			if(DotProduct(viewPos - v1, normal) < FLOATTYPE_EPSILON) return;
		}
		else {
			Vector3 viewDir = _objToViewMatrixInv * Vector3(0,0,1);
			if(DotProduct(viewDir, normal) < FLOATTYPE_EPSILON) return;
		}
	}

	// Project vertices into screen space.
	Vector4 clipPoints[3];
	clipPoints[0] = objectToScreenMatrix() * Vector4(v1.X, v1.Y, v1.Z, 1.0);
	clipPoints[1] = objectToScreenMatrix() * Vector4(v2.X, v2.Y, v2.Z, 1.0);
	clipPoints[2] = objectToScreenMatrix() * Vector4(v3.X, v3.Y, v3.Z, 1.0);

	// Compute clipping codes.
	unsigned int clipMasks[3];
	for(int v=0; v<3; v++) {
		clipMasks[v] = 0;
		if(clipPoints[v][X] > clipPoints[v][W])			clipMasks[v] |= CLIP_RIGHT_BIT;
		else if(clipPoints[v][X] < -clipPoints[v][W])	clipMasks[v] |= CLIP_LEFT_BIT;
		if(clipPoints[v][Y] > clipPoints[v][W])			clipMasks[v] |= CLIP_TOP_BIT;
		else if(clipPoints[v][Y] < -clipPoints[v][W])	clipMasks[v] |= CLIP_BOTTOM_BIT;
		if(clipPoints[v][Z] > clipPoints[v][W])			clipMasks[v] |= CLIP_FAR_BIT;
		else if(clipPoints[v][Z] < -clipPoints[v][W])	clipMasks[v] |= CLIP_NEAR_BIT;
	}
	// Are all points clipped by common plane?
	if(clipMasks[0] & clipMasks[1] & clipMasks[2]) return;

	if(clipMasks[0] | clipMasks[1] | clipMasks[2]) {
		hitTestClippedTriangle(clipPoints);
	}
	else {
		// Project points to screen space.
		Point2I winPoints[3];
		FloatType wInv;
		for(int v=0; v<3; v++) {
			wInv = 1.0 / clipPoints[v][W];
			winPoints[v] = viewportToScreen(Point2(clipPoints[v][X] * wInv, clipPoints[v][Y] * wInv));
		}

		switch(pickingRegion()->type()) {
			case PickRegion::POINT: {
				const Point2I& p0 = static_cast<const PointPickRegion*>(pickingRegion())->center();
				Point3 s;
				if(ComputeBarycentric(winPoints[0], winPoints[1], winPoints[2], p0, s)) {
					// Interpolate coordinates using barycentric position.
					Vector4 intersect = (clipPoints[0] * s[0]) + (clipPoints[1] * s[1]) + (clipPoints[2] * s[2]);
					logHit(intersect[Z] / intersect[W]);
				}
			}
			break;
			case PickRegion::RECTANGLE: {
				const RectanglePickRegion* rectRegion = static_cast<const RectanglePickRegion*>(pickingRegion());
				const QRect& rect = rectRegion->rect();
				if(!rectRegion->crossing()) {
					// Is face completely in rect?
					if(rect.contains(winPoints[0].X, winPoints[0].Y)
						&& rect.contains(winPoints[1].X, winPoints[1].Y)
						&& rect.contains(winPoints[2].X, winPoints[2].Y)) logHit();
				}
				else {
					// Is any vertex in rect?
					if(rect.contains(winPoints[0].X, winPoints[0].Y)
						|| rect.contains(winPoints[1].X, winPoints[1].Y)
						|| rect.contains(winPoints[2].X, winPoints[2].Y))
					{
						logHit();
					} else {
						// Face crosses rect?
						if(LineCrossesRect(rect, winPoints[0], winPoints[1])
							|| LineCrossesRect(rect, winPoints[1], winPoints[2])
							|| LineCrossesRect(rect, winPoints[2], winPoints[0]))
						{
							logHit();
						} else {
							// Rect completely in face?
							if(InsideTriangle(winPoints, Point2(rect.x(), rect.y()))) {
								logHit();
							}
						}
					}
				}
			}
			break;
			default: OVITO_ASSERT_MSG(false, "Window3D::hitTestFace()", "Hit testing is not implemented yet for this picking region type.");
		}
	}
}

/******************************************************************************
* Clips a triangle and performs a hit test.
******************************************************************************/
void Window3D::hitTestClippedTriangle(const Vector4 clipPoints[3])
{
	// Clip against view volume in clip coord space.

	// Allocate a new array for the polygon created from the clipped triangle.
	Vector4 points[16];
	points[0] = clipPoints[0];
	points[1] = clipPoints[1];
	points[2] = clipPoints[2];

	int vlistA[16];
	int vlistB[16];

	vlistA[0] = 0; vlistA[1] = 1; vlistA[2] = 2;
	int nA = 3;
	int nB;
	int nCount = 3;

	int previ, prevj;
	int curri, currj;
	FloatType dx, dy, dz, dw, t, neww;

	// Clip against +X
	if(nA<3) return;
	previ = nA - 1;		// let previous = last vertex
	prevj = vlistA[previ];
	nB = 0;
	for(curri=0; curri<nA; curri++) {
		currj = vlistA[curri];
		if(points[currj][X] <= points[currj][W]) {
			if(points[prevj][X] <= points[prevj][W]) {
				// both verts are inside ==> copy current to outlist
				vlistB[nB] = currj;
				nB++;
			}
			else {
				// current is inside and previous is outside ==> clip
				dx = points[prevj][X] - points[currj][X];
				dw = points[prevj][W] - points[currj][W];
				t = (points[currj][X]-points[currj][W]) / (dw-dx);
				neww = points[currj][W] + t * dw;
				points[nCount][X] = neww;
				points[nCount][Y] = points[currj][Y] + t * (points[prevj][Y] - points[currj][Y]);
				points[nCount][Z] = points[currj][Z] + t * (points[prevj][Z] - points[currj][Z]);
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if(t>0.0) {
					// output new point
					vlistB[nB] = nCount;
					nCount++;
					nB++;
				}
				// Output current
				vlistB[nB] = currj;
				nB++;
			}
		}
		else {
			if(points[prevj][X] <= points[prevj][W]) {
				// current is outside and previous is inside ==> clip
				dx = points[currj][X] - points[prevj][X];
				dw = points[currj][W] - points[prevj][W];
				t = (points[prevj][X]-points[prevj][W]) / (dw-dx);
				neww = points[prevj][W] + t * dw;
				points[nCount][X] = neww;
				points[nCount][Y] = points[prevj][Y] + t * (points[currj][Y] - points[prevj][Y]);
				points[nCount][Z] = points[prevj][Z] + t * (points[currj][Z] - points[prevj][Z]);
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if (t>0.0) {
					// output new point
					vlistB[nB] = nCount;
					nCount++;
					nB++;
				}
			}
			// else both verts are outside ==> do nothing
		}
		// let previous = current
		previ = curri;
		prevj = currj;
	}

	// Clip against -X
	if(nB<3) return;
	previ = nB - 1;		// let previous = last vertex
	prevj = vlistB[previ];
	nA = 0;
	for(curri=0; curri<nB; curri++) {
		currj = vlistB[curri];
		if(points[currj][X] >= -points[currj][W]) {
			if(points[prevj][X] >= -points[prevj][W]) {
				// both verts are inside ==> copy current to outlist
				vlistA[nA] = currj;
				nA++;
			}
			else {
				// current is inside and previous is outside ==> clip
				dx = points[prevj][X] - points[currj][X];
				dw = points[prevj][W] - points[currj][W];
				t = -(points[currj][X]+points[currj][W]) / (dw+dx);
				neww = points[currj][W] + t * dw;
				points[nCount][X] = -neww;
				points[nCount][Y] = points[currj][Y] + t * (points[prevj][Y] - points[currj][Y]);
				points[nCount][Z] = points[currj][Z] + t * (points[prevj][Z] - points[currj][Z]);
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if(t>0.0) {
					// output new point
					vlistA[nA] = nCount;
					nCount++;
					nA++;
				}
				// Output current
				vlistA[nA] = currj;
				nA++;
			}
		}
		else {
			if(points[prevj][X] >= -points[prevj][W]) {
				// current is outside and previous is inside ==> clip
				dx = points[currj][X] - points[prevj][X];
				dw = points[currj][W] - points[prevj][W];
				t = -(points[prevj][X]+points[prevj][W]) / (dw+dx);
				neww = points[prevj][W] + t * dw;
				points[nCount][X] = -neww;
				points[nCount][Y] = points[prevj][Y] + t * (points[currj][Y] - points[prevj][Y]);
				points[nCount][Z] = points[prevj][Z] + t * (points[currj][Z] - points[prevj][Z]);
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if (t>0.0) {
					// output new point
					vlistA[nA] = nCount;
					nCount++;
					nA++;
				}
			}
			// else both verts are outside ==> do nothing
		}
		// let previous = current
		previ = curri;
		prevj = currj;
	}

	// Clip against +Y
	if(nA<3) return;
	previ = nA - 1;		// let previous = last vertex
	prevj = vlistA[previ];
	nB = 0;
	for(curri=0; curri<nA; curri++) {
		currj = vlistA[curri];
		if(points[currj][Y] <= points[currj][W]) {
			if(points[prevj][Y] <= points[prevj][W]) {
				// both verts are inside ==> copy current to outlist
				vlistB[nB] = currj;
				nB++;
			}
			else {
				// current is inside and previous is outside ==> clip
				dy = points[prevj][Y] - points[currj][Y];
				dw = points[prevj][W] - points[currj][W];
				t = (points[currj][Y]-points[currj][W]) / (dw-dy);
				neww = points[currj][W] + t * dw;
				points[nCount][X] = points[currj][X] + t * (points[prevj][X] - points[currj][X]);
				points[nCount][Y] = neww;
				points[nCount][Z] = points[currj][Z] + t * (points[prevj][Z] - points[currj][Z]);
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if(t>0.0) {
					// output new point
					vlistB[nB] = nCount;
					nCount++;
					nB++;
				}
				// Output current
				vlistB[nB] = currj;
				nB++;
			}
		}
		else {
			if(points[prevj][Y] <= points[prevj][W]) {
				// current is outside and previous is inside ==> clip
				dy = points[currj][Y] - points[prevj][Y];
				dw = points[currj][W] - points[prevj][W];
				t = (points[prevj][Y]-points[prevj][W]) / (dw-dy);
				neww = points[prevj][W] + t * dw;
				points[nCount][X] = points[prevj][X] + t * (points[currj][X] - points[prevj][X]);
				points[nCount][Y] = neww;
				points[nCount][Z] = points[prevj][Z] + t * (points[currj][Z] - points[prevj][Z]);
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if (t>0.0) {
					// output new point
					vlistB[nB] = nCount;
					nCount++;
					nB++;
				}
			}
			// else both verts are outside ==> do nothing
		}
		// let previous = current
		previ = curri;
		prevj = currj;
	}

	// Clip against -Y
	if(nB<3) return;
	previ = nB - 1;		// let previous = last vertex
	prevj = vlistB[previ];
	nA = 0;
	for(curri=0; curri<nB; curri++) {
		currj = vlistB[curri];
		if(points[currj][Y] >= -points[currj][W]) {
			if(points[prevj][Y] >= -points[prevj][W]) {
				// both verts are inside ==> copy current to outlist
				vlistA[nA] = currj;
				nA++;
			}
			else {
				// current is inside and previous is outside ==> clip
				dy = points[prevj][Y] - points[currj][Y];
				dw = points[prevj][W] - points[currj][W];
				t = -(points[currj][Y]+points[currj][W]) / (dw+dy);
				neww = points[currj][W] + t * dw;
				points[nCount][X] = points[currj][X] + t * (points[prevj][X] - points[currj][X]);
				points[nCount][Y] = -neww;
				points[nCount][Z] = points[currj][Z] + t * (points[prevj][Z] - points[currj][Z]);
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if(t>0.0) {
					// output new point
					vlistA[nA] = nCount;
					nCount++;
					nA++;
				}
				// Output current
				vlistA[nA] = currj;
				nA++;
			}
		}
		else {
			if(points[prevj][Y] >= -points[prevj][W]) {
				// current is outside and previous is inside ==> clip
				dy = points[currj][Y] - points[prevj][Y];
				dw = points[currj][W] - points[prevj][W];
				t = -(points[prevj][Y]+points[prevj][W]) / (dw+dy);
				neww = points[prevj][W] + t * dw;
				points[nCount][X] = points[prevj][X] + t * (points[currj][X] - points[prevj][X]);
				points[nCount][Y] = -neww;
				points[nCount][Z] = points[prevj][Z] + t * (points[currj][Z] - points[prevj][Z]);
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if (t>0.0) {
					// output new point
					vlistA[nA] = nCount;
					nCount++;
					nA++;
				}
			}
			// else both verts are outside ==> do nothing
		}
		// let previous = current
		previ = curri;
		prevj = currj;
	}

	// Clip against +Z
	if(nA<3) return;
	previ = nA - 1;		// let previous = last vertex
	prevj = vlistA[previ];
	nB = 0;
	for(curri=0; curri<nA; curri++) {
		currj = vlistA[curri];
		if(points[currj][Z] <= points[currj][W]) {
			if(points[prevj][Z] <= points[prevj][W]) {
				// both verts are inside ==> copy current to outlist
				vlistB[nB] = currj;
				nB++;
			}
			else {
				// current is inside and previous is outside ==> clip
				dz = points[prevj][Z] - points[currj][Z];
				dw = points[prevj][W] - points[currj][W];
				t = (points[currj][Z]-points[currj][W]) / (dw-dz);
				neww = points[currj][W] + t * dw;
				points[nCount][X] = points[currj][X] + t * (points[prevj][X] - points[currj][X]);
				points[nCount][Y] = points[currj][Y] + t * (points[prevj][Y] - points[currj][Y]);
				points[nCount][Z] = neww;
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if(t>0.0) {
					// output new point
					vlistB[nB] = nCount;
					nCount++;
					nB++;
				}
				// Output current
				vlistB[nB] = currj;
				nB++;
			}
		}
		else {
			if(points[prevj][Z] <= points[prevj][W]) {
				// current is outside and previous is inside ==> clip
				dz = points[currj][Z] - points[prevj][Z];
				dw = points[currj][W] - points[prevj][W];
				t = (points[prevj][Z]-points[prevj][W]) / (dw-dz);
				neww = points[prevj][W] + t * dw;
				points[nCount][X] = points[prevj][X] + t * (points[currj][X] - points[prevj][X]);
				points[nCount][Y] = points[prevj][Y] + t * (points[currj][Y] - points[prevj][Y]);
				points[nCount][Z] = neww;
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if (t>0.0) {
					// output new point
					vlistB[nB] = nCount;
					nCount++;
					nB++;
				}
			}
			// else both verts are outside ==> do nothing
		}
		// let previous = current
		previ = curri;
		prevj = currj;
	}

	// Clip against -Z
	if(nB<3) return;
	previ = nB - 1;		// let previous = last vertex
	prevj = vlistB[previ];
	nA = 0;
	for(curri=0; curri<nB; curri++) {
		currj = vlistB[curri];
		if(points[currj][Z] >= -points[currj][W]) {
			if(points[prevj][Z] >= -points[prevj][W]) {
				// both verts are inside ==> copy current to outlist
				vlistA[nA] = currj;
				nA++;
			}
			else {
				// current is inside and previous is outside ==> clip
				dz = points[prevj][Z] - points[currj][Z];
				dw = points[prevj][W] - points[currj][W];
				t = -(points[currj][Z]+points[currj][W]) / (dw+dz);
				neww = points[currj][W] + t * dw;
				points[nCount][X] = points[currj][X] + t * (points[prevj][X] - points[currj][X]);
				points[nCount][Y] = points[currj][Y] + t * (points[prevj][Y] - points[currj][Y]);
				points[nCount][Z] = -neww;
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if(t>0.0) {
					// output new point
					vlistA[nA] = nCount;
					nCount++;
					nA++;
				}
				// Output current
				vlistA[nA] = currj;
				nA++;
			}
		}
		else {
			if(points[prevj][Z] >= -points[prevj][W]) {
				// current is outside and previous is inside ==> clip
				dz = points[currj][Z] - points[prevj][Z];
				dw = points[currj][W] - points[prevj][W];
				t = -(points[prevj][Z]+points[prevj][W]) / (dw+dz);
				neww = points[prevj][W] + t * dw;
				points[nCount][X] = points[prevj][X] + t * (points[currj][X] - points[prevj][X]);
				points[nCount][Y] = points[prevj][Y] + t * (points[currj][Y] - points[prevj][Y]);
				points[nCount][Z] = -neww;
				points[nCount][W] = neww;

				// if new point not coincident with previous point...
				if(t>0.0) {
					// output new point
					vlistA[nA] = nCount;
					nCount++;
					nA++;
				}
			}
			// else both verts are outside ==> do nothing
		}
		// let previous = current
		previ = curri;
		prevj = currj;
	}

   	if(nA<3) return;

	// Transform from clipping to view space.
	Point2I projPoints[16];
	for(int v=0; v<nA; v++) {
		FloatType wInv = 1.0 / points[vlistA[v]][W];
		projPoints[v] = viewportToScreen(Point2(points[vlistA[v]][X] * wInv, points[vlistA[v]][Y] * wInv));
	}

	// Process triangle fan.
	switch(pickingRegion()->type()) {
		case PickRegion::POINT: {
			const Point2I& p0 = static_cast<const PointPickRegion*>(pickingRegion())->center();
			Point3 s;
			for(int v=2; v<nA; v++) {
				if(ComputeBarycentric(projPoints[0], projPoints[v-1], projPoints[v], p0, s)) {
					// Interpolate coordinates using barycentric position.
					Vector4 intersect = (points[vlistA[0]] * s[0]) + (points[vlistA[v-1]] * s[1]) + (points[vlistA[v]] * s[2]);
					logHit(intersect[Z] / intersect[W]);
					break;
				}
			}
		}
		break;
		case PickRegion::RECTANGLE: {
			const RectanglePickRegion* rectRegion = static_cast<const RectanglePickRegion*>(pickingRegion());
			const QRect& rect = rectRegion->rect();
			if(rectRegion->crossing()) {
				if(rect.contains(projPoints[0].X, projPoints[0].Y) || rect.contains(projPoints[1].X, projPoints[1].Y)) { logHit(); return; }
				// Face crosses rect?
				if(LineCrossesRect(rect, projPoints[0], projPoints[1])) { logHit(); return; }
				const Point2I* pp = projPoints + 2;
				for(int v=2; v<nA; v++, ++pp) {
					if(rect.contains(pp->X, pp->Y)) { logHit(); return; }
					// face crosses rect?
					if(LineCrossesRect(rect, projPoints[v-1], projPoints[v])) { logHit(); return; }
					// rect completely in face?
					if(InsideTriangle(projPoints[0], projPoints[v-1], projPoints[v], Point2I(rect.x(), rect.y()))) {
						logHit(); return;
					}
				}
			}
			else {
				int insideCount = nA;
				const Point2I* pp = projPoints;
				for(int v=0; v<nA; v++, ++pp) {
					if(rect.contains(pp->X, pp->Y)) insideCount--;
				}
				if(insideCount == 0) {
					logHit(); return;
				}
			}
		}
		break;
		default: OVITO_ASSERT_MSG(false, "Window3D::hitTestClippedTriangle()", "Hit testing is not implemented yet for this picking region type.");
	}
}

};
