///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file MeshObject.h
 * \brief Contains the definition of the Core::MeshObject class.
 */

#ifndef __OVITO_MESH_OBJECT_H
#define __OVITO_MESH_OBJECT_H

#include <core/Core.h>
#include <mesh/tri/TriMesh.h>
#include "../SceneObject.h"

namespace Core {

/**
 * \brief A scene object that consists of a static triangle mesh.
 *
 * \author Alexander Stukowski
 *
 * \sa Mesh
 * \sa SimpleGeometryObject
 */
class CORE_DLLEXPORT MeshObject : public SceneObject
{
public:

	/// \brief Default constructor that creates an empty MeshObject.
	/// \param isLoading Indicates whether the object is being loaded from a file.
	///                  This parameter is only used by the object serialization system.
	MeshObject(bool isLoading = false) : SceneObject(isLoading), _meshValidity(TimeForever) {}

	/// \brief Asks the object for its validity interval at the given time.
	/// \param time The animation time at which the validity interval should be computed.
	/// \return The maximum time interval that contains \a time and during which the object is valid.
	///
	/// This implementation just returns the validity interval set for the mesh using setObjectValidity().
	/// \sa setObjectValidity()
	virtual TimeInterval objectValidity(TimeTicks time) {
		return _meshValidity.contains(time) ? _meshValidity : TimeNever;
	}

	/// \brief Sets the validity interval for the internal mesh of this object.
	/// \return interval The new time interval during which the internal mesh is considered valid.
	/// \sa objectValidity()
	void setObjectValidity(TimeInterval interval) { _meshValidity = interval; }

	/// \brief Makes the object render itself into a viewport.
	/// \param time The animation time at which to render the object
	/// \param contextNode The node context used to render the object.
	/// \param vp The viewport to render in.
	///
	/// This implementation renders the internal Mesh in the viewport.
	virtual void renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp);

	/// \brief Asks the object whether it should show in a rendered image.
	virtual bool isRenderable() { return true; }

	/// \brief Renders the object in preview rendering mode using OpenGL.
	/// \param time The animation time at which to render the object
	/// \param view Describes the camera projection.
	/// \param contextNode The node context used to render the object.
	/// \param imageWidth The width of the rendering buffer in pixels.
	/// \param imageHeight The height of the rendering buffer in pixels.
	/// \param glcontext The window that provides the OpenGL rendering context.
	/// \return \c false if the rendering has been aborted by the user; \c true otherwise.
	virtual bool renderPreview(TimeTicks time, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext);

	/// \brief Computes the bounding box of the object.
	/// \param time The animation time for which the bounding box should be computed.
	/// \param contextNode The scene node to which this scene object belongs to.
	/// \return The bounding box of the object in local object coordinates.
	///
	/// This implementation returns the bounding box of the internal mesh.
	virtual Box3 boundingBox(TimeTicks time, ObjectNode* contextNode) {
		return mesh().boundingBox();
	}

	/// \brief Allows direct read-write access to the internal mesh.
	/// \return A reference to the internal mesh. This can be used to modify the mesh.
	TriMesh& mesh() { return _mesh; }

	/// \brief Allows direct read access to the mesh.
	/// \return A const reference to the internal mesh. This can be used to work with the mesh.
	const TriMesh& mesh() const { return _mesh; }

	/// \brief Sets the mesh of the MeshObject.
	/// \param mesh The new mesh.
	/// \note When setting the mesh the validity interval should also be updated using setObjectValidity().
	/// \sa mesh()
	void setMesh(const TriMesh& mesh) { _mesh = mesh; }

protected:

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream);

	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);

	/// Creates a copy of this object.
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

protected:

	/// The validity interval of the mesh.
	TimeInterval _meshValidity;

	/// The internal mesh of this object.
	TriMesh _mesh;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(MeshObject)
};


};

#endif // __OVITO_MESH_OBJECT_H
