///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/scene/animation/controller/TransformationController.h>
#include <core/undo/UndoManager.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(Controller, RefTarget)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(FloatController, Controller)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(IntegerController, Controller)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(BooleanController, Controller)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(VectorController, Controller)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(PositionController, Controller)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(RotationController, Controller)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ScalingController, Controller)

///////////////////////////// SINGLETON CLASS METHODS ///////////////////////////////

/// The singleton instance of this class.
ControllerManager* ControllerManager::_singletonInstance = NULL;

/******************************************************************************
* Initializes the controller manager.
******************************************************************************/
ControllerManager::ControllerManager()
{
	defaultMap[PLUGINCLASSINFO(FloatController)] = PLUGINCLASSINFO(LinearFloatController);
	defaultMap[PLUGINCLASSINFO(IntegerController)] = PLUGINCLASSINFO(ConstIntegerController);
	defaultMap[PLUGINCLASSINFO(BooleanController)] = PLUGINCLASSINFO(ConstBooleanController);
	defaultMap[PLUGINCLASSINFO(VectorController)] = PLUGINCLASSINFO(LinearVectorController);
	defaultMap[PLUGINCLASSINFO(PositionController)] = PLUGINCLASSINFO(SplinePositionController);
	defaultMap[PLUGINCLASSINFO(RotationController)] = PLUGINCLASSINFO(LinearRotationController);
	defaultMap[PLUGINCLASSINFO(ScalingController)] = PLUGINCLASSINFO(LinearScalingController);
	defaultMap[PLUGINCLASSINFO(TransformationController)] = PLUGINCLASSINFO(PRSTransformationController);
}

/// Creates a new instance of the default implementation for the given base controller type.
Controller::SmartPtr ControllerManager::createDefaultController(PluginClassDescriptor* controllerBaseClass)
{
	map<PluginClassDescriptor*, PluginClassDescriptor*>::const_iterator iter = defaultMap.find(controllerBaseClass);
	if(iter == defaultMap.end()) {
		OVITO_ASSERT_MSG(false, "ControllerManager::createDefaultController", "Unknown controller base class specified.");
		return NULL;
	}
	OVITO_ASSERT_MSG(iter->second != NULL, "ControllerManager::createDefaultController", "No default controller class available.");
	return static_object_cast<Controller>(iter->second->createInstance());
}


};
