///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file ObjectLoadStream.h 
 * \brief Contains definition of the Core::ObjectLoadStream class. 
 */

#ifndef __OVITO_OBJECT_LOADSTREAM_H
#define __OVITO_OBJECT_LOADSTREAM_H

#include <core/Core.h>
#include <base/io/LoadStream.h>

namespace Core {

/**
 * \brief An input stream that is used to parse a graph of objects from a file.
 * 
 * \sa ObjectSaveStream
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT ObjectLoadStream : public LoadStream
{
	Q_OBJECT	
public:
	/// \brief Opens the stream for reading.
	/// \param source The data stream from which the binary data is read. This must be 
	///               stream that supports random access.
	/// \throw Exception when the given data stream \a source does only support sequential access. 	
	ObjectLoadStream(QDataStream& source);

	/// \brief The destructor closes the stream.
	/// \sa close()
	virtual ~ObjectLoadStream() { close(); }

	/// \brief Closes the stream.
	/// \note The underlying data stream is not closed by this method.
	virtual void close();

	/// Loads an object with runtime type information from the stream.
	/// The method returns a pointer to the object but this object will be
	/// in an unintialized state until it is loaded at a later time.
	template<class T>
	intrusive_ptr<T> loadObject() {
		intrusive_ptr<PluginClass> ptr = loadObject();
		OVITO_ASSERT(!ptr || ptr->pluginClassDescriptor()->isKindOf(PLUGINCLASSINFO(T)));
		if(ptr && !ptr->pluginClassDescriptor()->isKindOf(PLUGINCLASSINFO(T))) 
			throw Exception(tr("Class hierarchy mismatch in file. The object class '%1' is not derived from class '%2' as expected.").arg(ptr->pluginClassDescriptor()->name()).arg(PLUGINCLASSINFO(T)->name()));
		return static_object_cast<T>(ptr);
	}

private:

	/// Loads an object with runtime type information from the stream.
	/// The method returns a pointer to the object but this object will be
	/// in an unintialized state until it is loaded at a later time.
	intrusive_ptr<PluginClass> loadObject();
	
	struct PropertyFieldEntry {
		QByteArray identifier;
		PluginClassDescriptor* definingClass;
		int flags;
		bool isReferenceField;
		PluginClassDescriptor* targetClass;
		const PropertyFieldDescriptor* field;
	};

	struct ClassEntry {
		PluginClassDescriptor* descriptor;
		QVector<PropertyFieldEntry> propertyFields;
	};
	
	struct ObjectEntry {
		intrusive_ptr<PluginClass> object;
		ClassEntry* pluginClass;
		qint64 fileOffset;
	};

	/// The plugin classes used in the current scene file.
	QVector<ClassEntry> classes;
	
	/// List all the objects of the current scene file.
	QVector<ObjectEntry> objects;

	/// Indices of those objects that need to be loaded.
	QVector<quint32> objectsToLoad;
	
	/// This points to the current object while the objects are being loaded from the stream.
	ObjectEntry* currentObject;
	
	friend class RefMaker;
};

};

#endif // __OVITO_OBJECT_LOADSTREAM_H
