/***************************** LICENSE START ***********************************

 Copyright 2014 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "ServiceEditor.h"

#include "EditorFactory.h"
#include "IconObject.h"
#include "Request.h"
#include "IconClass.h"
#include "Editor.h"
#include "ReplyObserver.h"

#include <assert.h>

#define SERVICEEDITOR_MV_DEBUG__

ServiceEditor::ServiceEditor(const IconClass& name, const string& kind) :
    Editor(name, kind),
    replyObserver_(nullptr)
{
}

ServiceEditor::~ServiceEditor()
{
}

void ServiceEditor::raiseIt()
{
    if (replyObserver_)
        replyObserver_->raise();
}

void ServiceEditor::closeIt()
{
    editDone();

    //TODO: what else can we do?
}


// -----------------------------------------------------------------
// ServiceEditor::edit
// Called to start editing a macro. We call the macroedit service to
// bring up the Qt macro editor.

void ServiceEditor::edit(void)
{
    string path = current_->path().str();

    // replyObserver is reference counted so it is safe to create a new object
    replyObserver_ = new ServiceEditorObserver(*this, path);
    replyObserver_->start();
}


//=================================================================

ServiceEditorObserver::ServiceEditorObserver(ServiceEditor& editor, string& path) :
    editor_(editor),
    path_(path)
{
}

ServiceEditorObserver::~ServiceEditorObserver()
{
}


void ServiceEditorObserver::start()
{
    // Editor::edit will already have set up the path that we need
    string kind = editor_.iconClass().name();

    if (kind == "WMSCLIENT") {
        MvRequest r;
        r.read(path_.c_str());

        r("_NAME")    = path_.c_str();
        r("_CLASS")   = "WMSCLIENT";
        r("_ACTION")  = "edit";
        r("_SERVICE") = "wmsclient";
        r.print();

        callService("wmsclient", r);
    }
    else if (kind == "WCSCLIENT") {
        MvRequest r;
        r.read(path_.c_str());

        r("_NAME")    = path_.c_str();
        r("_CLASS")   = "WCSCLIENT";
        r("_ACTION")  = "edit";
        r("_SERVICE") = "wcsclient";
        r.print();

        callService("wcsclient", r);
    }
    else if (kind == "RTTOV_INPUT_DATA") {
        MvRequest r("RTTOV_INPUT_DATA");

        r("PATH")     = path_.c_str();
        r("_NAME")    = path_.c_str();
        r("_CLASS")   = "RTTOV_INPUT_DATA";
        r("_ACTION")  = "edit";
        r("_SERVICE") = "ScmDataEditor";
        r.print();

        callService("ScmDataEditor", r);
    }
    else if (kind == "SCM_INPUT_DATA") {
        MvRequest r("SCM_INPUT_DATA");

        r("PATH")     = path_.c_str();
        r("_NAME")    = path_.c_str();
        r("_CLASS")   = "SCM_INPUT_DATA";
        r("_ACTION")  = "edit";
        r("_SERVICE") = "ScmDataEditor";
        r.print();

        callService("ScmDataEditor", r);
    }


    else {
        MvRequest r("MACROEDIT");

        r("PATH")     = path_.c_str();                       // path to macro
        r("LANGUAGE") = editor_.iconClass().name().c_str();  // icon class, e.g. MACRO or MAGML
        r("_CLASS")   = editor_.iconClass().name().c_str();  // icon class, e.g. MACRO or MAGML
        r.print();
        callService("macroedit", r);
    }
}

// the qt editor application finished
void ServiceEditorObserver::reply(const Request& r, int err)
{
    editor_.notifyObserverApply();
    editor_.Editor::editDone();
}

void ServiceEditorObserver::progress(const Request& r)
{
#ifdef SERVICEEDITOR_MV_DEBUG__
    std::cout << "ServiceEditor progress: " << std::endl;
    r.print();
#endif

    // register pid of the qt editor application
    if (r.getVerb() != nullptr && strcmp(r.getVerb(), "REGISTER_QTAPP_PID") == 0) {
        if (const char* ch = r("PID"))
            remotePid_ = std::string(ch);
    }
}

void ServiceEditorObserver::message(const string& msg)
{
#ifdef SERVICEEDITOR_MV_DEBUG__
    std::cout << "ServiceEditor message: " << msg << std::endl;
#endif
}

void ServiceEditorObserver::raise()
{
    if (!remotePid_.empty()) {
#ifdef SERVICEEDITOR_MV_DEBUG__
        std::cout << "ServiceEditorObserver::raise PID=" << remotePid_ << std::endl;
#endif
        MvRequest r("RAISE_WINDOW");
        r("PID") = remotePid_.c_str();
        MvApplication::sendMessage(r);
    }
}


void ServiceEditorObserver::print(ostream& /*s*/) const
{
}

static EditorMaker<ServiceEditor> maker1("QtMacroEditor");
static EditorMaker<ServiceEditor> maker2("QtOgcClientEditor");
static EditorMaker<ServiceEditor> maker3("QtScmDataEditor");
