/*
 * page.c: Page size functions
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * See the AUTHORS file for a list of people who have hacked on 
 * this code. 
 * See the ChangeLog file for a list of changes.
 *
 *
 * Contents:
 *
 *   ppd_get_page_size()   - Get the page size record for the given size.
 *   ppd_get_page_width()  - Get the page width for the given size.
 *   ppd_get_page_length() - Get the page length for the given size.
 *
 * OLD Contents:
 *
 *   ppdPageSize()   - Get the page size record for the given size.
 *   ppdPageWidth()  - Get the page width for the given size.
 *   ppdPageLength() - Get the page length for the given size.
 */

/*
 * Include necessary headers...
 */

#include "ppd.h"
#include <ctype.h>
#include <string.h>

/*
 * 'ppd_get_page_size()' - Get the page size record for the given size.
 */

PpdSize *			/* O - Size record for page or NULL */
ppd_get_page_size(PpdFile * ppd,	/* I - PPD file record */
		  const char *name)
{				/* I - Size name */
  float w, l;			/* Width and length of page */
  char units[255];		/* Page size units... */
  GSList *list;			/* List iteration var */
  PpdSize *size;		/* Current PpdSize */
  gboolean found = FALSE;	/* Indication of search result */

  if (ppd == NULL)
    return (NULL);

  if (name != NULL) {
    if (strncmp(name, "Custom.", 7) == 0 && ppd->variable_sizes) {
      /* 
       * Find the custom page size...
       */

      list = ppd->sizes;
      while (list) {
	size = PPD_SIZE(list->data);
	if (strcmp(size->name->str, "Custom") == 0) {
	  found = TRUE;
	  break;
	}
	list = g_slist_next(list);
      }

      if (!found)
	return (NULL);
      found = FALSE;		// set it back in case we need to use it again

      /* 
       * Variable size; size name can be one of the following:
       *
       *    Custom.WIDTHxLENGTHin    - Size in inches
       *    Custom.WIDTHxLENGTHcm    - Size in centimeters
       *    Custom.WIDTHxLENGTHmm    - Size in millimeters
       *    Custom.WIDTHxLENGTH[pt]  - Size in points
       */

      units[0] = '\0';
      if (sscanf(name + 7, "%fx%f%254s", &w, &l, units) < 2)
	return (NULL);

      if (strcasecmp(units, "in") == 0) {
	size->width = w * 72.0f;
	size->length = l * 72.0f;
	size->left = ppd->custom_margins[0];
	size->bottom = ppd->custom_margins[1];
	size->right = w * 72.0f - ppd->custom_margins[2];
	size->top = l * 72.0f - ppd->custom_margins[3];
      } else if (strcasecmp(units, "cm") == 0) {
	size->width = w / 2.54f * 72.0f;
	size->length = l / 2.54f * 72.0f;
	size->left = ppd->custom_margins[0];
	size->bottom = ppd->custom_margins[1];
	size->right = w / 2.54f * 72.0f - ppd->custom_margins[2];
	size->top = l / 2.54f * 72.0f - ppd->custom_margins[3];
      } else if (strcasecmp(units, "mm") == 0) {
	size->width = w / 25.4f * 72.0f;
	size->length = l / 25.4f * 72.0f;
	size->left = ppd->custom_margins[0];
	size->bottom = ppd->custom_margins[1];
	size->right = w / 25.4f * 72.0f - ppd->custom_margins[2];
	size->top = l / 25.4f * 72.0f - ppd->custom_margins[3];
      } else {
	size->width = w;
	size->length = l;
	size->left = ppd->custom_margins[0];
	size->bottom = ppd->custom_margins[1];
	size->right = w - ppd->custom_margins[2];
	size->top = l - ppd->custom_margins[3];
      }

      return (size);
    } else {
      /* 
       * Lookup by name...
       */
      list = ppd->sizes;
      while (list) {
	size = PPD_SIZE(list->data);
	if (strcmp(name, size->name->str) == 0)
	  return (size);
	list = g_slist_next(list);
      }
    }
  } else {
    /* 
     * Find default...
     */

    list = ppd->sizes;
    while (list) {
      size = PPD_SIZE(list->data);
      if (size->marked)
	return (size);
      list = g_slist_next(list);
    }
  }

  return (NULL);
}


/*
 * 'ppd_get_page_width()' - Get the page width for the given size.
 */

float /* O - Width of page in points or 0.0 */ ppd_get_page_width(PpdFile * ppd,	/* I 
											   - 
											   PPD 
											   file 
											   record 
											 */
								  const char
								  *name)
{				/* I - Size name */
  PpdSize *size;		/* Page size */


  if ((size = ppd_get_page_size(ppd, name)) == NULL)
    return (0.0);
  else
    return (size->width);
}


/*
 * 'ppd_get_page_length()' - Get the page length for the given size.
 */

float /* O - Length of page in points or 0.0 */ ppd_get_page_length(PpdFile * ppd,	/* I 
											   - 
											   PPD 
											   file 
											 */
								    const char
								    *name)
{				/* I - Size name */
  PpdSize *size;		/* Page size */


  if ((size = ppd_get_page_size(ppd, name)) == NULL)
    return (0.0);
  else
    return (size->length);
}
