/*
  libnzb

  Copyright (C) 2004-2005 Mattias Nordstrom <matta at ftlight net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


  Authors:
    Mattias Nordstrom <matta at ftlight net>

  $Id: yenc.cpp,v 1.7 2005/06/10 18:39:18 mnordstr Exp $
    This file provides the yenc decoding functions.

*/


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "yenc.h"
#include "crc32.h"

#include <iostream>
#include <stdio.h>

using namespace std;

extern "C" ydata ydecode(string data)
{
	// This decoder has been written to comply with yEnc revision 1.3
	// http://www.yenc.org/yenc-draft.1.3.txt
	
	unsigned int i;
	int crc32;
	ydata retdata;
	string pcrc32, header;
	string::size_type ybegin, begin, end, pos = 0, pos2 = 0;
	
	// Initialize struct with defaults.
	retdata.status = YRES_ERROR;
	retdata.filename = "";
	retdata.part = 0;
	retdata.size = 0;
	retdata.total = 0;
	retdata.total_size = 0;
	retdata.crc32 = "";
	retdata.block_begin = 0;
	retdata.block_end = 0;
	
	// Check for yenc encoded data.
	if ((pos = data.find("=ybegin ", 0)) == string::npos) {
		return retdata;
	}
	
	ybegin = pos;
	pos2 = data.find("\r\n", pos);
	begin = pos2 + 2;
	
	header = data.substr(ybegin, 2048);

	// part keyword, also means that this is a multi-part file.
	if ((pos = header.find(" part=")) != string::npos && pos < (begin-ybegin)) {
		pos2 = data.find("\r\n", begin);
		begin = pos2 + 2;
		
		pos2 = data.find_first_of(" \r\n", ybegin+pos+1);		
		retdata.part = atoi((data.substr(ybegin + pos + 6, pos2 - (ybegin + pos + 6))).c_str());
	}

	// total keyword.
	if ((pos = header.find(" total=")) != string::npos && pos < (begin-ybegin)) {
		pos2 = header.find_first_of(" \r\n", pos+1);
		retdata.total = atoi((header.substr(pos + 7, pos2 - (pos + 7))).c_str());
	}

	// size keyword.
	if ((pos = data.find(" size=", ybegin)) != string::npos && pos < begin) {
		pos2 = data.find_first_of(" \r\n", pos+1);		
		retdata.total_size = atol((data.substr(pos + 6, pos2 - (pos + 6))).c_str());
	}
	
	// name keyword.
	if ((pos = data.find(" name=", ybegin)) != string::npos && pos < begin) {
		pos2 = data.find("\r\n", pos);		
		retdata.filename = data.substr(pos + 6, pos2 - (pos + 6));
	}
	
	// Multi-part messages have their own =ypart header line.
	if (retdata.part != 0) {
		if ((pos = data.find("\r\n", ybegin)) != string::npos && pos < begin) {
			if (data.substr(pos + 2, 7) != "=ypart ") {
				return retdata;
			}
			
			ybegin = pos + 2;
			
			// begin keyword.
			if ((pos = data.find(" begin=", ybegin)) != string::npos && pos < begin) {
				pos2 = data.find_first_of(" \r\n", pos+1);		
				retdata.block_begin = atol((data.substr(pos + 7, pos2 - (pos + 7))).c_str());
			}
			
			// end keyword.
			if ((pos = data.find(" end=", ybegin)) != string::npos && pos < begin) {
				pos2 = data.find_first_of(" \r\n", pos+1);		
				retdata.block_end = atol((data.substr(pos + 5, pos2 - (pos + 5))).c_str());
			}

		} else {
			return retdata;
		}
	}
	
	if ((pos = data.rfind("=yend ")) == string::npos) {
		return retdata;
	}
	
	end = pos;

	// size keyword.
	if ((pos = data.find(" size=", end)) != string::npos) {
		pos2 = data.find_first_of(" \r\n", pos+1);
		retdata.size = atol((data.substr(pos + 6, pos2 - (pos + 6))).c_str());
	}
	
	// part keyword.
	if ((pos = data.find(" part=", end)) != string::npos) {
		pos2 = data.find_first_of(" \r\n", pos+1);
		if (retdata.part != (unsigned int)atoi((data.substr(pos + 6, pos2 - (pos + 6))).c_str())) {
			retdata.status = YRES_ERRPART;
			return retdata;
		}
	}
	
	// pcrc32 keyword.
	if ((pos = data.find(" pcrc32=", end)) != string::npos) {
		pos2 = data.find_first_of(" \r\n", pos+1);
		pcrc32 = data.substr(pos + 8, pos2 - (pos + 8));
	}
	
	// crc32 keyword.
	if ((pos = data.find(" crc32=", end)) != string::npos) {
		pos2 = data.find_first_of(" \r\n", pos+1);
		retdata.crc32 = data.substr(pos + 7, pos2 - (pos + 7));
	}
	
	data = data.substr(begin, end - begin);
	
	/* pos = 0;
	while ((pos = data.find("\r\n", pos)) != string::npos) {
		data.erase(pos, 2);
	}*/
	
	crc_init(&crc32);
	
	unsigned char c;
	
	for (i=0; i<data.length(); i++){
		c = (unsigned char)data[i];
		if (c == '\r' && data[i+1] == '\n') {
			i = i + 2;
			if (i == data.length()) continue;
			c = (unsigned char)data[i];
		}
		if (c == '=') {
			i++;
			c = (unsigned char)data[i];
			if (c == 0) continue;
			
			if (c-64 < 0) {
				c = c+256;
			}
			c = c-64;
		}
			
		if (c-42 < 0) {
			c = c+256;
		}
		retdata.data += c-42;
		crc_add(&crc32, c-42);
	}
	
	if (pcrc32 != "" && !crc32_check(crc32, pcrc32, retdata.data)) {
		retdata.status = YRES_ERRPCRC32;
	} else {
		retdata.status = YRES_OK;
	}
	if (retdata.data.length() != retdata.size) {
		retdata.status = YRES_ERRSIZE;
	}
	if (retdata.part != 0 && (retdata.block_end - (retdata.block_begin-1)) != retdata.size) {
		retdata.status = YRES_ERRSIZE;
	}

	return retdata;
}

bool crc32_check(int crc32, string c_crc, string data)
{
	unsigned long crc;
	string calc_crc;
	
	if (c_crc.length() % 2 != 0) {
		c_crc.insert(0, "0");
	}
	
	crc = crc32 ^ 0xFFFFFFFFl;
	
	char *calc_c_crc;
	asprintf(&calc_c_crc, "%08x", (unsigned int)crc);
	calc_crc = calc_c_crc;
	free(calc_c_crc);
	
/*	if (calc_crc.length() == 7) {
		calc_crc = "0" + calc_crc;
	}
	if (calc_crc.length() == 6) {
		calc_crc = "00" + calc_crc;
	}*/
	
	if (c_crc == calc_crc) {
		return true;
	} else {
		cerr << "libnzb: yEnc CRC " << calc_crc << " != " << c_crc << "\n\n";
		return false;
	}
}
