/*--------------------------------------------------------------------------*/
/* ALBERTA:   an Adaptive multi Level finite element toolbox using          */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques                                                     */
/*                                                                          */
/* file:     ellipt2.c                                                      */
/*                                                                          */
/* description:  solver for an elliptic model problem                       */
/*                                                                          */
/*                        -\Delta u = f  in \Omega                          */
/*                                u = g  on \partial \Omega                 */
/*                                                                          */
/* This program is meant to show off some of the new features of            */
/* ALBERTA 2.0, including submeshes, parametric meshes and the GMV          */
/* visualization interface.                                                 */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  author:    Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.alberta-fem.de                                               */
/*                                                                          */
/*  (c) by D. Koester (2002-2005)                                           */
/*--------------------------------------------------------------------------*/

#include <alberta.h>

/*--------------------------------------------------------------------------*/
/* global variables: finite element space, discrete solution                */
/*                   load vector and system matrix                          */
/*--------------------------------------------------------------------------*/

static const FE_SPACE *fe_space;         /* initialized by main()           */
static DOF_REAL_VEC   *u_h = nil;        /* initialized by build()          */
static DOF_REAL_VEC   *error_h = nil;    /* initialized by estimate()       */
static DOF_REAL_VEC   *f_h = nil;        /* initialized by build()          */
static DOF_MATRIX     *matrix = nil;     /* initialized by build()          */

/*--------------------------------------------------------------------------*/
/* struct ellipt_leaf_data: structure for storing one REAL value on each    */
/*                          leaf element as LEAF_DATA                       */
/* rw_el_est():  return a pointer to the memory for storing the element     */
/*               estimate (stored as LEAF_DATA), called by ellipt_est()     */
/* get_el_est(): return the value of the element estimates (from LEAF_DATA),*/
/*               called by adapt_method_stat()                              */
/*--------------------------------------------------------------------------*/

struct ellipt_leaf_data
{
  REAL estimate;            /*  one real for the estimate                   */
};

static REAL *rw_el_est(EL *el)
{
  if (IS_LEAF_EL(el))
    return(&((struct ellipt_leaf_data *)LEAF_DATA(el))->estimate);
  else
    return(nil);
}

static REAL get_el_est(EL *el)
{
  if (IS_LEAF_EL(el))
    return(((struct ellipt_leaf_data *)LEAF_DATA(el))->estimate);
  else
    return(0.0);
}

/*--------------------------------------------------------------------------*/
/* For test purposes: exact solution and its gradient (optional)            */
/*--------------------------------------------------------------------------*/

static void phi_theta(const REAL_D old_x, REAL *phi, REAL *theta)
{
  FUNCNAME("phi_theta");
  REAL_D x;
  REAL   r;

  COPY_DOW(old_x, x);
  r= NORM_DOW(x);

  if(r < 1.0E-15)
    ERROR_EXIT("Radius is too small!\n");
  else
    AXEY_DOW(1.0/r, old_x, x);

  if(ABS(x[0]) < 1.0E-15) {
    if(x[1] > 0.0)
      *phi = M_PI / 2.0;
    else
      *phi = -M_PI / 2.0;
  }
  else {
    if(x[0] > 0.0) 
      *phi = atan(x[1]/x[0]);
    else
      *phi = atan(x[1]/x[0]) + M_PI;
  }

  *theta = acos(x[2]);

  return;
}

static REAL u(const REAL_D x)
{
  REAL phi, theta, result;

  phi_theta(x, &phi, &theta);

  result = sin(2.0 * theta);
  result *= result * result * cos(3.0 * phi); 

  return result;
}

static const REAL *grd_u(const REAL_D x, REAL_D input)
{
  static REAL_D buffer = {};
  REAL         *grd = input ? input : buffer;
  REAL          ph, th;

  phi_theta(x, &ph, &th);

  if(th < 1.0E-15)
    SET_DOW(0.0, grd);
  else {
    REAL s1p = sin(ph), s3p = sin(3.0*ph), 
         c1p = cos(ph), c3p = cos(3.0*ph),
         s1t = sin(th), s2t = sin(2.0*th), 
         c1t = cos(th), c2t = cos(2.0*th);
    
    grd[0] =  3.0*s1p*s3p*s2t*s2t*s2t/s1t + 6.0*c1p*c3p*c1t*s2t*s2t*c2t;
    grd[1] = -3.0*c1p*s3p*s2t*s2t*s2t/s1t + 6.0*s1p*c3p*c1t*s2t*s2t*c2t;
    grd[2] = -6.0*c3p*s1t*s2t*s2t*c2t;
  }

  return(grd);
}

/*--------------------------------------------------------------------------*/
/* problem data: right hand side, boundary values                           */
/*--------------------------------------------------------------------------*/

static REAL g(const REAL_D x)              /* boundary values, not optional */
{
  return u(x);
}

static REAL f(const REAL_D x)              /* -Delta u, not optional        */
{
  REAL result, ph, th;

  phi_theta(x, &ph, &th);

  if(th < 1.0E-15)
    result = 0.0;
  else {
    REAL c3p = cos(3.0*ph),
         s1t = sin(th), s2t = sin(2.0*th), 
         c1t = cos(th), c2t = cos(2.0*th);
    
    result = 9.0*c3p*s2t*s2t*s2t/(s1t*s1t) - 6.0*(c3p*s2t/s1t)
      * (c1t*s2t*c2t + 4.0*s1t*c2t*c2t - 2.0*s1t*s2t*s2t);
  }

  return result;
}

static void ball_proj_func(REAL_D vertex, const EL_INFO *eli, 
			   const REAL_B lambda)
{
  REAL norm = NORM_DOW(vertex);
  
  norm = 1.0 / MAX(1.0E-15, norm);
  SCAL_DOW(norm, vertex);

  return;
}

static NODE_PROJECTION *init_node_proj(MESH *mesh, MACRO_EL *mel, int c)
{
  static NODE_PROJECTION ball_proj = {ball_proj_func};

  if(!mesh || ((c > 0) && mel->face_bound[c-1] == 2))
    return &ball_proj;

  return nil;
}

static int binding_method(MESH *master, MACRO_EL *mel, int face, void *ud)
{
  if(mel->face_bound[face] == 2)
    return true;

  return false;
}


/*--------------------------------------------------------------------------*/
/* build(): assemblage of the linear system: matrix, load vector,           */
/*          boundary values, called by adapt_method_stat()                  */
/*          on the first call initialize u_h, f_h, matrix and information   */
/*          for assembling the system matrix                                */
/*                                                                          */
/* struct op_info: structure for passing information from init_element() to */
/*                 LALt()                                                   */
/* init_element(): initialization on the element; calculates the            */
/*                 coordinates and |det DF_S| used by LALt; passes these    */
/*                 values to LALt via user_data,                            */
/*                 called on each element by update_matrix()                */
/* LALt():         implementation of -Lambda id Lambda^t for -Delta u,      */
/*                 called by update_matrix() after init_element()           */
/*--------------------------------------------------------------------------*/

struct op_info
{
  REAL_D      Lambda[MAX_N_QUAD_POINTS][N_LAMBDA];
  REAL        det[MAX_N_QUAD_POINTS];
};

static int init_element(const EL_INFO *el_info, const QUAD *quad[3], void *ud)
{
  struct op_info *info = (struct op_info *)ud;
  PARAMETRIC     *parametric = el_info->mesh->parametric;
  int             result = 0;

  if(parametric) {
    result = parametric->init_element(el_info, parametric);
    parametric->grd_lambda(el_info, quad[2], 0, nil, info->Lambda,
			   info->det);
  } 
  else
    info->det[0] = el_grd_lambda_2d(el_info, info->Lambda[0]);

  return result;
}

const REAL (*LALt(const EL_INFO *el_info, const QUAD *quad, 
		  int iq, void *ud))[N_LAMBDA]
{
  struct op_info *info = (struct op_info *)ud;
  int            i, j, k;
  static REAL    LALt[N_LAMBDA][N_LAMBDA];

  for (i = 0; i <= 2; i++)
    for (j = i; j <= 2; j++)
    {
      for (LALt[i][j] = k = 0; k < DIM_OF_WORLD; k++)
	LALt[i][j] += info->Lambda[iq][i][k]*info->Lambda[iq][j][k];
      LALt[i][j] *= info->det[iq];
      LALt[j][i] = LALt[i][j];
    }

  return((const REAL (*)[N_LAMBDA]) LALt);
}

static void build(MESH *mesh, U_CHAR flag)
{
  FUNCNAME("build");
  static const EL_MATRIX_INFO *matrix_info = nil;
  static const QUAD           *quad = nil;

  dof_compress(mesh);
  MSG("%d DOFs for %s\n", fe_space->admin->size_used, fe_space->name);

  if (!u_h)                 /*  access matrix and vector for linear system */
  {
    matrix = get_dof_matrix("A", fe_space, fe_space);
    f_h    = get_dof_real_vec("f_h", fe_space);
    u_h    = get_dof_real_vec("u_h", fe_space);
    u_h->refine_interpol = fe_space->bas_fcts->real_refine_inter;
    u_h->coarse_restrict = fe_space->bas_fcts->real_coarse_inter;
    dof_set(0.0, u_h);      /*  initialize u_h  !                          */
  }

  if (!matrix_info)           /* information for matrix assembling         */
  {
    OPERATOR_INFO  o_info = {nil};

    if(mesh->parametric)
      quad = get_quadrature(2, 2*fe_space->bas_fcts->degree + 2);
    else
      quad = get_quadrature(2, 2*fe_space->bas_fcts->degree-2);

    o_info.quad[2]        = quad;
    o_info.row_fe_space   = o_info.col_fe_space = fe_space;
    o_info.init_element   = init_element;
    o_info.LALt           = LALt;
    o_info.LALt_pw_const  = true;        /* pw const. assemblage is faster */
    o_info.LALt_symmetric = true;        /* symmetric assemblage is faster */
    o_info.use_get_bound  = true;        /* Dirichlet boundary conditions! */
    o_info.user_data = MEM_ALLOC(1, struct op_info);         /* user data! */

    o_info.fill_flag = CALL_LEAF_EL|FILL_COORDS;

    matrix_info = fill_matrix_info(&o_info, nil);
  }

  clear_dof_matrix(matrix);                /* assembling of matrix         */
  update_matrix(matrix, matrix_info);

  dof_set(0.0, f_h);                       /* assembling of load vector    */
  L2scp_fct_bas(f, quad, f_h);

  dirichlet_bound(g, f_h, u_h, nil);           /*  boundary values         */
  return;
}


/*--------------------------------------------------------------------------*/
/* solve(): solve the linear system, called by adapt_method_stat()          */
/*--------------------------------------------------------------------------*/

static void solve(MESH *mesh)
{
  FUNCNAME("solve");
  static REAL       tol = 1.e-8;
  static int        miter = 1000, info = 2, icon = 1, restart = 0;
  static OEM_SOLVER solver = NoSolver;

  if (solver == NoSolver)
  {
    tol = 1.e-8;
    GET_PARAMETER(1, "solver", "%d", &solver);
    GET_PARAMETER(1, "solver tolerance", "%f", &tol);
    GET_PARAMETER(1, "solver precon", "%d", &icon);
    GET_PARAMETER(1, "solver max iteration", "%d", &miter);
    GET_PARAMETER(1, "solver info", "%d", &info);
    if (solver == GMRes)
      GET_PARAMETER(1, "solver restart", "%d", &restart);
  }
  oem_solve_s(matrix, f_h, u_h, solver, tol, icon, restart, miter, info);

  return;
}

/*--------------------------------------------------------------------------*/
/* Functions for error estimate:                                            */
/* estimate():   calculates error estimate via ellipt_est()                 */
/*               calculates exact error also (only for test purpose),       */
/*               called by adapt_method_stat()                              */
/* r():          calculates the lower order terms of the element residual   */
/*               on each element at the quadrature node iq of quad          */
/*               argument to ellipt_est() and called by ellipt_est()        */
/*--------------------------------------------------------------------------*/

static REAL r(const EL_INFO *el_info, const QUAD *quad, int iq, REAL uh_iq, 
              const REAL_D grd_uh_iq)
{
  REAL_D      x;
  coord_to_world(el_info, quad->lambda[iq], x);
  return(-f(x));
}

#define EOC(e,eo) log(eo/MAX(e,1.0e-15))/M_LN2

static REAL estimate(MESH *mesh, ADAPT_STAT *adapt)
{
  FUNCNAME("estimate");
  static int     degree, norm = -1;
  static REAL    C[3] = {1.0, 1.0, 0.0};
  static REAL    est, est_old = -1.0, err, err_old = -1.0;
  static FLAGS r_flag = 0;  /* = (INIT_UH | INIT_GRD_UH),  if needed by r() */
  REAL_DD        A = {{1.0, 0.0, 0.0},{0.0, 1.0, 0.0}, {0.0, 0.0, 1.0}};
  const QUAD     *quad;
  
  if (norm < 0) {
    norm = H1_NORM;
    GET_PARAMETER(1, "error norm", "%d", &norm);
    GET_PARAMETER(1, "estimator C0", "%f", &C[0]);
    GET_PARAMETER(1, "estimator C1", "%f", &C[1]);
    GET_PARAMETER(1, "estimator C2", "%f", &C[2]);

    error_h = get_dof_real_vec("u-u_h", u_h->fe_space);
  }
  degree = 2*u_h->fe_space->bas_fcts->degree;

  /* Error estimation is not yet implemented for parametric meshes.         */
  if(!mesh->parametric) {
    est = ellipt_est(u_h, adapt, rw_el_est, nil, degree, norm, C, 
		     (const REAL_D *) A, r, r_flag);
  
    MSG("estimate   = %.8le", est);
    if (est_old >= 0)
      print_msg(", EOC: %.2lf\n", EOC(est,est_old));
    else
      print_msg("\n");
    est_old = est;
  } 
  else
    adapt->err_sum = 1.0; /* To keep mesh refinement going. */

  quad = get_quadrature(2, degree);
  if (norm == L2_NORM)
    err = L2_err(u, u_h, quad, 0, nil, nil);
  else
    err = H1_err(grd_u, u_h, quad, 0, nil, nil);

  interpol(u, error_h);
  dof_axpy(-1.0, u_h, error_h);

  MSG("||u-uh||%s = %.8le", norm == L2_NORM ? "L2" : "H1", err);
  if (err_old >= 0)
    print_msg(", EOC: %.2lf\n", EOC(err,err_old));
  else
    print_msg("\n");
  err_old = err;

  if(!mesh->parametric)
    MSG("||u-uh||%s/estimate = %.2lf\n", norm == L2_NORM ? "L2" : "H1",
	err/MAX(est,1.e-15));

  {
    DOF_REAL_VEC *drv_list[2] = {u_h, error_h};

    write_mesh_gmv(mesh, "ellipt2.gmv", true, true, 2,
		   drv_list, 0, nil, nil, 0.0);
  }
  return(adapt->err_sum);
}

/*--------------------------------------------------------------------------*/
/* main program                                                             */
/*--------------------------------------------------------------------------*/

int main(int argc, char **argv)
{
  FUNCNAME("main");
  MACRO_DATA        *data;
  MESH              *volume_mesh, *surf_mesh;
  int                n_refine = 0, degree = 1;
  const BAS_FCTS    *lagrange;
  static ADAPT_STAT *adapt;
  char               filename[100];

/*--------------------------------------------------------------------------*/
/*  first of all, init parameters of the init file                          */
/*--------------------------------------------------------------------------*/

  init_parameters(0, "INIT/ellipt2.dat");
  GET_PARAMETER(1, "macro file name", "%s", filename);
  GET_PARAMETER(1, "polynomial degree", "%d", &degree);
  GET_PARAMETER(1, "global refinements", "%d", &n_refine);
  
/*--------------------------------------------------------------------------*/
/*  get a mesh, and read the macro triangulation from file                  */
/*--------------------------------------------------------------------------*/

  data = read_macro(filename);

  volume_mesh = GET_MESH(3, "ALBERTA volume mesh", data, init_node_proj);

  free_macro_data(data);

  surf_mesh = get_submesh(volume_mesh, "ALBERTA surface mesh",
			  binding_method, nil);

  use_lagrange_parametric(volume_mesh, 2, init_node_proj(nil, nil, 0), 2);

  free_mesh(volume_mesh);

  init_leaf_data(surf_mesh, sizeof(struct ellipt_leaf_data), nil, nil);

  lagrange = get_lagrange(2, degree);
  TEST_EXIT(lagrange, "no lagrange BAS_FCTS\n");

  fe_space = get_fe_space(surf_mesh, lagrange->name, nil, lagrange, false);

  global_refine(surf_mesh, n_refine);

/*--------------------------------------------------------------------------*/
/*  init adapt structure and start adaptive method                          */
/*--------------------------------------------------------------------------*/

  adapt = get_adapt_stat(2, "ellipt", "adapt", 2, nil);
  adapt->estimate = estimate;
  adapt->get_el_est = get_el_est;
  adapt->build_after_coarsen = build;
  adapt->solve = solve;

  adapt_method_stat(surf_mesh, adapt);

  return(0);
}
