/*
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 */


/**
    \file   lcmaps_verify_account_from_pem.h
    \brief  This interface verifies the  credential mapping from user
            credential in pem format.
    \author Martijn Steenbakkers for the EU DataGrid.

    \brief  This interface verifies the  credential mapping from user
            credential in pem format.
    As input it requires the user credential in PEM format and the lcmaps
    account from the type lcmaps_account_info_t.

    -# lcmaps_verify_account_from_pem: interface function

    In general the behaviour of LCMAPS can be influenced by setting the following
    environment variables:
        Env variable                        Description
        (default value)
        ---------------                     -----------
        LCMAPS_LOG_FILE                     The logfile to which lcmaps log
        ("/var/log/lcmaps.log")             messages will be written

        LCMAPS_DB_FILE                      the location (name) of the lcmaps policy file
        (LCMAPS_ETC_DIR"/lcmaps.db")

        LCMAPS_POLICY_NAME                  the list of the lcmaps policies (in the
        ("acquisition_policy")              policy file) that should be evaluated until
                                            a policy succeeds. The policies are separated
                                            by ':', e.g. "policy_preferred:policy_fallback:".

        LCMAPS_LOG_STRING                   A string that will be prepended to all
        ("poolindex :")                     LCMAPS logging messages. If this variable is not
                                            set a default will be taken which includes a 
                                            timestamp

        LCMAPS_DEBUG_LEVEL                  The debug level influences the logging
        ("0")                               verbosity. values 0-5.

        LCMAPS_MODULES_DIR                  The directory in which modules can be found.
        ("")                                Note: may be overwritten by what is in
                                            the policy file (the convential method)

        LCMAPS_ETC_DIR                      The directory of the lcmaps policy files
        ("/opt/glite/etc/lcmaps")           and other config files. In this case this location
                                            is hardcoded to be "/opt/glite/etc/lcmaps", because this
                                            lcmaps interface is used by VO-services and, therefore, not
                                            controlled by the site.

        GRIDMAPDIR                          The directory in which the poolaccount leases
        ("")                                are registered.
                                            Note: may be overwritten by what is in
                                            the policy file (the convential method)

    \ingroup LcmapsBasicInterface
*/

#ifndef LCMAPS_VERIFY_ACCOUNT_FROM_PEM_H
#define LCMAPS_VERIFY_ACCOUNT_FROM_PEM_H

/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_account.h"
#include "lcmaps_basic.h"

/******************************************************************************
                   Module definition
 *****************************************************************************/

/******************************************************************************
Function:   lcmaps_verify_account_from_pem
Description:
    Verify the account mapping from the supplied user credential.
 
Parameters:
    pem_string:    :  The PEM-encoded string containing the user proxy
    lcmaps_account :  A structure that contains account information: (input)
                      uid, gids (primary and secondary).
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure
******************************************************************************/
/*!
        lcmaps_verify_account_from_pem(
        char * pem_string,
        lcmaps_account_info_t lcmaps_account
    )

    \brief Verify the account mapping from the supplied user credential.

    \param pem_string
            The PEM-encoded string containing the user proxy
    \param lcmaps_account
            A structure that contains account information: (input)
            uid, gids (primary and secondary) and the poolindex
            Please use lcmaps_account_info_clean() to clean this
            structure after use.

    \retval 0 success.
    \retval 1 failure.
*/
typedef int
lcmaps_verify_account_from_pem_t(
        char * pem_string,
        lcmaps_account_info_t lcmaps_account
);

#include "_lcmaps_verify_account_from_pem.h"

#ifndef LCMAPS_USE_DLOPEN
lcmaps_verify_account_from_pem_t lcmaps_verify_account_from_pem;
#endif /* LCMAPS_USE_DLOPEN */
/**
   The opaque handle type that the API user should declare a single instance of.
   This must be passed to nearly all macros (in OO terms, this would be a derived
   type of LCMAPS_HANDLE).
   Usage: 
   LCMAPS_HANDLE_VERIFY_ACCOUNT_FROM_PEM handle;
 */
typedef _lcmaps_handle_verify_account_from_pem_t LCMAPS_HANDLE_VERIFY_RETURN_ACCOUNT_FROM_PEM;

/**
   The declarator can be used to declare and init the handle in one go.
 */
#define LCMAPS_DECL_HANDLE_VERIFY_ACCOUNT_FROM_PEM(l) \
    _LCMAPS_DECL_HANDLE_VERIFY_ACCOUNT_FROM_PEM(l)

/**
   The initializer of the handle object. This macro must be called before any
   other macro using an LCMAPS_HANDLE.
   \param l An instance of LCMAPS_HANDLE_VERIFY_ACCOUNT_FROM_PEM
   \retval 1 always.
 */
#define LCMAPS_INIT_HANDLE_VERIFY_ACCOUNT_FROM_PEM(l) \
    _LCMAPS_INIT_HANDLE_VERIFY_ACCOUNT_FROM_PEM(l)

/**
   This macro loads and prepares LCMAPS; must be called before any 
   use of the interface functions, but after LCMAPS_INIT_HANDLE_VERIFY_ACCOUNT_FROM_PEM.
   \param l an initialized handle
   \retval 0 in case of failure, inspect LCMAPS_ERRMSG(l)
   \retval 1 on success
 */
#define LCMAPS_LOAD_VERIFY_ACCOUNT_FROM_PEM(l) \
    _LCMAPS_LOAD_VERIFY_ACCOUNT_FROM_PEM(l)

/**
    \brief close the LCMAPS_HANDLE

    This macro closes any opened interface libraries and clears the handle.
    After this call the handle must be considered invalid. The LCMAPS_CLOSE_HANDLE
    can be found in lcmaps.h.
*/
#define LCMAPS_CLOSE_HANDLE_VERIFY_ACCOUNT_FROM_PEM(l)	\
    LCMAPS_CLOSE_HANDLE(l)




#endif /* LCMAPS_VERIFY_ACCOUNT_FROM_PEM_H */
