# bzr-dbus: dbus support for bzr/bzrlib.
# Copyright (C) 2007 Canonical Limited.
#   Author: Robert Collins.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
#

"""Tests for the dbus activity service."""

import os
import signal
import socket
import subprocess
import tempfile
import thread
import time
import weakref

import dbus
import dbus.bus
import dbus.mainloop.glib
try:
    from gi.repository import GLib
except ImportError:
    import glib as GLib


import bzrlib.plugins
from bzrlib.smart.protocol import _encode_tuple, _decode_tuple

# done here to just do it once, and not in the plugin module to avoid doing it
# by default.
dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)

from bzrlib.tests import TestCaseWithTransport, TestSkipped

import bzrlib.plugins.dbus
from bzrlib.plugins.dbus import activity


def create_daemon():
    """Create a dbus daemon."""
    config_file = tempfile.NamedTemporaryFile()
    config_file.write('''
<!DOCTYPE busconfig PUBLIC "-//freedesktop//DTD D-Bus Bus Configuration 1.0//EN"
 "http://www.freedesktop.org/standards/dbus/1.0/busconfig.dtd">
<busconfig>
  <!-- Our well-known bus type, don't change this -->
  <type>session</type>

  <listen>unix:tmpdir=/tmp</listen>

  <policy context="default">
    <!-- Allow everything to be sent -->
    <allow send_destination="*"/>
    <!-- Allow everything to be received -->
    <allow eavesdrop="true"/>
    <!-- Allow anyone to own anything -->
    <allow own="*"/>
  </policy>

</busconfig>
''')
    config_file.flush()
    proc = subprocess.Popen(
        ['dbus-daemon', '--config-file=%s' % config_file.name,
        '--print-address'], stdout=subprocess.PIPE)
    address = proc.stdout.readline()
    return address.strip(), proc


class TemporaryBus(dbus.bus.BusConnection):
    """A TemporaryBus created within this application."""

    def __new__(cls):
        address, proc = create_daemon()
        try:
            result = dbus.bus.BusConnection.__new__(cls, address)
        except:
            os.kill(proc.pid, signal.SIGINT)
            raise
        result._test_process = proc
        result._bus_type = address
        return result

    def nuke(self):
        """Ensure the daemon is shutdown."""
        os.kill(self._test_process.pid, signal.SIGINT)


class TestCaseWithDBus(TestCaseWithTransport):

    def setUp(self):
        TestCaseWithTransport.setUp(self)
        # setup a private dbus session so we dont spam
        # the users desktop!
        self.bus = TemporaryBus()
        self.addCleanup(self.bus.nuke)


class TestActivity(TestCaseWithDBus):

    def test_advertise_branch_no_service_running(self):
        # should not error: just call it
        activity.Activity(bus=self.bus).advertise_branch(self.make_branch('.'))

    def test_advertise_branch_service_running_no_commits(self):
        """No commits in a branch leads to it being show with a revid of ''."""
        # We could construct a test-broadcaster to test with, but for now the
        # sheer overhead of that scares me.
        # attach a Broadcaster to our test branch: creates a running service.
        broadcaster = activity.Broadcast(self.bus)
        # get the object so we can subscribe to callbacks from it.
        dbus_object = self.bus.get_object(activity.Broadcast.DBUS_NAME,
            activity.Broadcast.DBUS_PATH)
        # we want to recieve the callbacks to inspect them.
        branches = []
        def catch_branch(revision, urls):
            branches.append((revision, urls))
        dbus_object.connect_to_signal("Revision", catch_branch,
            dbus_interface=activity.Broadcast.DBUS_INTERFACE)

        # now call our convenience JustDoIt method.
        branch = self.make_branch('.')
        activity.Activity(bus=self.bus).advertise_branch(branch)
        # now, let the broadcast method interactions all happen

        # '' because the branch has no commits.
        self.assertEqual([('', [branch.base])], branches)

    def test_advertise_branch_service_running_with_commits(self):
        """With commits, the commit is utf8 encoded."""
        # We could construct a test-broadcaster to test with, but for now the
        # sheer overhead of that scares me.
        # attach a Broadcaster to our test branch: creates a running service.
        broadcaster = activity.Broadcast(self.bus)
        # get the object so we can subscribe to callbacks from it.
        dbus_object = self.bus.get_object(activity.Broadcast.DBUS_NAME,
            activity.Broadcast.DBUS_PATH)
        # we want to recieve the callbacks to inspect them.
        branches = []
        def catch_branch(revision, urls):
            branches.append((revision, urls))
        dbus_object.connect_to_signal("Revision", catch_branch,
            dbus_interface=activity.Broadcast.DBUS_INTERFACE)

        # now call our convenience JustDoIt method.
        tree = self.make_branch_and_memory_tree('.')
        tree.lock_write()
        tree.add('')
        tree.commit('commit', rev_id=u'\xc8'.encode('utf8'))
        tree.unlock()
        activity.Activity(bus=self.bus).advertise_branch(tree.branch)
        # now, let the broadcast method interactions all happen

        # in theory, the revid here would be utf8 encoded, but dbus seems to
        # consider 'string' == 'unicode', and 'Magic happens' as far as wire
        # level decoding -> unicode results. CRY.
        self.assertEqual([(u'\xc8', [tree.branch.base])], branches)

    def test_constructor(self):
        """The constructor should setup a good working environment."""
        # This test appears to be written to specifically test connecting to an
        # externally available session bus, so skip it if no such bus exists
        # (e.g. on a buildd)
        if 'DBUS_SESSION_BUS_ADDRESS' not in os.environ:
            raise TestSkipped('No session bus available')
        obj = activity.Activity()
        # dbus uses a singleton approach for SessionBus, and we dont know if
        # it has __eq__ or if it would be reliable, so we use object identity.
        self.assertTrue(obj.bus is dbus.SessionBus(),
            "%r is not %r" % (obj.bus, dbus.SessionBus()))

    def test_server(self):
        """Calling Activity.serve() provides a convenient means to serve."""
        # to test the server, we can't easily run it in process due to
        # apparent glib/dbus/dbus-python interactions: so we fire it off
        # externally. As all the server does is serve requests until its
        # killed or quits, this is ok, if not ideal.
        process = self.start_bzr_subprocess(['dbus-broadcast'],
            skip_if_plan_to_signal=True,
            env_changes={'DBUS_SESSION_BUS_ADDRESS':self.bus._bus_type,
                         'BZR_PLUGINS_AT':
                             'dbus@%s' % (bzrlib.plugins.dbus.__path__[0],)},
            allow_plugins=True)
        # subscribe to the server : will fail if it did not startup correctly,
        # so we spin for up to 5 seconds then abort
        start = time.time()
        started = False
        while not started and time.time() - start < 5:
            try:
                dbus_object = self.bus.get_object(activity.Broadcast.DBUS_NAME,
                    activity.Broadcast.DBUS_PATH)
            except dbus.DBusException, e:
                if (e.get_dbus_name() ==
                    'org.freedesktop.DBus.Error.ServiceUnknown'):
                    # service not available - relinquish cpu
                    time.sleep(0.001)
                else:
                    # some other error
                    raise
            else:
                started = True
        self.assertTrue(started)
        # catch revisions
        revisions = []
        def catch_revision(revision, urls):
            revisions.append((revision, urls))
            # quit the loop as soon as it idles.
        dbus_object.connect_to_signal("Revision", catch_revision,
            dbus_interface=activity.Broadcast.DBUS_INTERFACE)
        # finally, announce something
        activity.Activity(bus=self.bus).announce_revision('foo', 'bar')
        # now, we need to block until the server has exited.
        # and finally, we can checkout our results.
        self.assertEqual([('foo', ['bar'])], revisions)
        # FUGLY: there seems to be a race condition where the finish
        # call hung, and I've not the time to debug it right now.
        time.sleep(0.05)
        self.finish_bzr_subprocess(process, 3, send_signal=signal.SIGINT)


class TestBroadcast(TestCaseWithDBus):

    def test_announce_revision(self):
        """Calling announce_revision via dbus should work and return nothing."""
        # attach a Broadcaster.
        broadcaster = activity.Broadcast(self.bus)
        dbus_object = self.bus.get_object(activity.Broadcast.DBUS_NAME,
            activity.Broadcast.DBUS_PATH)
        dbus_iface = dbus.Interface(dbus_object,
            activity.Broadcast.DBUS_INTERFACE)
        # we want to recieve the callbacks to inspect them.
        signals1 = []
        def catch_signal1(revision, urls):
            signals1.append((revision, urls))
        # second method to avoid any possible dbus muppetry
        signals2 = []
        def catch_signal2(revision, urls):
            signals2.append((revision, urls))
        dbus_object.connect_to_signal("Revision", catch_signal1,
            dbus_interface=activity.Broadcast.DBUS_INTERFACE)
        dbus_object.connect_to_signal("Revision", catch_signal2,
            dbus_interface=activity.Broadcast.DBUS_INTERFACE)

        # now try to call the announce method, which should call the signal
        # handlers - all of them..
        # we dont use announce_revision here because we want to catch errors
        # should they occur.
        errors = []
        results = []
        def handle_reply():
            results.append(None)
            if len(results) + len(errors) == 2:
                mainloop.quit() # end the mainloop
        def handle_error(error):
            errors.append(error)
            if len(results) + len(errors) == 2:
                mainloop.quit() # end the mainloop
        dbus_iface.announce_revision('revision1', 'url1',
            reply_handler=handle_reply,
            error_handler=handle_error)
        # for each revision.
        dbus_iface.announce_revision('revision2', 'url2',
            reply_handler=handle_reply,
            error_handler=handle_error)
        mainloop = GLib.MainLoop()
        mainloop.run()
        if errors:
            raise errors[0]
        self.assertEqual([('revision1', ['url1']), ('revision2', ['url2'])],
            signals1)
        self.assertEqual([('revision1', ['url1']), ('revision2', ['url2'])],
            signals2)

    def test_announce_revision_maps(self):
        """Calling announce_revision via dbus should work and return nothing."""
        # attach a Broadcaster.
        broadcaster = activity.Broadcast(self.bus)
        # grab an activity object for ease of use: we've tested the low level
        # behaviour already.
        an_activity = activity.Activity(self.bus)
        an_activity.add_url_map('foo/', 'bar/')
        dbus_object = self.bus.get_object(activity.Broadcast.DBUS_NAME,
            activity.Broadcast.DBUS_PATH)
        dbus_iface = dbus.Interface(dbus_object,
            activity.Broadcast.DBUS_INTERFACE)
        # we want to recieve the callbacks to inspect them.
        signals1 = []
        def catch_signal1(revision, urls):
            signals1.append((revision, urls))
        dbus_object.connect_to_signal("Revision", catch_signal1,
            dbus_interface=activity.Broadcast.DBUS_INTERFACE)
        # now try to call the announce method, which should call the signal
        # handlers.
        an_activity.announce_revision('revision1', 'foo/baz')
        self.assertEqual([('revision1', ['bar/baz'])], signals1)

    def test_announce_revision_urls(self):
        """Calling announce_revision_urls via dbus should work and return nothing."""
        # attach a Broadcaster.
        broadcaster = activity.Broadcast(self.bus)
        dbus_object = self.bus.get_object(activity.Broadcast.DBUS_NAME,
            activity.Broadcast.DBUS_PATH)
        dbus_iface = dbus.Interface(dbus_object,
            activity.Broadcast.DBUS_INTERFACE)
        # we want to recieve the callbacks to inspect them.
        signals1 = []
        def catch_signal1(revision, urls):
            signals1.append((revision, urls))
        # second method to avoid any possible dbus muppetry
        signals2 = []
        def catch_signal2(revision, urls):
            signals2.append((revision, urls))
        dbus_object.connect_to_signal("Revision", catch_signal1,
            dbus_interface=activity.Broadcast.DBUS_INTERFACE)
        dbus_object.connect_to_signal("Revision", catch_signal2,
            dbus_interface=activity.Broadcast.DBUS_INTERFACE)

        # now try to call the announce method, which should call the signal
        # handlers - all of them..
        # we dont use announce_revision here because we want to catch errors
        # should they occur.
        errors = []
        results = []
        def handle_reply():
            results.append(None)
            if len(results) + len(errors) == 2:
                mainloop.quit() # end the mainloop
        def handle_error(error):
            errors.append(error)
            if len(results) + len(errors) == 2:
                mainloop.quit() # end the mainloop
        dbus_iface.announce_revision_urls('revision1', ['url1'],
            reply_handler=handle_reply,
            error_handler=handle_error)
        # for each revision.
        dbus_iface.announce_revision_urls('revision2', ['url2'],
            reply_handler=handle_reply,
            error_handler=handle_error)
        mainloop = GLib.MainLoop()
        mainloop.run()
        if errors:
            raise errors[0]
        self.assertEqual([('revision1', ['url1']), ('revision2', ['url2'])],
            signals1)
        self.assertEqual([('revision1', ['url1']), ('revision2', ['url2'])],
            signals2)

    def test_announce_revision_urls_doesnt_map(self):
        """Calling announce_revision via dbus should work and return nothing."""
        # attach a Broadcaster.
        broadcaster = activity.Broadcast(self.bus)
        # grab an activity object for ease of use: we've tested the low level
        # behaviour already.
        an_activity = activity.Activity(self.bus)
        an_activity.add_url_map('foo/', 'bar/')
        dbus_object = self.bus.get_object(activity.Broadcast.DBUS_NAME,
            activity.Broadcast.DBUS_PATH)
        dbus_iface = dbus.Interface(dbus_object,
            activity.Broadcast.DBUS_INTERFACE)
        # we want to recieve the callbacks to inspect them.
        signals1 = []
        def catch_signal1(revision, urls):
            signals1.append((revision, urls))
        dbus_object.connect_to_signal("Revision", catch_signal1,
            dbus_interface=activity.Broadcast.DBUS_INTERFACE)
        # now try to call the announce method, which should call the signal
        # handlers.
        an_activity.announce_revision_urls('revision1', ['foo/baz'])
        self.assertEqual([('revision1', ['foo/baz'])], signals1)

    def test_add_url_map(self):
        """Calling add_url_map via dbus should succeed and return nothing."""
        # attach a Broadcaster.
        broadcaster = activity.Broadcast(self.bus)
        dbus_object = self.bus.get_object(activity.Broadcast.DBUS_NAME,
            activity.Broadcast.DBUS_PATH)
        dbus_iface = dbus.Interface(dbus_object,
            activity.Broadcast.DBUS_INTERFACE)

        # call add_url_map.
        errors = []
        results = []
        def handle_reply():
            results.append(None)
            if len(results) + len(errors) == 2:
                mainloop.quit() # end the mainloop
        def handle_error(error):
            errors.append(error)
            if len(results) + len(errors) == 2:
                mainloop.quit() # end the mainloop
        dbus_iface.add_url_map('foo/', 'bar/',
            reply_handler=handle_reply,
            error_handler=handle_error)
        # for two locations at the same url
        dbus_iface.add_url_map('foo/', 'baz/',
            reply_handler=handle_reply,
            error_handler=handle_error)
        mainloop = GLib.MainLoop()
        mainloop.run()
        if errors:
            raise errors[0]
        self.assertEqual([None, None], results)
        self.assertEqual([], errors)
        self.assertEqual({'foo/':['bar/', 'baz/']}, broadcaster.url_mapper.maps)

    def test_remove_url_map(self):
        """Calling remove_url_map via dbus should succeed and return nothing."""
        # attach a Broadcaster.
        broadcaster = activity.Broadcast(self.bus)
        dbus_object = self.bus.get_object(activity.Broadcast.DBUS_NAME,
            activity.Broadcast.DBUS_PATH)
        dbus_iface = dbus.Interface(dbus_object,
            activity.Broadcast.DBUS_INTERFACE)

        # call add_url_map.
        errors = []
        results = []
        def handle_reply():
            results.append(None)
            if len(results) + len(errors) == 4:
                mainloop.quit() # end the mainloop
        def handle_error(error):
            errors.append(error)
            if len(results) + len(errors) == 4:
                mainloop.quit() # end the mainloop
        # add two locations at the same prefix
        dbus_iface.add_url_map('foo/', 'bar/',
            reply_handler=handle_reply,
            error_handler=handle_error)
        dbus_iface.add_url_map('foo/', 'baz/',
            reply_handler=handle_reply,
            error_handler=handle_error)
        # and remove them both
        dbus_iface.remove_url_map('foo/', 'bar/',
            reply_handler=handle_reply,
            error_handler=handle_error)
        dbus_iface.remove_url_map('foo/', 'baz/',
            reply_handler=handle_reply,
            error_handler=handle_error)
        mainloop = GLib.MainLoop()
        mainloop.run()
        if errors:
            raise errors[0]
        self.assertEqual([None, None, None, None], results)
        self.assertEqual([], errors)
        self.assertEqual({}, broadcaster.url_mapper.maps)


class TestLanGateway(TestCaseWithDBus):

    # catch urls from network
    # urls from network: add to time-limited-cache, put on dbus.
    # deserialise content

    def test_create(self):
        gateway = activity.LanGateway(self.bus)
        # if no mainloop was supplied, a new one is created.
        self.assertNotEqual(None, gateway.mainloop)

    def test_run_binds(self):
        self.error = None
        mainloop = GLib.MainLoop()
        # we want run to: open a listening socket on localhost:4155 udp, the
        # bzr protocol port number.
        # to test this we want to check the port is in use during the loop.
        def check_socket_bound():
            # try to bind to the socket which should be already bound.
            sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            try:
                try:
                    self.assertRaises(socket.error, sock.bind, ('127.0.0.1', gateway.port))
                except Exception, e:
                    self.error = e
            finally:
                GLib.timeout_add(0, mainloop.quit)
        GLib.timeout_add(0, check_socket_bound)
        gateway = activity.LanGateway(self.bus, mainloop)
        # disable talking to dbus by making it a noop
        gateway.activity.listen_for_revisions = lambda x:x
        gateway.run(_port=0)
        if self.error:
            raise self.error

    def test_network_packets_trigger_handle_network_packet(self):
        mainloop = GLib.MainLoop()
        # we want network packets to call handle_network_packet,
        # so we override that to shutdown the mainloop and log the call.
        def send_packet():
            sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
            sock.sendto('data_to_handle', ('127.0.0.1', gateway.port))
            sock.close()
        GLib.timeout_add(0, send_packet)
        GLib.timeout_add(1000, mainloop.quit)
        gateway = activity.LanGateway(self.bus, mainloop)
        calls = []
        def handle_data(data):
            calls.append(('handle', data))
            GLib.timeout_add(0, mainloop.quit)
        gateway.handle_network_data = handle_data
        # disable talking to dbus by making it a noop
        gateway.activity.listen_for_revisions = lambda x:x
        gateway.run(_port=0)
        self.assertEqual([('handle', 'data_to_handle')], calls)

    def test_broadcast_data_calls_sock(self):
        # the method broadcast_data on a LanGateway should invoke
        # socket.sendto(data, ('<broadcast>', 4155))
        mainloop = GLib.MainLoop()
        gateway = activity.LanGateway(self.bus, mainloop)
        calls = []
        class StubSocket(object):
            def sendto(self, data, address):
                calls.append(('sendto', data, address))
        gateway.sock = StubSocket()
        gateway.broadcast_data('some data')
        self.assertEqual([('sendto', 'some data', ('<broadcast>', 4155))],
            calls)

    def test_run_listens_for_revisions(self):
        mainloop = GLib.MainLoop()
        GLib.timeout_add(0, mainloop.quit)
        # avoid asking dbus if something is subscribed.
        class StubActivity(object):
            def listen_for_revisions(self, callback):
                self.calls.append(('listen', callback))
        an_activity = StubActivity()
        an_activity.calls = []
        gateway = activity.LanGateway(self.bus, mainloop, activity=an_activity)
        gateway.run(_port=0)
        self.assertEqual([('listen', gateway.catch_dbus_revision)], an_activity.calls)

    def test_catch_dbus_revision_ignore_file_only(self):
        """catch_dbus_revision should ignore file:/// urls."""
        mainloop = GLib.MainLoop()
        gateway = activity.LanGateway(self.bus, mainloop)
        # instrument the transmission apis used by the LanGateway.
        calls = []
        def broadcast_data(data):
            calls.append(('broadcast_data', data))
        gateway.broadcast_data = broadcast_data
        gateway.catch_dbus_revision('a revid', ['file:///'])
        self.assertEqual([], calls)

    def test_catch_dbus_revision_strip_file(self):
        """catch_dbus_revision should strip file:/// urls if others exist."""
        mainloop = GLib.MainLoop()
        gateway = activity.LanGateway(self.bus, mainloop)
        # instrument the transmission apis used by the LanGateway.
        calls = []
        def broadcast_data(data):
            calls.append(('broadcast_data', data))
        gateway.broadcast_data = broadcast_data
        gateway.catch_dbus_revision('revid', ['file:///foo/', 'http://bar'])
        self.assertEqual(
            [('broadcast_data',
              _encode_tuple(('announce_revision', 'revid', 'http://bar')))],
            calls)

    def test_catch_dbus_revision_encodes_smart_tuple(self):
        """catch_dbus_revision should encode as _encode_tuple does."""
        mainloop = GLib.MainLoop()
        gateway = activity.LanGateway(self.bus, mainloop)
        # instrument the transmission apis used by the LanGateway.
        calls = []
        def broadcast_data(data):
            calls.append(('broadcast_data', data))
        gateway.broadcast_data = broadcast_data
        gateway.catch_dbus_revision('a revid', ['http://bar', 'bzr://host/'])
        self.assertEqual(
            [('broadcast_data',
              _encode_tuple(('announce_revision', 'a revid', 'http://bar', 'bzr://host/')))],
            calls)

    def test_catch_dbus_revision_ignores_revision_url_all_from_network(self):
        """There is a time limited window within which revisions are remembered."""
        mainloop = GLib.MainLoop()
        gateway = activity.LanGateway(self.bus, mainloop)
        # instrument the transmission apis used by the LanGateway.
        calls = []
        def broadcast_data(data):
            calls.append(('broadcast_data', data))
        gateway.broadcast_data = broadcast_data
        now = time.time()
        gateway.note_network_revision(now, 'a revid', ['url1', 'url2'])
        gateway.catch_dbus_revision('a revid', ['url1', 'url2'])
        self.assertEqual([], calls)

    def test_catch_dbus_revision_preserves_non_network_urls(self):
        """When some urls for a revision were not from the network the rest are sent."""
        mainloop = GLib.MainLoop()
        gateway = activity.LanGateway(self.bus, mainloop)
        # instrument the transmission apis used by the LanGateway.
        calls = []
        def broadcast_data(data):
            calls.append(('broadcast_data', data))
        gateway.broadcast_data = broadcast_data
        now = time.time()
        gateway.note_network_revision(now, 'a revid', ['url1'])
        gateway.catch_dbus_revision('a revid', ['url2'])
        self.assertEqual(
            [('broadcast_data',
              _encode_tuple(('announce_revision', 'a revid', 'url2')))],
            calls)

    def test_catch_dbus_revision_notes_revision(self):
        """Revisions coming in from dbus are noted too, to prevent reemission."""
        mainloop = GLib.MainLoop()
        gateway = activity.LanGateway(self.bus, mainloop)
        # instrument the transmission apis used by the LanGateway.
        calls = []
        def broadcast_data(data):
            calls.append(('broadcast_data', data))
        def note_network_revision(now, revid, urls):
            calls.append(('note', now, revid, urls))
        gateway.broadcast_data = broadcast_data
        gateway.note_network_revision = note_network_revision
        start = time.time()
        gateway.catch_dbus_revision('a revid', ['url2'])
        finish = time.time()
        self.assertEqual(
            [('note', calls[0][1], 'a revid', ['url2']),
             ('broadcast_data',
              _encode_tuple(('announce_revision', 'a revid', 'url2'))),
             ],
            calls)
        self.assertTrue(start <= calls[0][1])
        self.assertTrue(calls[0][1] <= finish)

    def test_note_network_revision(self):
        mainloop = GLib.MainLoop()
        gateway = activity.LanGateway(self.bus, mainloop)
        now = time.time()
        # noting the we've seen 'rev' now should cache it at minute granularity.
        gateway.note_network_revision(now, 'rev', ['url1', 'url2'])
        self.assertEqual({int(now/60):{'rev':['url1', 'url2']}},
            gateway.seen_revisions)

    def test_note_network_revision_trims_cache(self):
        mainloop = GLib.MainLoop()
        gateway = activity.LanGateway(self.bus, mainloop)
        now = time.time()
        # noting a time when there are stale entries should remove them.
        gateway.note_network_revision(now - 60*6, 'rev', ['url1', 'url2'])
        gateway.note_network_revision(now - 60*4, 'rev1', ['url3', 'url4'])
        gateway.note_network_revision(now, 'rev2', ['url5', 'url6'])
        self.assertEqual({
            int((now-60*4)/60):{'rev1':['url3', 'url4']},
            int(now/60):{'rev2':['url5', 'url6']},
            },
            gateway.seen_revisions)

    def test_handle_network_data(self):
        """data from the network is deserialised, passed to note_revision and dbus."""

        mainloop = GLib.MainLoop()
        # instrument the transmission apis used by the LanGateway.
        calls = []
        start = time.time()
        def note_network_revision(now, revid, urls):
            calls.append(('note_rev', now, revid, urls))
        class StubActivity(object):
            def announce_revision_urls(self, revision, urls):
                calls.append(('announce', revision, urls))
        an_activity = StubActivity()
        gateway = activity.LanGateway(self.bus, mainloop, an_activity)
        gateway.note_network_revision = note_network_revision
        network_data = _encode_tuple(('announce_revision', 'a revid', 'url2'))
        gateway.handle_network_data(network_data)
        finish = time.time()
        self.assertEqual(
            [('note_rev', calls[0][1], 'a revid', ('url2',)),
             ('announce', 'a revid', ('url2',))],
             calls)
        self.assertTrue(start <= calls[0][1])
        self.assertTrue(calls[0][1] <= finish)

# TODO: garbage from the network should be handled cleanly.
