/*******************************************************************************
 *  PROJECT: Palette Parser
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <cstdlib>
#include <cstring>
#include <sstream>
#include "palette.h"
#include "error.h"
#include "core/log-stream.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H


namespace pp
{

    Palette::Palette(void)
    {
    }


    Palette::~Palette(void)
    {
    }


    Glib::ustring Palette::name(void) const
    {
        return m_name;
    }


    void Palette::set_name(Glib::ustring name)
    {
        m_name = name;
    }


    Palette::size_type Palette::size(void) const
    {
        return m_colors.size();
    }


    Palette::size_type Palette::columns(void) const
    {
        return m_numColumns;
    }


    void Palette::set_columns(size_type cols)
    {
        m_numColumns = cols;
    }


    const Palette::color_list_type& Palette::colors(void) const
    {
        return m_colors;
    }


    Palette::color_list_type& Palette::colors(void)
    {
        return m_colors;
    }


    std::ostream& operator<<(std::ostream& out, const Palette& p)
    {
        using std::endl;
	out << "GIMP Palette" << endl
	    << "Name: " << p.name() << endl
	    << "Columns: " << p.columns() << endl;

	for (Palette::const_iterator it = p.colors().begin(); it != p.colors().end(); ++it)
	  {
	    out << (*it) << endl;
	  }

        return out;
    }


    bool Palette::parse(std::istream& in)
    {
        if (in)
        {
            try {
                in >> *this;
                return true;
            }
            catch (const std::exception& error)
            {
                std::cerr << error.what () << std::endl;
            }
            // fall through to return false
        }
        return false;
    }

    std::istream& operator>>(std::istream& in, Palette& p)
    {
        p.reset();
        const int N = 256;
        char line[N];
        LOG("Parsing Palette");
        while (in.getline(line, N))
        {
            if (!g_utf8_validate (line, -1, NULL))
            {
                g_warning ("Palette files must be encoded in UTF-8");
                continue;
            }
            p.m_emptyFile = false;
            p.m_currentLine++;
            // check if the line starts with the phrase GIMP Palette
            if (!p.m_headerSeen)
            {
                LOG("Looking for header...");
                p.m_headerSeen = p.parse_header(line);
            }
            else
            {
                if (line[0] == 'N')
                {
                    p.m_name = p.parse_name(line);
                    LOG("Got name: " << p.m_name);
                }
                else if (line[0] == 'C')
                {
                    p.m_numColumns = p.parse_columns(line);
                    LOG("Got columns: " << p.m_numColumns);
                }
                else if (line[0] == '#' || strlen(line) == 0)
                {
                    // ignore it, it's a comment line or a blank line
                }
                else
                {
                    // it should be a color definition
                    try
                    {
                        Palette::value_type rgb = p.parse_color(line);
                        //LOG("got color " << rgb);
                        p.m_colors.push_back(rgb);
                    }
                    catch (ParseError& err)
                    {
                        // ignore it if it's invalid
                        LOG("bad color");
                    }
                }
            }
        }
        // done parsing -- reset the current line back to the beginning.
        p.m_currentLine = 0;
        LOG("done parsing. colors found: " << p.m_colors.size());
        if (!p.m_headerSeen && !p.m_emptyFile)
            throw(ParseError("Gimp Palette Header not found"));
        return in;
    }


    Palette::value_type Palette::parse_color(const char* line) throw(ParseError)
    {
        const unsigned int MAX_RGB = 255;
        std::istringstream def(line);
        gint r, g, b;
        if (def >> r && def >> g && def >> b)
        {
            // valid values are 0 to 255
            if (!(r <= (int) MAX_RGB && r >= 0 &&
                        g <= (int) MAX_RGB && g >= 0 &&
                        b <= (int) MAX_RGB && b >= 0))
            {
                throw ParseError(line);
            }
            // the rest of the line should be the comment / name of the color
            const int N = 256;
            char cmt[N];
            // need to use getline instead of << in case comment has whitespace
            // in it
            def.getline(cmt, N); 
            value_type rgb = gcs::Color::create(r, g, b, trim(cmt));
            //LOG("result: " << rgb << endl;
            return rgb;
        }
        else
        {
            throw ParseError(line);
        }
    }


    void Palette::reset(void)
    {
        LOG("Resetting Palette");
        // clear the color definitions
        m_colors.clear();
        m_headerSeen = false;
        m_emptyFile = true;
        m_name = Glib::ustring();
        m_numColumns = 0;
        m_currentLine = 0;
    }

} // namespace pp
