#ifndef __MARTIAN_WATCH_H
#define __MARTIAN_WATCH_H

#include <time.h>
#include "common.h" /* config */

typedef enum {
	Min = 0,
	Max = 1,
	Avg = 2
} watchparam_t;
#define WParams 3


struct _actionwatch {
	union {
		struct {
		unsigned long long save_start;
		unsigned long long all, min, max, last;
		unsigned long long thresh;
		} tsc;

		struct {
		struct timespec save_start;
		struct timespec all, min, max, last;
		struct timespec thresh;
		} conv;
	};

	unsigned started;
	unsigned num, threshnum;
	/* cached unit */
	char units[WParams][3];
};

typedef struct _actionwatch actionwatch_t;

static inline unsigned long long tsc_to_nsec (unsigned long long tsc) {
	return	(tsc * 1000) / config.freq;
}

static inline unsigned long long nsec_to_tsc (unsigned long long nsec) {
	return	(nsec * config.freq) / 1000;
}

static inline unsigned long long conv_to_nsec (const struct timespec *t) {
	return 1000000000 * t->tv_sec + t->tv_nsec;
}

static inline void nsec_to_conv (struct timespec *t, unsigned long long nsec) {
	t->tv_sec = nsec / 1000000000; 	
	t->tv_nsec = nsec % 1000000000;	
}
	

static inline void conv_set (const struct timespec *t0, struct timespec *var) {
	var->tv_sec = t0->tv_sec;
	var->tv_nsec = t0->tv_nsec;
}

static inline int conv_less (const struct timespec *t1, const struct timespec *t2) {
	return (t1->tv_sec < t2->tv_sec) || 
		(t1->tv_sec == t2->tv_sec && t1->tv_nsec < t2->tv_nsec);
}

static inline void conv_sub (const struct timespec *t1, const struct timespec *t2, struct timespec *res) {
	if (t2->tv_nsec < t1->tv_nsec) {
		res->tv_nsec = t2->tv_nsec + 1000000000 - t1->tv_nsec;
		res->tv_sec = t2->tv_sec - t1->tv_sec - 1;
	}
	else {
		res->tv_nsec = t2->tv_nsec - t1->tv_nsec;
		res->tv_sec = t2->tv_sec - t1->tv_sec;
	}
}

static inline void conv_add (const struct timespec *t1, const struct timespec *t2, struct timespec *res) {
	res->tv_sec = t2->tv_sec + t1->tv_sec;
	res->tv_nsec = t2->tv_nsec + t1->tv_nsec;

	if (res->tv_nsec >= 1000000000) {
		res->tv_nsec -= 1000000000;
		res->tv_sec--;
	}
}

#define rdtsc(cnt) __asm__ __volatile__ ("rdtsc" : "=A" (cnt))

static inline void watch_init (struct _actionwatch *watch)
{
	if (config.haveTSC) {
		watch->tsc.all = 
		watch->tsc.max = 0;

		watch->tsc.min		= 
		watch->tsc.thresh	= -1;
	}
	else {
		watch->conv.all.tv_sec  = 
		watch->conv.all.tv_nsec = 
		watch->conv.max.tv_nsec = 
		watch->conv.max.tv_sec  = 0;

		watch->conv.min.tv_sec = -1;

		watch->conv.thresh.tv_sec = -1;
	}

	watch->num 		= 0;
	watch->threshnum 	= 0;
	watch->started 		= 0;
}

static inline void watch_set_thresh (actionwatch_t *w, unsigned long long nsec) {
	if (config.haveTSC) {
		w->tsc.thresh = nsec_to_tsc (nsec);
	}
	else {
		nsec_to_conv (&w->conv.thresh, nsec);
	}
}

static inline void watch_start (struct _actionwatch *watch)
{
	if (watch->started) return;

	if (config.haveTSC) 
		rdtsc (watch->tsc.save_start);
	else
		clock_gettime (CLOCK_REALTIME, &watch->conv.save_start);

	watch->started = 1;
}

static inline void watch_stop (struct _actionwatch *watch)
{
	if (! watch->started) return;
	watch->started = 0;
	if (config.haveTSC) {
		unsigned long long end, thetime;

		rdtsc (end);
		thetime = end - watch->tsc.save_start;

		watch->tsc.last = thetime;
		if (watch->tsc.thresh <= thetime) {
			watch->threshnum++;
			return;
		}

		watch->tsc.all += thetime;
		
		if (thetime < watch->tsc.min)
			watch->tsc.min = thetime;

		if (thetime > watch->tsc.max)
			watch->tsc.max = thetime;

	}
	else {
		struct timespec end, *thetime = &watch->conv.last;

		clock_gettime (CLOCK_REALTIME, &end);
		conv_sub (&watch->conv.save_start, &end, thetime);

		if (! conv_less (thetime, &watch->conv.thresh)) {
			watch->threshnum++;
			return;
		}

		conv_add (thetime, &watch->conv.all, &watch->conv.all);

		if (conv_less (thetime, &watch->conv.min)) 
			conv_set (thetime, &watch->conv.min);

		if (conv_less (&watch->conv.max, thetime)) 
			conv_set (thetime, &watch->conv.max);
	}

	watch->num++;
}

#define watch_nsec(watch, member) 	\
	(config.haveTSC ?		\
		(unsigned )((watch.tsc.member * 1000.0) / config.freq) : \
		watch.conv.member.tv_nsec)

#define watch_unit(watch, member) 	&watch.units[member][0]
static inline double watch_format (struct _actionwatch *watch, watchparam_t member) {
	unsigned nsec; 	/* up to 4 sec */
	char *unit = &watch->units[member][0];

	if (config.haveTSC) {
		unsigned long long t;

		switch (member) {
		case Min:
			t = (watch->tsc.min != -1) ? watch->tsc.min : 0;
			break;

		case Max:
			t = watch->tsc.max;
			break;
		case Avg:
			t = (watch->num == 0) ? 0 
				: watch->tsc.all / (double ) watch->num;
			break;
		}

		t = tsc_to_nsec (t);

		if (t  >= 1000000000) {
			unit[0] = 's';
			unit[1] = '\0';
			return 1.e-9 * t;
		}

		nsec = t;
	}
	else {
		struct timespec *ts, tmp;
		switch (member) {
		case Min:
			ts = (watch->conv.min.tv_sec == -1) ? 
				&watch->conv.min : &watch->conv.max;
			break;

		case Max:
			ts = &watch->conv.max;
			break;

		case Avg:
			if (watch->num == 0) 
				break;

			nsec_to_conv (&tmp, 
				conv_to_nsec (&watch->conv.all) / watch->num);
			ts = & tmp;
			
			break;
		}
		if (ts->tv_sec != 0) {
			unit[0] = 's';
			unit[1] = '\0';
			return ts->tv_sec + ts->tv_nsec * (double ) 1.e-9;
		}
		nsec = ts->tv_nsec;
	}

	double val = nsec;

	if (nsec < 1000) 
		unit[0] = 'n';

	else if (nsec < 1000000) {
		unit[0] = 'u';
		val *= 1e-3;
	}
	else {
		unit[0] = 'm';
		val *= 1e-6;
	}

	unit[1] = 's';
	unit[2] = '\0';

	return val;
}

#endif /*__MARTIAN_WATCH_H */
