/*******************************************
* Copyright (C) 2023 Intel Corporation
* SPDX-License-Identifier: BSD-3-Clause
*******************************************/

/*
 * ALGORITHM DESCRIPTION:
 *     
 *      ( optimized for throughput, with small table lookup, works when HW FMA is available )
 *     
 *      Implementation reduces argument x to |R|<pi/32
 *      16-entry tables used to store high and low parts of tan(x0)
 *      Argument x = N*pi + x0 + (R);   x0 = k*pi/16, with k in {0, 1, ..., 15}
 *      (very large arguments reduction resolved in _vdreduction_core.i)
 *      Compute result as (tan(x0) + tan(R))/(1-tan(x0)*tan(R))
 *      High accuracy ( < 1 ulp ) version keeps extra precision for numerator, denominator, and during
 *      final NR-iteration computing quotient.
 *     
 *     
 */


	.text
.L_2__routine_start___svml_tan8_z0_0:

	.align    16,0x90
	.globl __svml_tan8

__svml_tan8:


	.cfi_startproc
..L2:

        pushq     %rbp
	.cfi_def_cfa_offset 16
        movq      %rsp, %rbp
	.cfi_def_cfa 6, 16
	.cfi_offset 6, -16
        andq      $-64, %rsp
        subq      $192, %rsp
        xorl      %edx, %edx

/* Large values check */
        vmovups   28160+__svml_dtan_data_internal(%rip), %zmm1

/*
 * ----------------------------------------------------------
 * Main path
 * ----------------------------------------------------------
 * start arg. reduction
 */
        vmovups   1152+__svml_dtan_data_internal(%rip), %zmm2
        vmovups   64+__svml_dtan_data_internal(%rip), %zmm5
        vmovups   128+__svml_dtan_data_internal(%rip), %zmm3
        vmovups   192+__svml_dtan_data_internal(%rip), %zmm4
        vmovaps   %zmm0, %zmm11
        vandpd    1024+__svml_dtan_data_internal(%rip), %zmm11, %zmm0
        vcmppd    $22, {sae}, %zmm1, %zmm0, %k1
        vmovups   __svml_dtan_data_internal(%rip), %zmm1

/*
 * ----------------------------------------------------------
 * End of main path
 * ----------------------------------------------------------
 */
        kortestw  %k1, %k1
        vfmadd213pd {rn-sae}, %zmm2, %zmm11, %zmm1
        vsubpd    {rn-sae}, %zmm2, %zmm1, %zmm8
        vfnmadd213pd {rn-sae}, %zmm11, %zmm8, %zmm5
        vfnmadd231pd {rn-sae}, %zmm8, %zmm3, %zmm5
        vfnmadd213pd {rn-sae}, %zmm5, %zmm4, %zmm8
        jne       .LBL_1_12

.LBL_1_2:

        vmovups   384+__svml_dtan_data_internal(%rip), %zmm0
        vmovups   832+__svml_dtan_data_internal(%rip), %zmm5
        vmovups   768+__svml_dtan_data_internal(%rip), %zmm2
        vmovups   704+__svml_dtan_data_internal(%rip), %zmm3
        vmovups   640+__svml_dtan_data_internal(%rip), %zmm4
        vmulpd    {rn-sae}, %zmm8, %zmm8, %zmm6
        vpermt2pd 448+__svml_dtan_data_internal(%rip), %zmm1, %zmm0
        vmovups   896+__svml_dtan_data_internal(%rip), %zmm1
        vfmadd231pd {rn-sae}, %zmm6, %zmm1, %zmm5
        vfmadd213pd {rn-sae}, %zmm2, %zmm6, %zmm5
        vfmadd213pd {rn-sae}, %zmm3, %zmm6, %zmm5
        vfmadd213pd {rn-sae}, %zmm4, %zmm6, %zmm5
        vmulpd    {rn-sae}, %zmm8, %zmm5, %zmm7
        vfmadd213pd {rn-sae}, %zmm8, %zmm6, %zmm7

/*
 * Computer Denominator:
 * dDenominator - dDlow ~= 1-(dTh+dTl)*(dP+dPlow)
 */
        vmovups   960+__svml_dtan_data_internal(%rip), %zmm8

/*
 * Compute Numerator:
 * dNumerator + dNlow ~= dTh+dTl+dP+dPlow
 */
        vaddpd    {rn-sae}, %zmm0, %zmm7, %zmm9
        vfnmadd213pd {rn-sae}, %zmm8, %zmm7, %zmm0

/*
 * Now computes (dNumerator + dNlow)/(dDenominator - dDlow)
 * Choose NR iteration instead of hardware division
 */
        vrcp14pd  %zmm0, %zmm10

/* One NR iteration to refine dRcp */
        vfnmadd231pd {rn-sae}, %zmm10, %zmm0, %zmm8
        vfmadd213pd {rn-sae}, %zmm10, %zmm8, %zmm10
        vmulpd    {rn-sae}, %zmm9, %zmm10, %zmm12

/* One NR iteration to refine dQuotient */
        vfmsub213pd {rn-sae}, %zmm9, %zmm12, %zmm0
        vfnmadd213pd {rn-sae}, %zmm12, %zmm10, %zmm0
        testl     %edx, %edx
        jne       .LBL_1_4

.LBL_1_3:


/* no invcbrt in libm, so taking it out here */
        movq      %rbp, %rsp
        popq      %rbp
	.cfi_def_cfa 7, 8
	.cfi_restore 6
        ret
	.cfi_def_cfa 6, 16
	.cfi_offset 6, -16

.LBL_1_4:

        vmovups   %zmm11, 64(%rsp)
        vmovups   %zmm0, 128(%rsp)
        je        .LBL_1_3


        xorl      %eax, %eax


        vzeroupper
        kmovw     %k4, 24(%rsp)
        kmovw     %k5, 16(%rsp)
        kmovw     %k6, 8(%rsp)
        kmovw     %k7, (%rsp)
        movq      %rsi, 40(%rsp)
        movq      %rdi, 32(%rsp)
        movq      %r12, 56(%rsp)
	.cfi_escape 0x10, 0x04, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x68, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0x05, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x60, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0x0c, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x78, 0xff, 0xff, 0xff, 0x22
        movl      %eax, %r12d
        movq      %r13, 48(%rsp)
	.cfi_escape 0x10, 0x0d, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x70, 0xff, 0xff, 0xff, 0x22
        movl      %edx, %r13d
	.cfi_escape 0x10, 0xfa, 0x00, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x58, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0xfb, 0x00, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x50, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0xfc, 0x00, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x48, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0xfd, 0x00, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x40, 0xff, 0xff, 0xff, 0x22

.LBL_1_8:

        btl       %r12d, %r13d
        jc        .LBL_1_11

.LBL_1_9:

        incl      %r12d
        cmpl      $8, %r12d
        jl        .LBL_1_8


        kmovw     24(%rsp), %k4
	.cfi_restore 122
        kmovw     16(%rsp), %k5
	.cfi_restore 123
        kmovw     8(%rsp), %k6
	.cfi_restore 124
        kmovw     (%rsp), %k7
	.cfi_restore 125
        vmovups   128(%rsp), %zmm0
        movq      40(%rsp), %rsi
	.cfi_restore 4
        movq      32(%rsp), %rdi
	.cfi_restore 5
        movq      56(%rsp), %r12
	.cfi_restore 12
        movq      48(%rsp), %r13
	.cfi_restore 13
        jmp       .LBL_1_3
	.cfi_escape 0x10, 0x04, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x68, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0x05, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x60, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0x0c, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x78, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0x0d, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x70, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0xfa, 0x00, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x58, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0xfb, 0x00, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x50, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0xfc, 0x00, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x48, 0xff, 0xff, 0xff, 0x22
	.cfi_escape 0x10, 0xfd, 0x00, 0x0e, 0x38, 0x1c, 0x0d, 0xc0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0x40, 0xff, 0xff, 0xff, 0x22

.LBL_1_11:

        lea       64(%rsp,%r12,8), %rdi
        lea       128(%rsp,%r12,8), %rsi

        call      __svml_dtan_cout_rare_internal
        jmp       .LBL_1_9
	.cfi_restore 4
	.cfi_restore 5
	.cfi_restore 12
	.cfi_restore 13
	.cfi_restore 122
	.cfi_restore 123
	.cfi_restore 124
	.cfi_restore 125

.LBL_1_12:

        vmovups   1088+__svml_dtan_data_internal(%rip), %zmm14

/*
 * Get the (2^a / 2pi) mod 1 values from the table.
 * Because VLANG doesn't have L-type gather, we need a trivial cast
 */
        lea       __svml_dtan_reduction_data_internal(%rip), %rax
        vmovups   %zmm1, (%rsp)
        vpbroadcastq .L_2il0floatpacket.18(%rip), %zmm9
        vandpd    %zmm0, %zmm14, %zmm6
        vmovups   %zmm8, 64(%rsp)
        vcmppd    $4, {sae}, %zmm14, %zmm6, %k2
        vmovaps   %zmm9, %zmm5
        vpandq    .L_2il0floatpacket.19(%rip){1to8}, %zmm11, %zmm8
        vpsrlq    $52, %zmm8, %zmm10
        vpsllq    $1, %zmm10, %zmm7
        vpaddq    %zmm10, %zmm7, %zmm4
        vpsllq    $3, %zmm4, %zmm0
        vpmovqd   %zmm0, %ymm3
        vpandnq   %zmm6, %zmm6, %zmm5{%k2}
        vcmppd    $3, {sae}, %zmm5, %zmm5, %k0

/*
 * Break the P_xxx and m into 32-bit chunks ready for
 * the long multiplication via 32x32->64 multiplications
 */
        vpbroadcastq .L_2il0floatpacket.22(%rip), %zmm6
        kxnorw    %k0, %k0, %k3
        kxnorw    %k0, %k0, %k2
        kmovw     %k0, %edx
        vpxord    %zmm2, %zmm2, %zmm2
        vgatherdpd (%rax,%ymm3), %zmm2{%k3}
        kxnorw    %k0, %k0, %k3
        vpsrlq    $32, %zmm2, %zmm5
        vpxord    %zmm1, %zmm1, %zmm1
        vpxord    %zmm8, %zmm8, %zmm8
        vgatherdpd 8(%rax,%ymm3), %zmm1{%k2}
        vgatherdpd 16(%rax,%ymm3), %zmm8{%k3}
        vpsrlq    $32, %zmm1, %zmm0
        vpsrlq    $32, %zmm8, %zmm13

/*
 * Also get the significand as an integer
 * NB: adding in the integer bit is wrong for denorms!
 * To make this work for denorms we should do something slightly different
 */
        vpandq    .L_2il0floatpacket.20(%rip){1to8}, %zmm11, %zmm15
        vpaddq    .L_2il0floatpacket.21(%rip){1to8}, %zmm15, %zmm14
        vpsrlq    $32, %zmm14, %zmm7
        vpmullq   %zmm0, %zmm7, %zmm3
        vpandq    %zmm6, %zmm2, %zmm10
        vpandq    %zmm6, %zmm1, %zmm12
        vpandq    %zmm6, %zmm8, %zmm15
        vpandq    %zmm6, %zmm14, %zmm14

/* Now do the big multiplication and carry propagation */
        vpmullq   %zmm10, %zmm7, %zmm4
        vpmullq   %zmm12, %zmm7, %zmm2
        vpmullq   %zmm13, %zmm7, %zmm1
        vpmullq   %zmm15, %zmm7, %zmm8
        vpmullq   %zmm5, %zmm14, %zmm7
        vpmullq   %zmm10, %zmm14, %zmm5
        vpmullq   %zmm0, %zmm14, %zmm10
        vpmullq   %zmm12, %zmm14, %zmm0
        vpmullq   %zmm13, %zmm14, %zmm12
        vpsrlq    $32, %zmm10, %zmm15
        vpsrlq    $32, %zmm0, %zmm13
        vpsrlq    $32, %zmm12, %zmm14
        vpsrlq    $32, %zmm5, %zmm12
        vpsrlq    $32, %zmm8, %zmm8
        vpaddq    %zmm14, %zmm1, %zmm1
        vpaddq    %zmm13, %zmm2, %zmm2
        vpaddq    %zmm15, %zmm3, %zmm15
        vpaddq    %zmm12, %zmm4, %zmm3
        vpandq    %zmm6, %zmm0, %zmm13
        vpaddq    %zmm1, %zmm13, %zmm4
        vpaddq    %zmm4, %zmm8, %zmm14
        vpsrlq    $32, %zmm14, %zmm0
        vpandq    %zmm6, %zmm10, %zmm10
        vpaddq    %zmm2, %zmm10, %zmm1
        vpaddq    %zmm1, %zmm0, %zmm8

/*
 * Now round at the 2^-9 bit position for reduction mod pi/2^8
 * instead of the original 2pi (but still with the same 2pi scaling).
 * Use a shifter of 2^43 + 2^42.
 * The N we get is our final version; it has an offset of
 * 2^9 because of the implicit integer bit, and anyway for negative
 * starting value it's a 2s complement thing. But we need to mask
 * off the exponent part anyway so it's fine.
 */
        vpbroadcastq .L_2il0floatpacket.25(%rip), %zmm1
        vpandq    %zmm6, %zmm7, %zmm7
        vpaddq    %zmm3, %zmm7, %zmm13
        vpsrlq    $32, %zmm8, %zmm3
        vpandq    %zmm6, %zmm5, %zmm5
        vpaddq    %zmm15, %zmm5, %zmm2
        vpaddq    %zmm2, %zmm3, %zmm15
        vpsrlq    $32, %zmm15, %zmm12
        vpaddq    %zmm13, %zmm12, %zmm5

/* Assemble reduced argument from the pieces */
        vpandq    %zmm6, %zmm14, %zmm10
        vpandq    %zmm6, %zmm15, %zmm7
        vpsllq    $32, %zmm5, %zmm6
        vpsllq    $32, %zmm8, %zmm5
        vpaddq    %zmm7, %zmm6, %zmm4
        vpaddq    %zmm10, %zmm5, %zmm10
        vpsrlq    $12, %zmm4, %zmm6

/*
 * We want to incorporate the original sign now too.
 * Do it here for convenience in getting the right N value,
 * though we could wait right to the end if we were prepared
 * to modify the sign of N later too.
 * So get the appropriate sign mask now (or sooner).
 */
        vpandq    .L_2il0floatpacket.23(%rip){1to8}, %zmm11, %zmm0
        vpandq    .L_2il0floatpacket.28(%rip){1to8}, %zmm10, %zmm13
        vpsllq    $28, %zmm13, %zmm14

/*
 * Create floating-point high part, implicitly adding integer bit 1
 * Incorporate overall sign at this stage too.
 */
        vpxorq    .L_2il0floatpacket.24(%rip){1to8}, %zmm0, %zmm8
        vporq     %zmm8, %zmm6, %zmm2
        vaddpd    {rn-sae}, %zmm2, %zmm1, %zmm12
        vsubpd    {rn-sae}, %zmm1, %zmm12, %zmm3
        vsubpd    {rn-sae}, %zmm3, %zmm2, %zmm7

/*
 * Create floating-point low and medium parts, respectively
 * lo_23, ... lo_0, 0, ..., 0
 * hi_11, ... hi_0, lo_63, ..., lo_24
 * then subtract off the implicitly added integer bits,
 * 2^-104 and 2^-52, respectively.
 * Put the original sign into all of them at this stage.
 */
        vpxorq    .L_2il0floatpacket.27(%rip){1to8}, %zmm0, %zmm6
        vporq     %zmm6, %zmm14, %zmm15
        vpandq    .L_2il0floatpacket.30(%rip){1to8}, %zmm4, %zmm4
        vsubpd    {rn-sae}, %zmm6, %zmm15, %zmm8

/*
 * If the magnitude of the input is <= 2^-20, then
 * just pass through the input, since no reduction will be needed and
 * the main path will only work accurately if the reduced argument is
 * about >= 2^-70 (which it is for all large pi multiples)
 */
        vpbroadcastq .L_2il0floatpacket.34(%rip), %zmm14
        vandpd    .L_2il0floatpacket.33(%rip){1to8}, %zmm11, %zmm15
        vpsrlq    $24, %zmm10, %zmm6
        vcmppd    $26, {sae}, %zmm14, %zmm15, %k2
        vcmppd    $22, {sae}, %zmm14, %zmm15, %k3
        vpxorq    .L_2il0floatpacket.29(%rip){1to8}, %zmm0, %zmm1
        vpsllq    $40, %zmm4, %zmm0
        vporq     %zmm6, %zmm0, %zmm0
        vporq     %zmm1, %zmm0, %zmm4
        vsubpd    {rn-sae}, %zmm1, %zmm4, %zmm2

/* Now add them up into 2 reasonably aligned pieces */
        vaddpd    {rn-sae}, %zmm2, %zmm7, %zmm13
        vsubpd    {rn-sae}, %zmm13, %zmm7, %zmm7
        vaddpd    {rn-sae}, %zmm7, %zmm2, %zmm3
        vaddpd    {rn-sae}, %zmm8, %zmm3, %zmm0
        vpbroadcastq .L_2il0floatpacket.32(%rip), %zmm8

/* Grab our final N value as an integer, appropriately masked mod 2^9 */
        vpandq    .L_2il0floatpacket.26(%rip){1to8}, %zmm12, %zmm5

/*
 * Now multiply those numbers all by 2 pi, reasonably accurately.
 * (RHi + RLo) * (pi_lead + pi_trail) ~=
 * RHi * pi_lead + (RHi * pi_trail + RLo * pi_lead)
 */
        vpbroadcastq .L_2il0floatpacket.31(%rip), %zmm12

/* The output is _VRES_Z (high) + _VRES_E (low), and the integer part is _VRES_IND */
        vpmovqd   %zmm5, %ymm4
        vmulpd    {rn-sae}, %zmm12, %zmm13, %zmm6
        vmovaps   %zmm12, %zmm10
        vfmsub213pd {rn-sae}, %zmm6, %zmm13, %zmm10
        vfmadd213pd {rn-sae}, %zmm10, %zmm8, %zmm13
        vpbroadcastq .L_2il0floatpacket.37(%rip), %zmm10
        vfmadd213pd {rn-sae}, %zmm13, %zmm12, %zmm0
        vpbroadcastq .L_2il0floatpacket.36(%rip), %zmm12
        vmovaps   %zmm9, %zmm8
        vpandnq   %zmm15, %zmm15, %zmm8{%k2}
        vpandnq   %zmm15, %zmm15, %zmm9{%k3}
        vandpd    %zmm11, %zmm9, %zmm9
        vandpd    %zmm6, %zmm8, %zmm14
        vorpd     %zmm14, %zmm9, %zmm6
        vandpd    %zmm0, %zmm8, %zmm9
        vmovdqu   .L_2il0floatpacket.35(%rip), %ymm0
        vpsrlq    $32, %zmm6, %zmm5
        vpmovqd   %zmm5, %ymm5
        vpsrld    $31, %ymm5, %ymm1
        vpsubd    %ymm1, %ymm0, %ymm2

/*
 * ----------------------------------------------------------
 * End of large arguments path
 * ----------------------------------------------------------
 * Merge results from main and large paths:
 */
        vmovups   (%rsp), %zmm1
        vpaddd    %ymm2, %ymm4, %ymm3
        vpsrld    $4, %ymm3, %ymm14
        vpslld    $4, %ymm14, %ymm7
        vpmovzxdq %ymm14, %zmm1{%k1}
        vpsubd    %ymm7, %ymm4, %ymm8
        vcvtdq2pd %ymm8, %zmm15
        vmovups   64(%rsp), %zmm8
        vfmadd231pd {rn-sae}, %zmm15, %zmm10, %zmm9
        vaddpd    {rn-sae}, %zmm6, %zmm9, %zmm13
        vfmadd213pd {rn-sae}, %zmm13, %zmm12, %zmm15
        vblendmpd %zmm15, %zmm8, %zmm8{%k1}
        jmp       .LBL_1_2
	.align    16,0x90

	.cfi_endproc

	.type	__svml_tan8,@function
	.size	__svml_tan8,.-__svml_tan8
..LN__svml_tan8.0:

.L_2__routine_start___svml_dtan_cout_rare_internal_1:

	.align    16,0x90

__svml_dtan_cout_rare_internal:


	.cfi_startproc
..L65:

        xorl      %eax, %eax
        movzwl    6(%rdi), %ecx
        movsd     (%rdi), %xmm1
        andl      $32752, %ecx
        movb      7(%rdi), %dl
        andb      $127, %dl
        movsd     %xmm1, -8(%rsp)
        movb      %dl, -1(%rsp)
        cmpl      $32752, %ecx
        je        .LBL_2_3


        ret

.LBL_2_3:

        cmpl      $0, -8(%rsp)
        jne       .LBL_2_6


        cmpl      $2146435072, -4(%rsp)
        jne       .LBL_2_6


        movsd     __dtan_la_Tab(%rip), %xmm0
        movl      $1, %eax
        mulsd     %xmm0, %xmm1
        movsd     %xmm1, (%rsi)
        ret

.LBL_2_6:

        mulsd     %xmm1, %xmm1
        movsd     %xmm1, (%rsi)
        ret
	.align    16,0x90

	.cfi_endproc

	.type	__svml_dtan_cout_rare_internal,@function
	.size	__svml_dtan_cout_rare_internal,.-__svml_dtan_cout_rare_internal
..LN__svml_dtan_cout_rare_internal.1:

	.section .rodata, "a"
	.align 64
	.align 64
__svml_dtan_data_internal:
	.long	1841940611
	.long	1075076912
	.long	1841940611
	.long	1075076912
	.long	1841940611
	.long	1075076912
	.long	1841940611
	.long	1075076912
	.long	1841940611
	.long	1075076912
	.long	1841940611
	.long	1075076912
	.long	1841940611
	.long	1075076912
	.long	1841940611
	.long	1075076912
	.long	1413754136
	.long	1070146043
	.long	1413754136
	.long	1070146043
	.long	1413754136
	.long	1070146043
	.long	1413754136
	.long	1070146043
	.long	1413754136
	.long	1070146043
	.long	1413754136
	.long	1070146043
	.long	1413754136
	.long	1070146043
	.long	1413754136
	.long	1070146043
	.long	856972294
	.long	1013032486
	.long	856972294
	.long	1013032486
	.long	856972294
	.long	1013032486
	.long	856972294
	.long	1013032486
	.long	856972294
	.long	1013032486
	.long	856972294
	.long	1013032486
	.long	856972294
	.long	1013032486
	.long	856972294
	.long	1013032486
	.long	688016905
	.long	958143697
	.long	688016905
	.long	958143697
	.long	688016905
	.long	958143697
	.long	688016905
	.long	958143697
	.long	688016905
	.long	958143697
	.long	688016905
	.long	958143697
	.long	688016905
	.long	958143697
	.long	688016905
	.long	958143697
	.long	855638016
	.long	1013032486
	.long	855638016
	.long	1013032486
	.long	855638016
	.long	1013032486
	.long	855638016
	.long	1013032486
	.long	855638016
	.long	1013032486
	.long	855638016
	.long	1013032486
	.long	855638016
	.long	1013032486
	.long	855638016
	.long	1013032486
	.long	3773204808
	.long	979655686
	.long	3773204808
	.long	979655686
	.long	3773204808
	.long	979655686
	.long	3773204808
	.long	979655686
	.long	3773204808
	.long	979655686
	.long	3773204808
	.long	979655686
	.long	3773204808
	.long	979655686
	.long	3773204808
	.long	979655686
	.long	0
	.long	2147483648
	.long	3763679576
	.long	1070167541
	.long	2583490354
	.long	1071284857
	.long	716700048
	.long	1071997368
	.long	0
	.long	1072693248
	.long	3797578849
	.long	1073213976
	.long	859807206
	.long	1073958991
	.long	3795994481
	.long	1075059710
	.long	4294967295
	.long	4293918719
	.long	3795994481
	.long	3222543358
	.long	859807206
	.long	3221442639
	.long	3797578849
	.long	3220697624
	.long	0
	.long	3220176896
	.long	716700048
	.long	3219481016
	.long	2583490354
	.long	3218768505
	.long	3763679576
	.long	3217651189
	.long	0
	.long	2147483648
	.long	1732516166
	.long	1009710547
	.long	2973134505
	.long	1014008623
	.long	561165941
	.long	1015523525
	.long	0
	.long	0
	.long	1767150223
	.long	1016338938
	.long	4129738197
	.long	1017254245
	.long	4275273486
	.long	1015088902
	.long	0
	.long	4238344192
	.long	4275273486
	.long	3162572550
	.long	4129738197
	.long	3164737893
	.long	1767150223
	.long	3163822586
	.long	0
	.long	0
	.long	561165941
	.long	3163007173
	.long	2973134505
	.long	3161492271
	.long	1732516166
	.long	3157194195
	.long	1431655900
	.long	1070945621
	.long	1431655900
	.long	1070945621
	.long	1431655900
	.long	1070945621
	.long	1431655900
	.long	1070945621
	.long	1431655900
	.long	1070945621
	.long	1431655900
	.long	1070945621
	.long	1431655900
	.long	1070945621
	.long	1431655900
	.long	1070945621
	.long	285935618
	.long	1069617425
	.long	285935618
	.long	1069617425
	.long	285935618
	.long	1069617425
	.long	285935618
	.long	1069617425
	.long	285935618
	.long	1069617425
	.long	285935618
	.long	1069617425
	.long	285935618
	.long	1069617425
	.long	285935618
	.long	1069617425
	.long	1218258378
	.long	1068212666
	.long	1218258378
	.long	1068212666
	.long	1218258378
	.long	1068212666
	.long	1218258378
	.long	1068212666
	.long	1218258378
	.long	1068212666
	.long	1218258378
	.long	1068212666
	.long	1218258378
	.long	1068212666
	.long	1218258378
	.long	1068212666
	.long	1716452265
	.long	1066820779
	.long	1716452265
	.long	1066820779
	.long	1716452265
	.long	1066820779
	.long	1716452265
	.long	1066820779
	.long	1716452265
	.long	1066820779
	.long	1716452265
	.long	1066820779
	.long	1716452265
	.long	1066820779
	.long	1716452265
	.long	1066820779
	.long	3351902813
	.long	1065508044
	.long	3351902813
	.long	1065508044
	.long	3351902813
	.long	1065508044
	.long	3351902813
	.long	1065508044
	.long	3351902813
	.long	1065508044
	.long	3351902813
	.long	1065508044
	.long	3351902813
	.long	1065508044
	.long	3351902813
	.long	1065508044
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	4294967295
	.long	2147483647
	.long	4294967295
	.long	2147483647
	.long	4294967295
	.long	2147483647
	.long	4294967295
	.long	2147483647
	.long	4294967295
	.long	2147483647
	.long	4294967295
	.long	2147483647
	.long	4294967295
	.long	2147483647
	.long	4294967295
	.long	2147483647
	.long	0
	.long	2146435072
	.long	0
	.long	2146435072
	.long	0
	.long	2146435072
	.long	0
	.long	2146435072
	.long	0
	.long	2146435072
	.long	0
	.long	2146435072
	.long	0
	.long	2146435072
	.long	0
	.long	2146435072
	.long	0
	.long	1127743488
	.long	0
	.long	1127743488
	.long	0
	.long	1127743488
	.long	0
	.long	1127743488
	.long	0
	.long	1127743488
	.long	0
	.long	1127743488
	.long	0
	.long	1127743488
	.long	0
	.long	1127743488
	.long	1413480448
	.long	1065951739
	.long	1413480448
	.long	1065951739
	.long	1413480448
	.long	1065951739
	.long	1413480448
	.long	1065951739
	.long	1413480448
	.long	1065951739
	.long	1413480448
	.long	1065951739
	.long	1413480448
	.long	1065951739
	.long	1413480448
	.long	1065951739
	.long	442499072
	.long	1029747809
	.long	442499072
	.long	1029747809
	.long	442499072
	.long	1029747809
	.long	442499072
	.long	1029747809
	.long	442499072
	.long	1029747809
	.long	442499072
	.long	1029747809
	.long	442499072
	.long	1029747809
	.long	442499072
	.long	1029747809
	.long	771977331
	.long	993204618
	.long	771977331
	.long	993204618
	.long	771977331
	.long	993204618
	.long	771977331
	.long	993204618
	.long	771977331
	.long	993204618
	.long	771977331
	.long	993204618
	.long	771977331
	.long	993204618
	.long	771977331
	.long	993204618
	.long	511
	.long	0
	.long	511
	.long	0
	.long	511
	.long	0
	.long	511
	.long	0
	.long	511
	.long	0
	.long	511
	.long	0
	.long	511
	.long	0
	.long	511
	.long	0
	.long	1841940611
	.long	1079271216
	.long	1841940611
	.long	1079271216
	.long	1841940611
	.long	1079271216
	.long	1841940611
	.long	1079271216
	.long	1841940611
	.long	1079271216
	.long	1841940611
	.long	1079271216
	.long	1841940611
	.long	1079271216
	.long	1841940611
	.long	1079271216
	.long	1413754136
	.long	1073291771
	.long	856972295
	.long	1016178214
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1072693248
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1431655769
	.long	1070945621
	.long	0
	.long	0
	.long	285114973
	.long	1069617425
	.long	0
	.long	0
	.long	3223786787
	.long	1068212718
	.long	1570481342
	.long	1073278903
	.long	2595802427
	.long	1014369127
	.long	0
	.long	0
	.long	75380366
	.long	1065951822
	.long	2999349512
	.long	3155830414
	.long	0
	.long	1072693248
	.long	2911696896
	.long	1059306942
	.long	451505506
	.long	1065952070
	.long	3089290734
	.long	1070946463
	.long	3705470160
	.long	1065403174
	.long	2568117119
	.long	1069618857
	.long	643394266
	.long	1064501452
	.long	501385640
	.long	1068202863
	.long	1727208548
	.long	1073266035
	.long	2531251621
	.long	3162145225
	.long	0
	.long	0
	.long	612012528
	.long	1067000646
	.long	549540416
	.long	3153143513
	.long	0
	.long	1072693248
	.long	2064191488
	.long	1061404484
	.long	4042886527
	.long	1067001638
	.long	149923164
	.long	1070948993
	.long	104448276
	.long	1066453158
	.long	2663431480
	.long	1069623160
	.long	4100741303
	.long	1065552806
	.long	3253087105
	.long	1068220130
	.long	1883935754
	.long	1073253167
	.long	2988318241
	.long	3163734974
	.long	0
	.long	0
	.long	3919601072
	.long	1067637674
	.long	2391050108
	.long	3158536696
	.long	0
	.long	1072693248
	.long	1803946752
	.long	1062615560
	.long	2247013512
	.long	1067639351
	.long	3277043521
	.long	1070953215
	.long	578207594
	.long	1067006653
	.long	1619551617
	.long	1069630354
	.long	2647612950
	.long	1066183733
	.long	2691224773
	.long	1068193529
	.long	2040662960
	.long	1073240299
	.long	4270081753
	.long	3164530442
	.long	0
	.long	0
	.long	2319126882
	.long	1068050215
	.long	2052183488
	.long	1006305725
	.long	0
	.long	1072693248
	.long	2962101888
	.long	1063503197
	.long	3966050657
	.long	1068054193
	.long	3178024141
	.long	1070959142
	.long	1146831000
	.long	1067507381
	.long	1630003540
	.long	1069640468
	.long	2204487828
	.long	1066612393
	.long	783402153
	.long	1068243248
	.long	2197390165
	.long	1073227431
	.long	3038089327
	.long	1016395176
	.long	0
	.long	0
	.long	3930011523
	.long	1068463253
	.long	3574549514
	.long	3158343440
	.long	0
	.long	1072693248
	.long	930814464
	.long	1064233867
	.long	3218126771
	.long	1068471034
	.long	873740287
	.long	1070966789
	.long	1580060562
	.long	1067789097
	.long	1505280051
	.long	1069653548
	.long	2221338604
	.long	1066937592
	.long	2641117162
	.long	1068312400
	.long	2354117371
	.long	1073214563
	.long	3512651629
	.long	1015129272
	.long	0
	.long	0
	.long	3439278129
	.long	1068687929
	.long	1073016494
	.long	1012431341
	.long	0
	.long	1072693248
	.long	2934847296
	.long	1064716672
	.long	2625704508
	.long	1068694663
	.long	3692939630
	.long	1070976174
	.long	3072165553
	.long	1068074393
	.long	223733198
	.long	1069669653
	.long	1158528257
	.long	1067269817
	.long	3197805108
	.long	1068357327
	.long	2510844577
	.long	1073201695
	.long	498468876
	.long	3161098111
	.long	0
	.long	0
	.long	1972895212
	.long	1068895135
	.long	3208334872
	.long	1010638399
	.long	0
	.long	1072693248
	.long	4107853760
	.long	1065246718
	.long	2229347510
	.long	1068905849
	.long	4172638464
	.long	1070987323
	.long	212469442
	.long	1068364021
	.long	1989165656
	.long	1069688852
	.long	4131454312
	.long	1067530480
	.long	3453538591
	.long	1068469531
	.long	2667571783
	.long	1073188827
	.long	1614402419
	.long	3163482232
	.long	0
	.long	0
	.long	2354785698
	.long	1069102779
	.long	959196066
	.long	1012432163
	.long	0
	.long	1072693248
	.long	2828230112
	.long	1065606626
	.long	3693284264
	.long	1069118808
	.long	1996210616
	.long	1071000265
	.long	3053541878
	.long	1068578846
	.long	1133334569
	.long	1069711236
	.long	977677893
	.long	1067705915
	.long	347264115
	.long	1068436198
	.long	2824298989
	.long	1073175959
	.long	2088964722
	.long	3164313480
	.long	0
	.long	0
	.long	931121154
	.long	1069310926
	.long	1121692745
	.long	3159637885
	.long	0
	.long	1072693248
	.long	709357472
	.long	1065955296
	.long	29972165
	.long	1069333808
	.long	254360647
	.long	1071015033
	.long	3732687627
	.long	1068729157
	.long	3224370683
	.long	1069736900
	.long	202592779
	.long	1067887231
	.long	1420423314
	.long	1068491719
	.long	2981026194
	.long	1073163091
	.long	924239062
	.long	1016612139
	.long	0
	.long	0
	.long	982366697
	.long	1069519640
	.long	1160213218
	.long	3158385836
	.long	0
	.long	1072693248
	.long	404678272
	.long	1066346403
	.long	1419607756
	.long	1069549319
	.long	1800406832
	.long	1071031665
	.long	3144361055
	.long	1068882825
	.long	1744148040
	.long	1069765960
	.long	3211597674
	.long	1068075289
	.long	404952187
	.long	1068609434
	.long	3137753400
	.long	1073150223
	.long	3579918397
	.long	1015563197
	.long	0
	.long	0
	.long	2242453104
	.long	1069638253
	.long	1143545012
	.long	3159361608
	.long	0
	.long	1072693248
	.long	1016752608
	.long	1066591110
	.long	1270001991
	.long	1069659269
	.long	1410785275
	.long	1071050206
	.long	1103929722
	.long	1069040275
	.long	3268015215
	.long	1069798554
	.long	957665168
	.long	1068270980
	.long	1067662813
	.long	1068610012
	.long	3294480606
	.long	1073137355
	.long	1489607747
	.long	3151418907
	.long	0
	.long	0
	.long	1940260416
	.long	1069743276
	.long	4144715194
	.long	1012060053
	.long	0
	.long	1072693248
	.long	1135818128
	.long	1066829850
	.long	867512158
	.long	1069770656
	.long	3821819258
	.long	1071070704
	.long	2890864845
	.long	1069201948
	.long	918459419
	.long	1069834828
	.long	797472495
	.long	1068475275
	.long	4093958433
	.long	1068635569
	.long	3451207812
	.long	1073124487
	.long	1547135652
	.long	3163048307
	.long	0
	.long	0
	.long	1959743063
	.long	1069848682
	.long	607727474
	.long	1013558026
	.long	0
	.long	1072693248
	.long	225834432
	.long	1067090614
	.long	732340398
	.long	1069883626
	.long	3221152670
	.long	1071093215
	.long	2810588359
	.long	1069368307
	.long	612200458
	.long	1069874948
	.long	645427023
	.long	1068594071
	.long	1966294363
	.long	1068694214
	.long	3607935018
	.long	1073111619
	.long	4202814986
	.long	3164096517
	.long	0
	.long	0
	.long	2429422902
	.long	1069954505
	.long	3297515896
	.long	3160006592
	.long	0
	.long	1072693248
	.long	528565440
	.long	1067373727
	.long	2395999282
	.long	1069998329
	.long	1512953454
	.long	1071117799
	.long	2702142222
	.long	1069539835
	.long	3515899576
	.long	1069919100
	.long	2767905785
	.long	1068706391
	.long	3127207959
	.long	1068808800
	.long	3764662223
	.long	1073098751
	.long	3105356094
	.long	1016829101
	.long	0
	.long	0
	.long	1425758568
	.long	1070060780
	.long	1224256500
	.long	1011856982
	.long	0
	.long	1072693248
	.long	2086713664
	.long	1067564956
	.long	1688303979
	.long	1070114921
	.long	938619730
	.long	1071144522
	.long	526120622
	.long	1069632280
	.long	3819295667
	.long	1069967496
	.long	2059466202
	.long	1068824653
	.long	4035080204
	.long	1068903334
	.long	3921389429
	.long	1073085883
	.long	3647185164
	.long	1015997122
	.long	0
	.long	0
	.long	3763679576
	.long	1070167541
	.long	448465424
	.long	1009708707
	.long	0
	.long	1072693248
	.long	64931152
	.long	1067729411
	.long	3593250320
	.long	1070233561
	.long	1127364104
	.long	1071173457
	.long	21320967
	.long	1069723988
	.long	4036384112
	.long	1070020367
	.long	3394693835
	.long	1068949482
	.long	4171096556
	.long	1068939101
	.long	4078116635
	.long	1073073015
	.long	39665264
	.long	1013608617
	.long	0
	.long	0
	.long	4023957346
	.long	1070274825
	.long	2652053984
	.long	1012417284
	.long	0
	.long	1072693248
	.long	1548193280
	.long	1067905620
	.long	2849460080
	.long	1070354416
	.long	586099284
	.long	1071204684
	.long	2426038050
	.long	1069819083
	.long	1209882932
	.long	1070077962
	.long	204929667
	.long	1069081561
	.long	2951798640
	.long	1069077118
	.long	4234843841
	.long	1073060147
	.long	1479868884
	.long	3162614382
	.long	0
	.long	0
	.long	3947580658
	.long	1070382668
	.long	4153467944
	.long	1013228820
	.long	0
	.long	1072693248
	.long	1306970432
	.long	1068093810
	.long	3320658510
	.long	1070477657
	.long	2638377394
	.long	1071238289
	.long	1218095092
	.long	1069917865
	.long	506153462
	.long	1070140566
	.long	2096772755
	.long	1069221614
	.long	2058474785
	.long	1069148484
	.long	96603751
	.long	1073047280
	.long	2021697954
	.long	3163879555
	.long	0
	.long	0
	.long	4068253482
	.long	1070491107
	.long	3098804354
	.long	1013932712
	.long	0
	.long	1072693248
	.long	3555466384
	.long	1068294223
	.long	3714292293
	.long	1070599779
	.long	58004546
	.long	1071274368
	.long	2889916307
	.long	1070020649
	.long	2725223781
	.long	1070208483
	.long	1960889140
	.long	1069370437
	.long	2173066945
	.long	1069259864
	.long	253330956
	.long	1073034412
	.long	991505830
	.long	1017046064
	.long	0
	.long	0
	.long	1971725998
	.long	1070598138
	.long	2628060598
	.long	1013294571
	.long	0
	.long	1072693248
	.long	3850616508
	.long	1068503032
	.long	715485429
	.long	1070664058
	.long	2831870887
	.long	1071313021
	.long	276112813
	.long	1070127773
	.long	4071473622
	.long	1070282053
	.long	3678820497
	.long	1069528889
	.long	928174949
	.long	1069392578
	.long	410058162
	.long	1073021544
	.long	4004709614
	.long	1016250595
	.long	0
	.long	0
	.long	1197423420
	.long	1070653011
	.long	176404688
	.long	1013428940
	.long	0
	.long	1072693248
	.long	3614211808
	.long	1068615863
	.long	2638984897
	.long	1070729808
	.long	1695604536
	.long	1071354361
	.long	2372043566
	.long	1070239593
	.long	136909610
	.long	1070361652
	.long	3640039434
	.long	1069622710
	.long	4088329288
	.long	1069449448
	.long	566785368
	.long	1073008676
	.long	2301849815
	.long	1014658654
	.long	0
	.long	0
	.long	1026922473
	.long	1070708240
	.long	4275552176
	.long	1013759157
	.long	0
	.long	1072693248
	.long	865263972
	.long	1068735226
	.long	1189261043
	.long	1070797132
	.long	602895285
	.long	1071398507
	.long	457371263
	.long	1070356493
	.long	1846858239
	.long	1070447677
	.long	3060498109
	.long	1069713009
	.long	4267979072
	.long	1069596604
	.long	723512574
	.long	1072995808
	.long	2825204233
	.long	3161855698
	.long	0
	.long	0
	.long	2664093517
	.long	1070763845
	.long	3574550848
	.long	1009790086
	.long	0
	.long	1072693248
	.long	2465054196
	.long	1068861279
	.long	3495559184
	.long	1070866134
	.long	2180799467
	.long	1071445588
	.long	3760731789
	.long	1070478879
	.long	3108073741
	.long	1070540584
	.long	1955989611
	.long	1069809654
	.long	166740423
	.long	1069654305
	.long	880239780
	.long	1072982940
	.long	4135548218
	.long	3163662592
	.long	0
	.long	0
	.long	895247324
	.long	1070819848
	.long	3063999364
	.long	3161289029
	.long	0
	.long	1072693248
	.long	729983844
	.long	1068994194
	.long	3631919317
	.long	1070936926
	.long	2031343772
	.long	1071495745
	.long	1505593195
	.long	1070601643
	.long	519428998
	.long	1070618477
	.long	3710689648
	.long	1069913254
	.long	3156018952
	.long	1069741343
	.long	1036966986
	.long	1072970072
	.long	1122344434
	.long	3164458061
	.long	0
	.long	0
	.long	1412148370
	.long	1070876269
	.long	2689479793
	.long	3161477235
	.long	0
	.long	1072693248
	.long	1375255072
	.long	1069134151
	.long	1493266594
	.long	1071009624
	.long	487968719
	.long	1071549129
	.long	4010395236
	.long	1070668994
	.long	2875570533
	.long	1070672565
	.long	2596803931
	.long	1070024476
	.long	2843491938
	.long	1069843051
	.long	1193694191
	.long	1072957204
	.long	1890859349
	.long	1016467558
	.long	0
	.long	0
	.long	3782220567
	.long	1070933130
	.long	2488122455
	.long	1014753484
	.long	0
	.long	1072693248
	.long	91372748
	.long	1069281344
	.long	318776714
	.long	1071084349
	.long	1791786485
	.long	1071605902
	.long	245863368
	.long	1070739794
	.long	3306699314
	.long	1070730899
	.long	492155886
	.long	1070144051
	.long	504705110
	.long	1069961586
	.long	1350421397
	.long	1072944336
	.long	1218191675
	.long	1015274036
	.long	0
	.long	0
	.long	1606100959
	.long	1070990455
	.long	3385800912
	.long	1013630535
	.long	0
	.long	1072693248
	.long	2255302008
	.long	1069435977
	.long	54085333
	.long	1071161229
	.long	2504974906
	.long	1071655456
	.long	4169947836
	.long	1070814310
	.long	2891827400
	.long	1070793803
	.long	1030744926
	.long	1070272780
	.long	2378131433
	.long	1070036506
	.long	1507148603
	.long	1072931468
	.long	1086374101
	.long	3160519057
	.long	0
	.long	0
	.long	4158096969
	.long	1071048265
	.long	3224523546
	.long	1013419306
	.long	0
	.long	1072693248
	.long	4240313622
	.long	1069572894
	.long	3745046477
	.long	1071240398
	.long	1955120038
	.long	1071687503
	.long	2661530988
	.long	1070892836
	.long	2615081511
	.long	1070861626
	.long	3172686013
	.long	1070411548
	.long	3468537064
	.long	1070173373
	.long	1663875809
	.long	1072918600
	.long	3908862373
	.long	3163337468
	.long	0
	.long	0
	.long	2200768388
	.long	1071106586
	.long	302579070
	.long	3162432935
	.long	0
	.long	1072693248
	.long	2338248308
	.long	1069657986
	.long	2617403810
	.long	1071322000
	.long	60078074
	.long	1071721530
	.long	2753634668
	.long	1070975684
	.long	626020947
	.long	1070934758
	.long	25846105
	.long	1070561329
	.long	1123371528
	.long	1070307499
	.long	1820603015
	.long	1072905732
	.long	3236194699
	.long	3164241098
	.long	0
	.long	0
	.long	2528366626
	.long	1071165441
	.long	3808697588
	.long	3160880158
	.long	0
	.long	1072693248
	.long	1938261590
	.long	1069747146
	.long	4282399603
	.long	1071406183
	.long	1922568281
	.long	1071757646
	.long	2428166646
	.long	1071063193
	.long	717464826
	.long	1071013623
	.long	2205664064
	.long	1070659645
	.long	1525962144
	.long	1070438229
	.long	1977330220
	.long	1072892864
	.long	4071976381
	.long	1016684520
	.long	0
	.long	0
	.long	2390846448
	.long	1071224856
	.long	3510510824
	.long	1014225707
	.long	0
	.long	1072693248
	.long	1208409702
	.long	1069840505
	.long	1006199412
	.long	1071493107
	.long	2504246873
	.long	1071795971
	.long	1040653613
	.long	1071155729
	.long	1713268218
	.long	1071098686
	.long	462276291
	.long	1070747214
	.long	4239134769
	.long	1070603989
	.long	2134057426
	.long	1072879996
	.long	1285458442
	.long	1015707961
	.long	0
	.long	0
	.long	2583490354
	.long	1071284857
	.long	27154665
	.long	1014008337
	.long	0
	.long	1072693248
	.long	2550940470
	.long	1069938201
	.long	2616040293
	.long	1071582937
	.long	2529278955
	.long	1071836633
	.long	2244405849
	.long	1071253687
	.long	2064814125
	.long	1071190460
	.long	1527853242
	.long	1070842074
	.long	354304617
	.long	1070708112
	.long	2290784632
	.long	1072867128
	.long	2660353234
	.long	1011979525
	.long	0
	.long	0
	.long	3368427099
	.long	1071345471
	.long	2187847358
	.long	3161070788
	.long	0
	.long	1072693248
	.long	1708021028
	.long	1070040382
	.long	459972195
	.long	1071660262
	.long	3210514347
	.long	1071879770
	.long	955408873
	.long	1071357497
	.long	1911386086
	.long	1071289511
	.long	3563284025
	.long	1070944951
	.long	547392146
	.long	1070790031
	.long	2447511838
	.long	1072854260
	.long	3841595606
	.long	3162903543
	.long	0
	.long	0
	.long	408840118
	.long	1071406727
	.long	4048150272
	.long	1011870625
	.long	0
	.long	1072693248
	.long	2794370512
	.long	1070147203
	.long	1550993351
	.long	1071708355
	.long	1305093678
	.long	1071925532
	.long	1748211158
	.long	1071467623
	.long	3244212116
	.long	1071396455
	.long	2145665988
	.long	1071056651
	.long	1098394636
	.long	1070891096
	.long	2604239044
	.long	1072841392
	.long	1055077667
	.long	3164024136
	.long	0
	.long	0
	.long	2191706112
	.long	1071468652
	.long	333457838
	.long	1013847700
	.long	0
	.long	1072693248
	.long	3435408456
	.long	1070258830
	.long	487319719
	.long	1071758184
	.long	1713711059
	.long	1071974079
	.long	130753286
	.long	1071584571
	.long	909725129
	.long	1071511974
	.long	2762442355
	.long	1071178067
	.long	932699057
	.long	1071027067
	.long	2760966249
	.long	1072828524
	.long	1958126117
	.long	1016901483
	.long	0
	.long	0
	.long	220675706
	.long	1071531278
	.long	3158460941
	.long	1014240139
	.long	0
	.long	1072693248
	.long	1565030890
	.long	1070375439
	.long	335263687
	.long	1071809854
	.long	419816567
	.long	1072025586
	.long	2098048102
	.long	1071676780
	.long	257229334
	.long	1071636821
	.long	3464457175
	.long	1071310192
	.long	2745951745
	.long	1071158056
	.long	2917693455
	.long	1072815656
	.long	676362605
	.long	1016106015
	.long	0
	.long	0
	.long	3649427950
	.long	1071594634
	.long	4113913432
	.long	1011178545
	.long	0
	.long	1072693248
	.long	2424145432
	.long	1070497215
	.long	2715404890
	.long	1071863477
	.long	2238283673
	.long	1072080240
	.long	1186867211
	.long	1071742923
	.long	660318792
	.long	1071708248
	.long	2091427899
	.long	1071454132
	.long	2591062578
	.long	1071291190
	.long	3074420661
	.long	1072802788
	.long	1873363667
	.long	1014080331
	.long	0
	.long	0
	.long	2606398711
	.long	1071651713
	.long	2533078623
	.long	1015070425
	.long	0
	.long	1072693248
	.long	2151361100
	.long	1070610226
	.long	1532348096
	.long	1071919174
	.long	408604530
	.long	1072138246
	.long	1039538740
	.long	1071813375
	.long	1234598850
	.long	1071781284
	.long	692507560
	.long	1071611119
	.long	4090315823
	.long	1071451983
	.long	3231147867
	.long	1072789920
	.long	3253690381
	.long	3162434021
	.long	0
	.long	0
	.long	1505561933
	.long	1071684172
	.long	2813788960
	.long	3158646164
	.long	0
	.long	1072693248
	.long	2997217365
	.long	1070676584
	.long	2681788575
	.long	1071977071
	.long	3989126937
	.long	1072199822
	.long	1049115510
	.long	1071888491
	.long	511642057
	.long	1071860360
	.long	4034728042
	.long	1071713598
	.long	1788921063
	.long	1071639550
	.long	3387875073
	.long	1072777052
	.long	3168927931
	.long	3163807173
	.long	0
	.long	0
	.long	419968236
	.long	1071717047
	.long	131821922
	.long	3163445114
	.long	0
	.long	1072693248
	.long	2960267235
	.long	1070745841
	.long	1037049094
	.long	1072037305
	.long	1447436626
	.long	1072265209
	.long	3197232138
	.long	1071968658
	.long	4004685083
	.long	1071946034
	.long	1412816411
	.long	1071807280
	.long	3184767140
	.long	1071740610
	.long	3544602278
	.long	1072764184
	.long	4139243149
	.long	1017118445
	.long	0
	.long	0
	.long	4270735453
	.long	1071750355
	.long	3565353881
	.long	3162826177
	.long	0
	.long	1072693248
	.long	2915760230
	.long	1070818115
	.long	638580392
	.long	1072100020
	.long	2252948193
	.long	1072334663
	.long	1194731830
	.long	1072054301
	.long	1353002943
	.long	1072038927
	.long	422364777
	.long	1071909798
	.long	3669605864
	.long	1071842023
	.long	3701329484
	.long	1072751316
	.long	2857479637
	.long	1016322977
	.long	0
	.long	0
	.long	4010047525
	.long	1071784117
	.long	3339961616
	.long	1012589046
	.long	0
	.long	1072693248
	.long	966450668
	.long	1070893532
	.long	803989702
	.long	1072165371
	.long	1934453966
	.long	1072408465
	.long	1455077850
	.long	1072145882
	.long	2709264343
	.long	1072139721
	.long	1918064545
	.long	1072022114
	.long	1476733491
	.long	1071960574
	.long	3858056690
	.long	1072738448
	.long	2007897202
	.long	1014948181
	.long	0
	.long	0
	.long	2591355529
	.long	1071818352
	.long	2351346752
	.long	1014424505
	.long	0
	.long	1072693248
	.long	3206845590
	.long	1070972224
	.long	4023349050
	.long	1072233523
	.long	3838822434
	.long	1072486918
	.long	3122427157
	.long	1072243909
	.long	2261703628
	.long	1072249178
	.long	1764870085
	.long	1072145307
	.long	3201774398
	.long	1072093301
	.long	4014783896
	.long	1072725580
	.long	3119156846
	.long	3161566171
	.long	0
	.long	0
	.long	2298773501
	.long	1071853080
	.long	4027796943
	.long	3162574088
	.long	0
	.long	1072693248
	.long	312748411
	.long	1071054335
	.long	2723913091
	.long	1072304655
	.long	3207063211
	.long	1072570353
	.long	1845780128
	.long	1072348940
	.long	752659336
	.long	1072368142
	.long	1097207438
	.long	1072280589
	.long	407427378
	.long	1072246732
	.long	4171511102
	.long	1072712712
	.long	987810900
	.long	3163590211
	.long	0
	.long	0
	.long	681498355
	.long	1071888323
	.long	3542613740
	.long	3163311347
	.long	0
	.long	1072693248
	.long	168222332
	.long	1071140014
	.long	776736583
	.long	1072378956
	.long	2405401100
	.long	1072659128
	.long	430546584
	.long	1072461586
	.long	2974083787
	.long	1072497552
	.long	849442051
	.long	1072429322
	.long	2727422945
	.long	1072409680
	.long	33271012
	.long	1072699845
	.long	2269574412
	.long	3164385679
	.long	0
	.long	0
	.long	3684492763
	.long	1071924102
	.long	2905877726
	.long	3163193209
	.long	0
	.long	1072693248
	.long	585640690
	.long	1071229422
	.long	1068301595
	.long	1072456629
	.long	2953900489
	.long	1072723440
	.long	950237889
	.long	1072582519
	.long	109735426
	.long	1072638455
	.long	792610960
	.long	1072593041
	.long	2904161704
	.long	1072594405
	.long	379996435
	.long	1072680706
	.long	2807708551
	.long	3162614199
	.long	0
	.long	0
	.long	142288477
	.long	1071960443
	.long	1256321232
	.long	3162016428
	.long	0
	.long	1072693248
	.long	925074332
	.long	1071322730
	.long	2134728327
	.long	1072537893
	.long	1886408555
	.long	1072773770
	.long	101893267
	.long	1072702864
	.long	2020085224
	.long	1072742631
	.long	2955803851
	.long	1072733362
	.long	343310803
	.long	1072748161
	.long	693450846
	.long	1072654970
	.long	3218699017
	.long	1015418799
	.long	0
	.long	0
	.long	716700048
	.long	1071997368
	.long	1043856658
	.long	1015522810
	.long	0
	.long	1072693248
	.long	1097907398
	.long	1071420120
	.long	3912524876
	.long	1072622983
	.long	4107784306
	.long	1072827408
	.long	937056065
	.long	1072772766
	.long	4128945134
	.long	1072826390
	.long	1032262383
	.long	1072832917
	.long	1825923576
	.long	1072864423
	.long	1006905258
	.long	1072629234
	.long	3348558652
	.long	3159471942
	.long	0
	.long	0
	.long	3841229102
	.long	1072034903
	.long	1782837269
	.long	1016032992
	.long	0
	.long	1072693248
	.long	866896685
	.long	1071521786
	.long	2743435231
	.long	1072702700
	.long	3017984361
	.long	1072884608
	.long	1491919678
	.long	1072848040
	.long	1507535811
	.long	1072917857
	.long	1966124961
	.long	1072942914
	.long	381628626
	.long	1072992589
	.long	1320359670
	.long	1072603498
	.long	1908355032
	.long	3163192705
	.long	0
	.long	0
	.long	343219638
	.long	1072073077
	.long	2238302242
	.long	1014248160
	.long	0
	.long	1072693248
	.long	3768132253
	.long	1071627934
	.long	3895546569
	.long	1072749460
	.long	1613056473
	.long	1072945645
	.long	3409294908
	.long	1072929183
	.long	3138966349
	.long	1073017844
	.long	3799067273
	.long	1073064607
	.long	359470462
	.long	1073133087
	.long	1633814081
	.long	1072577762
	.long	3941137777
	.long	1014659019
	.long	0
	.long	0
	.long	3860971117
	.long	1072111915
	.long	2891637204
	.long	1015730508
	.long	0
	.long	1073741824
	.long	80822128
	.long	3219034205
	.long	2077635078
	.long	1072798543
	.long	1552870456
	.long	1073010820
	.long	3607700650
	.long	1073016746
	.long	249538440
	.long	1073127265
	.long	2233833066
	.long	1073199420
	.long	2436428201
	.long	1073293556
	.long	1947268493
	.long	1072552026
	.long	1185916272
	.long	3161855333
	.long	0
	.long	0
	.long	3978924239
	.long	1072151449
	.long	680748782
	.long	3163204529
	.long	0
	.long	1073741824
	.long	3528982175
	.long	3218918413
	.long	3018376188
	.long	1072850106
	.long	2093086652
	.long	1073080463
	.long	451029217
	.long	1073111339
	.long	1097121536
	.long	1073247142
	.long	2277789226
	.long	1073348969
	.long	149166906
	.long	1073478822
	.long	2260722904
	.long	1072526290
	.long	3285965784
	.long	1015852724
	.long	0
	.long	0
	.long	997287281
	.long	1072191710
	.long	3612277486
	.long	1015833408
	.long	0
	.long	1073741824
	.long	3037674984
	.long	3218797431
	.long	2624215131
	.long	1072904321
	.long	464779768
	.long	1073154936
	.long	3700763630
	.long	1073213636
	.long	1200926910
	.long	1073378628
	.long	2934576429
	.long	1073515096
	.long	3287041855
	.long	1073681673
	.long	2574177316
	.long	1072500554
	.long	2889755040
	.long	1013031024
	.long	0
	.long	0
	.long	2079874795
	.long	1072232729
	.long	1885505696
	.long	3161954077
	.long	0
	.long	1073741824
	.long	1656469901
	.long	3218670989
	.long	823761676
	.long	1072961373
	.long	2491445454
	.long	1073234634
	.long	1584341555
	.long	1073324391
	.long	1976688438
	.long	1073523018
	.long	3918313340
	.long	1073699902
	.long	3178717687
	.long	1073829019
	.long	2887631728
	.long	1072474818
	.long	1841088264
	.long	3162758780
	.long	0
	.long	0
	.long	258163604
	.long	1072274542
	.long	3299923807
	.long	1015128357
	.long	0
	.long	1073741824
	.long	2865277319
	.long	3218538798
	.long	3110066496
	.long	1073021461
	.long	723074889
	.long	1073319994
	.long	1618819009
	.long	1073444438
	.long	1176243936
	.long	1073681775
	.long	1223848987
	.long	1073823806
	.long	2369274010
	.long	1073957581
	.long	3201086139
	.long	1072449082
	.long	4185319304
	.long	1015274218
	.long	0
	.long	0
	.long	719595600
	.long	1072317184
	.long	114895218
	.long	3162143748
	.long	0
	.long	1073741824
	.long	3811788216
	.long	3218400550
	.long	1264738948
	.long	1073084804
	.long	1958933439
	.long	1073411493
	.long	4246310163
	.long	1073574708
	.long	1367842247
	.long	1073799186
	.long	4112596162
	.long	1073938662
	.long	2100954442
	.long	1074105254
	.long	3514540551
	.long	1072423346
	.long	2102765473
	.long	3160518326
	.long	0
	.long	0
	.long	4189730214
	.long	1072360693
	.long	873070368
	.long	1011168520
	.long	0
	.long	1073741824
	.long	760634709
	.long	3218255915
	.long	2067827628
	.long	1073151636
	.long	1386756408
	.long	1073509659
	.long	404552634
	.long	1073716242
	.long	1143062645
	.long	1073895515
	.long	555342321
	.long	1074067009
	.long	3470170865
	.long	1074271418
	.long	3827994963
	.long	1072397610
	.long	941734744
	.long	3163337286
	.long	0
	.long	0
	.long	3247184626
	.long	1072405111
	.long	1245412628
	.long	1014426137
	.long	0
	.long	1073741824
	.long	2381906911
	.long	3218104536
	.long	1010924947
	.long	1073222214
	.long	2481381005
	.long	1073615072
	.long	1356661294
	.long	1073806011
	.long	1527070582
	.long	1074001844
	.long	3061693388
	.long	1074210655
	.long	1993183195
	.long	1074463435
	.long	4141449374
	.long	1072371874
	.long	1579411055
	.long	1014369858
	.long	0
	.long	0
	.long	166164867
	.long	1072450480
	.long	2304058808
	.long	3159574198
	.long	0
	.long	1073741824
	.long	2798667120
	.long	3217812323
	.long	4203596829
	.long	1073296815
	.long	1473339147
	.long	1073728374
	.long	4243021652
	.long	1073889850
	.long	1638747017
	.long	1074119375
	.long	1512673096
	.long	1074371687
	.long	2338684787
	.long	1074678259
	.long	159936490
	.long	1072346139
	.long	3547642993
	.long	3162144494
	.long	0
	.long	0
	.long	3342355758
	.long	1072496843
	.long	1667048854
	.long	1014547708
	.long	0
	.long	1073741824
	.long	2271288998
	.long	3217480250
	.long	4111284847
	.long	1073375744
	.long	2796272966
	.long	1073796048
	.long	1730314448
	.long	1073981280
	.long	1813001552
	.long	1074249474
	.long	603251414
	.long	1074552509
	.long	3079042420
	.long	1074856525
	.long	473390901
	.long	1072320403
	.long	4252586071
	.long	1015708143
	.long	0
	.long	0
	.long	2308259791
	.long	1072544250
	.long	4116261952
	.long	1009282152
	.long	0
	.long	1073741824
	.long	411343470
	.long	3217132228
	.long	3760686807
	.long	1073459331
	.long	4171367426
	.long	1073861688
	.long	1878500541
	.long	1074081117
	.long	108767834
	.long	1074393698
	.long	2824899539
	.long	1074755901
	.long	3321235986
	.long	1074997439
	.long	786845313
	.long	1072294667
	.long	627570489
	.long	1011980987
	.long	0
	.long	0
	.long	2949191430
	.long	1072592750
	.long	3989207545
	.long	1015613028
	.long	0
	.long	1073741824
	.long	2909112164
	.long	3216503445
	.long	3682557528
	.long	1073547937
	.long	4184421718
	.long	1073932454
	.long	380803270
	.long	1074190282
	.long	3399744364
	.long	1074553823
	.long	3842864947
	.long	1074887742
	.long	748075417
	.long	1075157395
	.long	1100299725
	.long	1072268931
	.long	874467977
	.long	3162903361
	.long	0
	.long	0
	.long	2668510354
	.long	1072642397
	.long	1701265160
	.long	3159690220
	.long	0
	.long	1073741824
	.long	1404962112
	.long	3215492924
	.long	3327263454
	.long	1073641957
	.long	1987673213
	.long	1074008829
	.long	231222583
	.long	1074309811
	.long	1756671652
	.long	1074731884
	.long	3921518313
	.long	1075017100
	.long	2691193077
	.long	1075341426
	.long	1413754136
	.long	1072243195
	.long	856972295
	.long	1015129638
	.long	0
	.long	1072693248
	.long	3072795147
	.long	3218177217
	.long	2531204756
	.long	1014677625
	.long	0
	.long	1071644672
	.long	573338107
	.long	3216966390
	.long	1757081539
	.long	3216009405
	.long	1043378852
	.long	1067696161
	.long	2698850995
	.long	3213513876
	.long	495882814
	.long	1064911053
	.long	3842541443
	.long	3210988571
	.long	2725665262
	.long	1061135526
	.long	1727208548
	.long	1072217459
	.long	2531251621
	.long	3161096649
	.long	0
	.long	1072693248
	.long	963359913
	.long	3218157757
	.long	2464811394
	.long	1013284229
	.long	0
	.long	1071644672
	.long	1498613974
	.long	3216992493
	.long	4138103782
	.long	3215985771
	.long	2840947366
	.long	1067675293
	.long	209537977
	.long	3213463826
	.long	3590357530
	.long	1064864171
	.long	73959547
	.long	3210940132
	.long	1511606549
	.long	1062390487
	.long	2040662960
	.long	1072191723
	.long	4270081753
	.long	3163481866
	.long	0
	.long	1072693248
	.long	4002124759
	.long	3218138375
	.long	3453342566
	.long	3160907529
	.long	0
	.long	1071644672
	.long	774562305
	.long	3217018021
	.long	377617675
	.long	3215942442
	.long	2495478050
	.long	1067655031
	.long	1508050213
	.long	3213415186
	.long	1921336534
	.long	1064819067
	.long	975923672
	.long	3210893594
	.long	2289965811
	.long	1060505011
	.long	2354117371
	.long	1072165987
	.long	3512651629
	.long	1014080696
	.long	0
	.long	1072693248
	.long	373017919
	.long	3218119072
	.long	3647199073
	.long	3162144529
	.long	0
	.long	1070596096
	.long	933359764
	.long	1069535705
	.long	1391441088
	.long	3215896667
	.long	53106651
	.long	1067635358
	.long	4157835043
	.long	3213367904
	.long	4055868050
	.long	1064775611
	.long	1529330423
	.long	3210848884
	.long	325278885
	.long	1062325273
	.long	2667571783
	.long	1072140251
	.long	1614402419
	.long	3162433656
	.long	0
	.long	1072693248
	.long	4148643959
	.long	3218099843
	.long	2887822595
	.long	3161698583
	.long	0
	.long	1070596096
	.long	4101770849
	.long	1069511300
	.long	1167052501
	.long	3215851606
	.long	2569727718
	.long	1067616256
	.long	1681502493
	.long	3213321931
	.long	3685967947
	.long	1064733780
	.long	2973319847
	.long	3210805911
	.long	472683240
	.long	1062549181
	.long	2981026194
	.long	1072114515
	.long	924239062
	.long	1015563563
	.long	0
	.long	1072693248
	.long	3747221382
	.long	3218080689
	.long	1912303920
	.long	3161550059
	.long	0
	.long	1070596096
	.long	1727686788
	.long	1069487445
	.long	681802874
	.long	3215807238
	.long	2531789465
	.long	1067597711
	.long	331202638
	.long	3213277217
	.long	2627596737
	.long	1064693515
	.long	566979211
	.long	3210764596
	.long	2424396143
	.long	1061741710
	.long	3294480606
	.long	1072088779
	.long	1489607747
	.long	3150370331
	.long	0
	.long	1072693248
	.long	1167858038
	.long	3218043472
	.long	2180016084
	.long	3158357424
	.long	0
	.long	1070596096
	.long	772792549
	.long	1069464130
	.long	3356872770
	.long	3215763542
	.long	3524844777
	.long	1067579707
	.long	1521756467
	.long	3213233715
	.long	3931969740
	.long	1064654734
	.long	2908114613
	.long	3210710011
	.long	3523476919
	.long	1061374592
	.long	3607935018
	.long	1072063043
	.long	4202814986
	.long	3163047941
	.long	0
	.long	1072693248
	.long	955408364
	.long	3218005450
	.long	2659929740
	.long	1013295819
	.long	0
	.long	1070596096
	.long	659174847
	.long	1069441347
	.long	2081794994
	.long	3215720500
	.long	2956616392
	.long	1067562230
	.long	1461867065
	.long	3213191381
	.long	879427918
	.long	1064617384
	.long	209598599
	.long	3210633576
	.long	60948554
	.long	1061529630
	.long	3921389429
	.long	1072037307
	.long	3647185164
	.long	1014948546
	.long	0
	.long	1072693248
	.long	1526324265
	.long	3217967566
	.long	4056331296
	.long	1010311990
	.long	0
	.long	1070596096
	.long	1830915900
	.long	1069419088
	.long	602185706
	.long	3215678092
	.long	555210277
	.long	1067545266
	.long	3690136761
	.long	3213150171
	.long	4115917121
	.long	1064581405
	.long	589230607
	.long	3210560017
	.long	3446641745
	.long	1062382364
	.long	4234843841
	.long	1072011571
	.long	1479868884
	.long	3161565806
	.long	0
	.long	1072693248
	.long	2049842017
	.long	3217929817
	.long	2491067243
	.long	1013757722
	.long	0
	.long	1070596096
	.long	1431833764
	.long	1069397346
	.long	551048935
	.long	3215636299
	.long	2575917112
	.long	1067528800
	.long	1559777140
	.long	3213110045
	.long	743322344
	.long	1064546764
	.long	1974991822
	.long	3210489213
	.long	3456487690
	.long	1062460308
	.long	253330956
	.long	1071985836
	.long	991505830
	.long	1015997488
	.long	0
	.long	1072693248
	.long	1894214069
	.long	3217892200
	.long	2182661104
	.long	3159506151
	.long	0
	.long	1070596096
	.long	1869133310
	.long	1069376113
	.long	1368144043
	.long	3215595103
	.long	242783364
	.long	1067512821
	.long	3692616565
	.long	3213070961
	.long	2890683542
	.long	1064513419
	.long	2407947192
	.long	3210421028
	.long	3811448635
	.long	1060561671
	.long	566785368
	.long	1071960100
	.long	2301849815
	.long	1013610078
	.long	0
	.long	1072693248
	.long	620845047
	.long	3217854712
	.long	314567190
	.long	3160767875
	.long	0
	.long	1070596096
	.long	2198158867
	.long	1069355382
	.long	222370857
	.long	3215554487
	.long	4028822361
	.long	1067497314
	.long	3610397327
	.long	3213032882
	.long	171354848
	.long	1064481283
	.long	693369877
	.long	3210355362
	.long	164303632
	.long	1061632559
	.long	880239780
	.long	1071934364
	.long	4135548218
	.long	3162614016
	.long	0
	.long	1072693248
	.long	2273547475
	.long	3217817349
	.long	4162051772
	.long	3158862098
	.long	0
	.long	1070596096
	.long	2392983423
	.long	1069335146
	.long	526970473
	.long	3215514433
	.long	628322840
	.long	1067482270
	.long	924033682
	.long	3212995771
	.long	854746898
	.long	1064450346
	.long	4004963123
	.long	3210292086
	.long	1474471147
	.long	1060747995
	.long	1193694191
	.long	1071908628
	.long	1890859349
	.long	1015418982
	.long	0
	.long	1072693248
	.long	2488075121
	.long	3217780109
	.long	1856288899
	.long	1013767581
	.long	0
	.long	1070596096
	.long	3322930631
	.long	1069315398
	.long	3277607933
	.long	3215474924
	.long	130253043
	.long	1067467675
	.long	1057196725
	.long	3212959591
	.long	3278604843
	.long	1064420531
	.long	4197429676
	.long	3210231108
	.long	1627850508
	.long	1062399550
	.long	1507148603
	.long	1071882892
	.long	1086374101
	.long	3159470481
	.long	0
	.long	1072693248
	.long	1371601499
	.long	3217742989
	.long	3050542164
	.long	3159667536
	.long	0
	.long	1070596096
	.long	2434996648
	.long	1069296132
	.long	2688049294
	.long	3215435945
	.long	1586964757
	.long	1067453518
	.long	2102156419
	.long	3212924308
	.long	2974407249
	.long	1064391847
	.long	249545848
	.long	3210172310
	.long	118754132
	.long	1061128535
	.long	1820603015
	.long	1071857156
	.long	3236194699
	.long	3163192522
	.long	0
	.long	1072693248
	.long	3497432648
	.long	3217705985
	.long	3403982176
	.long	3159219350
	.long	0
	.long	1070596096
	.long	2322009458
	.long	1069277341
	.long	3303184211
	.long	3215397479
	.long	3079386980
	.long	1067429210
	.long	3158396776
	.long	3212889889
	.long	500122468
	.long	1064364204
	.long	3226667041
	.long	3210115604
	.long	1839377740
	.long	1062167069
	.long	2134057426
	.long	1071831420
	.long	1285458442
	.long	1014659385
	.long	0
	.long	1072693248
	.long	719981842
	.long	3217669096
	.long	3396799415
	.long	1013109014
	.long	0
	.long	1070596096
	.long	2111724540
	.long	1069259019
	.long	2754706540
	.long	3215359511
	.long	790037588
	.long	1067402587
	.long	4097286835
	.long	3212856302
	.long	2918356813
	.long	1064337614
	.long	980543884
	.long	3210060878
	.long	1759882364
	.long	3204685716
	.long	2447511838
	.long	1071805684
	.long	3841595606
	.long	3161854967
	.long	0
	.long	1072693248
	.long	2234510834
	.long	3217632317
	.long	1077830624
	.long	3156276711
	.long	0
	.long	1070596096
	.long	1741599179
	.long	1069241160
	.long	288887764
	.long	3215322026
	.long	438316229
	.long	1067376778
	.long	2074161178
	.long	3212810170
	.long	2132352512
	.long	1064311981
	.long	321000190
	.long	3210008077
	.long	2574541397
	.long	1061549566
	.long	2760966249
	.long	1071779948
	.long	1958126117
	.long	1015852907
	.long	0
	.long	1072693248
	.long	212483252
	.long	3217595647
	.long	3496985358
	.long	1013615952
	.long	0
	.long	1070596096
	.long	1939348811
	.long	1069223758
	.long	1001569645
	.long	3215285008
	.long	764537997
	.long	1067351764
	.long	2326690580
	.long	3212746141
	.long	60750538
	.long	1064270005
	.long	2745665118
	.long	3209957083
	.long	1586280401
	.long	1060807371
	.long	3074420661
	.long	1071754212
	.long	1873363667
	.long	1013031755
	.long	0
	.long	1072693248
	.long	4156511795
	.long	3217559081
	.long	1791782369
	.long	3160930544
	.long	0
	.long	1070596096
	.long	4204226438
	.long	1069206807
	.long	2895376501
	.long	3215248443
	.long	3600185398
	.long	1067327526
	.long	2087275384
	.long	3212683597
	.long	347703885
	.long	1064222543
	.long	301495371
	.long	3209907826
	.long	876688468
	.long	3206713553
	.long	3387875073
	.long	1071728476
	.long	3168927931
	.long	3162758597
	.long	0
	.long	1072693248
	.long	2240974125
	.long	3217522619
	.long	3786179077
	.long	1012998644
	.long	0
	.long	1070596096
	.long	2199065489
	.long	1069190303
	.long	1118792048
	.long	3215212318
	.long	875809397
	.long	1067304048
	.long	2927616918
	.long	3212622482
	.long	3494337727
	.long	1064176822
	.long	4219965681
	.long	3209860241
	.long	4249954841
	.long	1060574506
	.long	3701329484
	.long	1071702740
	.long	2857479637
	.long	1015274401
	.long	0
	.long	1072693248
	.long	4257115870
	.long	3217486256
	.long	39728399
	.long	3160493923
	.long	0
	.long	1070596096
	.long	3207767847
	.long	1069174239
	.long	797077063
	.long	3215176618
	.long	287980713
	.long	1067281311
	.long	1596666478
	.long	3212562743
	.long	2775364
	.long	1064132847
	.long	3823518214
	.long	3209814240
	.long	3101392498
	.long	1060290467
	.long	4014783896
	.long	1071677004
	.long	3119156846
	.long	3160517595
	.long	0
	.long	1072693248
	.long	2953879462
	.long	3217449991
	.long	651182982
	.long	3159737324
	.long	0
	.long	1070596096
	.long	2348802841
	.long	1069158611
	.long	4095204310
	.long	3215141329
	.long	3730066556
	.long	1067259298
	.long	2541564470
	.long	3212504327
	.long	1012137407
	.long	1064090541
	.long	3007493349
	.long	3209769752
	.long	304719606
	.long	1058732349
	.long	33271012
	.long	1071651269
	.long	2269574412
	.long	3163337103
	.long	0
	.long	1072693248
	.long	4098310206
	.long	3217413820
	.long	678867490
	.long	1013738598
	.long	0
	.long	1070596096
	.long	4033975882
	.long	1069143413
	.long	1577500332
	.long	3215106440
	.long	2904242360
	.long	1067237995
	.long	383406633
	.long	3212447185
	.long	511745182
	.long	1064049810
	.long	3472404602
	.long	3209726723
	.long	3493824410
	.long	1061932631
	.long	693450846
	.long	1071606394
	.long	3218699017
	.long	1014370223
	.long	0
	.long	1072693248
	.long	701484222
	.long	3217377742
	.long	1189134796
	.long	1013549080
	.long	0
	.long	1070596096
	.long	478132658
	.long	1069128642
	.long	2223654597
	.long	3215071936
	.long	706118045
	.long	1067217386
	.long	1066336370
	.long	3212391267
	.long	3310393795
	.long	1064010705
	.long	1920669086
	.long	3209678984
	.long	1209448644
	.long	3206743753
	.long	1320359670
	.long	1071554922
	.long	1908355032
	.long	3162144129
	.long	0
	.long	1072693248
	.long	3079104278
	.long	3217341752
	.long	2593237472
	.long	3159688198
	.long	0
	.long	1070596096
	.long	1749059456
	.long	1069114291
	.long	3316773722
	.long	3215037805
	.long	2555119580
	.long	1067197455
	.long	59306268
	.long	3212336527
	.long	1944001738
	.long	1063973040
	.long	3607966944
	.long	3209598336
	.long	1548002341
	.long	1061515569
	.long	1947268493
	.long	1071503450
	.long	1185916272
	.long	3160806757
	.long	0
	.long	1072693248
	.long	192708373
	.long	3217305850
	.long	1013564964
	.long	3160609117
	.long	0
	.long	1070596096
	.long	1393443609
	.long	1069100357
	.long	3282624502
	.long	3215004035
	.long	3423023493
	.long	1067178189
	.long	3539283610
	.long	3212282918
	.long	1358653293
	.long	1063936902
	.long	1383848795
	.long	3209520160
	.long	1170879540
	.long	3207439782
	.long	2574177316
	.long	1071451978
	.long	2889755040
	.long	1011982448
	.long	0
	.long	1072693248
	.long	2595342993
	.long	3217270031
	.long	1222182906
	.long	3159411259
	.long	0
	.long	1070596096
	.long	2192927435
	.long	1069086835
	.long	1645422722
	.long	3214970614
	.long	648269084
	.long	1067159575
	.long	2319737250
	.long	3212230398
	.long	1121494038
	.long	1063902126
	.long	897314833
	.long	3209444434
	.long	2659165416
	.long	1060259181
	.long	3201086139
	.long	1071400506
	.long	4185319304
	.long	1014225642
	.long	0
	.long	1072693248
	.long	3772940467
	.long	3217234294
	.long	3998164722
	.long	1013347606
	.long	0
	.long	1070596096
	.long	3265990537
	.long	1069073721
	.long	3279971426
	.long	3214937529
	.long	2447085124
	.long	1067141598
	.long	1649985592
	.long	3212178923
	.long	3334399491
	.long	1063868754
	.long	454991634
	.long	3209370985
	.long	1938913543
	.long	3204215126
	.long	3827994963
	.long	1071349034
	.long	941734744
	.long	3162288710
	.long	0
	.long	1072693248
	.long	1615317960
	.long	3217198637
	.long	1400808192
	.long	1007479474
	.long	0
	.long	1070596096
	.long	55251407
	.long	1069061012
	.long	2380564598
	.long	3214875524
	.long	2078095393
	.long	1067124247
	.long	3759475360
	.long	3212128451
	.long	800747756
	.long	1063836662
	.long	3361780736
	.long	3209299766
	.long	3643416391
	.long	1062249457
	.long	159936490
	.long	1071297563
	.long	3547642993
	.long	3161095918
	.long	0
	.long	1072693248
	.long	2707383656
	.long	3217163056
	.long	554608178
	.long	3160521551
	.long	0
	.long	1070596096
	.long	1495139838
	.long	1069048702
	.long	183291521
	.long	3214810633
	.long	3540584774
	.long	1067107509
	.long	3317896890
	.long	3212078943
	.long	3844035447
	.long	1063805952
	.long	3695033041
	.long	3209230570
	.long	3731872264
	.long	3207727795
	.long	786845313
	.long	1071246091
	.long	627570489
	.long	1010932411
	.long	0
	.long	1072693248
	.long	850612498
	.long	3217127550
	.long	3957837480
	.long	1012739461
	.long	0
	.long	1070596096
	.long	4230382869
	.long	1069036788
	.long	2678394493
	.long	3214746346
	.long	42048419
	.long	1067091374
	.long	61549345
	.long	3212030360
	.long	3326100968
	.long	1063776440
	.long	377176410
	.long	3209163411
	.long	1539171840
	.long	3206122484
	.long	1413754136
	.long	1071194619
	.long	856972295
	.long	1014081062
	.long	0
	.long	1072693248
	.long	2829232582
	.long	3217092115
	.long	1037840994
	.long	1012412582
	.long	0
	.long	1070596096
	.long	1194701220
	.long	1069025268
	.long	1485063562
	.long	3214682643
	.long	3996610205
	.long	1067075828
	.long	2865183303
	.long	3211982662
	.long	2185691626
	.long	1063748117
	.long	435906073
	.long	3209098183
	.long	1583986254
	.long	1061922957
	.long	2040662960
	.long	1071143147
	.long	4270081753
	.long	3162433290
	.long	0
	.long	1072693248
	.long	2636871548
	.long	3217056750
	.long	3635907996
	.long	3159234439
	.long	0
	.long	1070596096
	.long	3074865967
	.long	1069014136
	.long	2827594877
	.long	3214619501
	.long	4018616278
	.long	1067060863
	.long	3744630852
	.long	3211935814
	.long	967547233
	.long	1063721046
	.long	1523268333
	.long	3209034706
	.long	370739821
	.long	1057912410
	.long	2667571783
	.long	1071091675
	.long	1614402419
	.long	3161385080
	.long	0
	.long	1072693248
	.long	1600848691
	.long	3217011737
	.long	3799011534
	.long	3158825865
	.long	0
	.long	1070596096
	.long	3940625627
	.long	1069003390
	.long	1694054617
	.long	3214556900
	.long	3525212904
	.long	1067046468
	.long	3400170401
	.long	3211889780
	.long	2025409789
	.long	1063695095
	.long	1861695371
	.long	3208972985
	.long	2099431114
	.long	1058280175
	.long	3294480606
	.long	1071040203
	.long	1489607747
	.long	3149321755
	.long	0
	.long	1072693248
	.long	163325898
	.long	3216941271
	.long	272531312
	.long	3158872717
	.long	0
	.long	1070596096
	.long	2709640876
	.long	1068993027
	.long	3241569416
	.long	3214494818
	.long	3499770285
	.long	1067032633
	.long	2428916378
	.long	3211844525
	.long	796870242
	.long	1063670282
	.long	2401385551
	.long	3208912885
	.long	417337253
	.long	3207953615
	.long	3921389429
	.long	1070988731
	.long	3647185164
	.long	1013899970
	.long	0
	.long	1072693248
	.long	2357811071
	.long	3216870929
	.long	1815527340
	.long	3157476592
	.long	0
	.long	1070596096
	.long	3138003653
	.long	1068983043
	.long	1253835798
	.long	3214433236
	.long	2444914170
	.long	1067019349
	.long	1113961907
	.long	3211800015
	.long	1725681990
	.long	1063646503
	.long	2521427548
	.long	3208854407
	.long	2469316955
	.long	1061668474
	.long	253330956
	.long	1070937260
	.long	991505830
	.long	1014948912
	.long	0
	.long	1072693248
	.long	1152452440
	.long	3216800708
	.long	2988871182
	.long	1011946816
	.long	0
	.long	1070596096
	.long	926263936
	.long	1068973436
	.long	3845070713
	.long	3214372132
	.long	626397172
	.long	1067006607
	.long	2777085683
	.long	3211724145
	.long	2172777456
	.long	1063623822
	.long	3188732906
	.long	3208797386
	.long	250185439
	.long	1062033866
	.long	880239780
	.long	1070885788
	.long	4135548218
	.long	3161565440
	.long	0
	.long	1072693248
	.long	2555242223
	.long	3216730602
	.long	1132402578
	.long	1012350135
	.long	0
	.long	1070596096
	.long	890626350
	.long	1068964202
	.long	3331100648
	.long	3214311488
	.long	30508518
	.long	1066994398
	.long	397665884
	.long	3211637908
	.long	1048931291
	.long	1063602233
	.long	4152525626
	.long	3208741715
	.long	51864355
	.long	3208005790
	.long	1507148603
	.long	1070834316
	.long	1086374101
	.long	3158421905
	.long	0
	.long	1072693248
	.long	4132779814
	.long	3216660607
	.long	3895880616
	.long	1012226511
	.long	0
	.long	1070596096
	.long	4069764223
	.long	1068955338
	.long	1936654463
	.long	3214251284
	.long	27137259
	.long	1066982714
	.long	112447739
	.long	3211552967
	.long	3681308921
	.long	1063581561
	.long	3611917249
	.long	3208687484
	.long	2182292750
	.long	1061868473
	.long	2134057426
	.long	1070782844
	.long	1285458442
	.long	1013610809
	.long	0
	.long	1072693248
	.long	3593905368
	.long	3216590719
	.long	2989246530
	.long	1011901384
	.long	0
	.long	1070596096
	.long	3421947019
	.long	1068946843
	.long	3144465174
	.long	3214191500
	.long	1618007405
	.long	1066971547
	.long	584787122
	.long	3211469261
	.long	1465845707
	.long	1063561956
	.long	939979903
	.long	3208626315
	.long	1695262346
	.long	3204506663
	.long	2760966249
	.long	1070731372
	.long	1958126117
	.long	1014804331
	.long	0
	.long	1072693248
	.long	3078460826
	.long	3216520933
	.long	4285081068
	.long	3159473614
	.long	0
	.long	1070596096
	.long	997373752
	.long	1068938714
	.long	3045840714
	.long	3214132118
	.long	3110377520
	.long	1066960890
	.long	2323415564
	.long	3211386730
	.long	18309569
	.long	1063543250
	.long	551124926
	.long	3208522770
	.long	249182975
	.long	1061811122
	.long	3387875073
	.long	1070679900
	.long	3168927931
	.long	3161710021
	.long	0
	.long	1072693248
	.long	561239310
	.long	3216451245
	.long	210497392
	.long	3156201161
	.long	0
	.long	1070596096
	.long	3636030687
	.long	1068930947
	.long	3167463735
	.long	3214073119
	.long	73152643
	.long	1066950737
	.long	3108019154
	.long	3211305316
	.long	2100498319
	.long	1063525559
	.long	282708930
	.long	3208421353
	.long	2551775895
	.long	3207605450
	.long	4014783896
	.long	1070628428
	.long	3119156846
	.long	3159469019
	.long	0
	.long	1072693248
	.long	2730856872
	.long	3216381649
	.long	4254293712
	.long	1009711154
	.long	0
	.long	1070596096
	.long	1486032260
	.long	1068923542
	.long	1824674436
	.long	3214014485
	.long	2488677748
	.long	1066941079
	.long	1925770461
	.long	3211224962
	.long	2531500940
	.long	1063508710
	.long	3007508061
	.long	3208322267
	.long	994458799
	.long	1061806090
	.long	693450846
	.long	1070557818
	.long	3218699017
	.long	1013321647
	.long	0
	.long	1072693248
	.long	3508967486
	.long	3216312142
	.long	3021155724
	.long	1011969550
	.long	0
	.long	1070596096
	.long	2356881417
	.long	1068916495
	.long	2655870529
	.long	3213956197
	.long	352865498
	.long	1066931912
	.long	3008575053
	.long	3211145611
	.long	2622088932
	.long	1063492845
	.long	3834913422
	.long	3208225014
	.long	1497815285
	.long	3207511002
	.long	1947268493
	.long	1070454874
	.long	1185916272
	.long	3159758181
	.long	0
	.long	1072693248
	.long	1224261120
	.long	3216242720
	.long	3122470664
	.long	3159031070
	.long	0
	.long	1070596096
	.long	1648557593
	.long	1068909805
	.long	1683250226
	.long	3213898238
	.long	2306165388
	.long	1066923228
	.long	2600821843
	.long	3211067209
	.long	3406795398
	.long	1063477798
	.long	3582359304
	.long	3208129845
	.long	3199024961
	.long	1061038571
	.long	3201086139
	.long	1070351930
	.long	4185319304
	.long	1013177066
	.long	0
	.long	1072693248
	.long	2901633254
	.long	3216173377
	.long	2788221280
	.long	1009319756
	.long	0
	.long	1070596096
	.long	3820542338
	.long	1068903469
	.long	288942435
	.long	3213795740
	.long	1237884341
	.long	1066915023
	.long	1606294444
	.long	3210989702
	.long	223992189
	.long	1063463656
	.long	1499707213
	.long	3208036378
	.long	3416758850
	.long	1060053511
	.long	159936490
	.long	1070248987
	.long	3547642993
	.long	3160047342
	.long	0
	.long	1072693248
	.long	2781619933
	.long	3216104110
	.long	3145354644
	.long	1011963772
	.long	0
	.long	1070596096
	.long	911495412
	.long	1068897487
	.long	290976546
	.long	3213681030
	.long	33241894
	.long	1066907291
	.long	977031819
	.long	3210913037
	.long	119357659
	.long	1063450345
	.long	1846248446
	.long	3207944648
	.long	4276662282
	.long	1061269448
	.long	1413754136
	.long	1070146043
	.long	856972295
	.long	1013032486
	.long	0
	.long	1072693248
	.long	3789571175
	.long	3216034914
	.long	2061677164
	.long	3158802525
	.long	0
	.long	1070596096
	.long	2598855658
	.long	1068891855
	.long	628750565
	.long	3213566872
	.long	2954249465
	.long	1066900026
	.long	1468236324
	.long	3210837162
	.long	3701994667
	.long	1063437912
	.long	310421843
	.long	3207854373
	.long	393637210
	.long	3207421006
	.long	2667571783
	.long	1070043099
	.long	1614402419
	.long	3160336504
	.long	0
	.long	1072693248
	.long	700360492
	.long	3215948980
	.long	4067889956
	.long	3158309941
	.long	0
	.long	1070596096
	.long	1834243786
	.long	1068886573
	.long	92010622
	.long	3213453232
	.long	2733460853
	.long	1066893225
	.long	3349627764
	.long	3210762026
	.long	2326014508
	.long	1063426279
	.long	2491557920
	.long	3207765637
	.long	2159546056
	.long	3208108974
	.long	3921389429
	.long	1069940155
	.long	3647185164
	.long	1012851394
	.long	0
	.long	1072693248
	.long	4008519973
	.long	3215810847
	.long	1156806192
	.long	3157002565
	.long	0
	.long	1070596096
	.long	608713188
	.long	1068881639
	.long	3200221055
	.long	3213340075
	.long	2024185290
	.long	1066886883
	.long	3737414605
	.long	3210635448
	.long	4102026093
	.long	1063415403
	.long	3274310324
	.long	3207678452
	.long	4280744879
	.long	1061910091
	.long	880239780
	.long	1069837212
	.long	4135548218
	.long	3160516864
	.long	0
	.long	1072693248
	.long	2381996637
	.long	3215672832
	.long	639270832
	.long	1008532593
	.long	0
	.long	1070596096
	.long	1358563144
	.long	1068877051
	.long	2925458633
	.long	3213227369
	.long	492833071
	.long	1066880996
	.long	4104628456
	.long	3210487835
	.long	3054991409
	.long	1063405423
	.long	2594255061
	.long	3207590366
	.long	2100320188
	.long	3208479772
	.long	2134057426
	.long	1069734268
	.long	1285458442
	.long	1012562233
	.long	0
	.long	1072693248
	.long	2319612510
	.long	3215534925
	.long	1111071278
	.long	3158888695
	.long	0
	.long	1070596096
	.long	2666415304
	.long	1068872808
	.long	2071510147
	.long	3213115080
	.long	3395467108
	.long	1066875559
	.long	2423946868
	.long	3210341405
	.long	80612299
	.long	1063396131
	.long	2656867111
	.long	3207420929
	.long	3004001956
	.long	1061836610
	.long	3387875073
	.long	1069631324
	.long	3168927931
	.long	3160661445
	.long	0
	.long	1072693248
	.long	1865079417
	.long	3215397118
	.long	2937833612
	.long	1011350737
	.long	0
	.long	1070596096
	.long	3552516355
	.long	1068868909
	.long	294617109
	.long	3213003175
	.long	76221622
	.long	1066870571
	.long	3376425735
	.long	3210196061
	.long	2723275475
	.long	1063387705
	.long	3315521969
	.long	3207252871
	.long	3590370444
	.long	3207679867
	.long	693450846
	.long	1069509242
	.long	3218699017
	.long	1012273071
	.long	0
	.long	1072693248
	.long	3481201957
	.long	3215259402
	.long	1636311970
	.long	3157709699
	.long	0
	.long	1070596096
	.long	3471360849
	.long	1068865353
	.long	2600335524
	.long	3212891620
	.long	2613792462
	.long	1066866026
	.long	3847698423
	.long	3210051709
	.long	663967578
	.long	1063380003
	.long	493409710
	.long	3207087119
	.long	1127477260
	.long	3207199264
	.long	3201086139
	.long	1069303354
	.long	4185319304
	.long	1012128490
	.long	0
	.long	1072693248
	.long	1154356116
	.long	3215121770
	.long	1840040117
	.long	3159097207
	.long	0
	.long	1070596096
	.long	2308597781
	.long	1068862139
	.long	4143524575
	.long	3212723904
	.long	2850915343
	.long	1066861923
	.long	1191668407
	.long	3209908256
	.long	2952843839
	.long	1063373033
	.long	3172237721
	.long	3206923432
	.long	2684115468
	.long	1061237060
	.long	1413754136
	.long	1069097467
	.long	856972295
	.long	1011983910
	.long	0
	.long	1072693248
	.long	1858778712
	.long	3214984212
	.long	552307712
	.long	1007653492
	.long	0
	.long	1070596096
	.long	378217452
	.long	1068859266
	.long	2507068679
	.long	3212502004
	.long	2446302718
	.long	1066858259
	.long	1361087111
	.long	3209765608
	.long	761263112
	.long	1063366881
	.long	2488333367
	.long	3206760542
	.long	3134826447
	.long	3208438003
	.long	3921389429
	.long	1068891579
	.long	3647185164
	.long	1011802818
	.long	0
	.long	1072693248
	.long	3847531748
	.long	3214759425
	.long	2484055616
	.long	1005679197
	.long	0
	.long	1070596096
	.long	2714982805
	.long	1068856732
	.long	2649074978
	.long	3212280611
	.long	4275478210
	.long	1066855031
	.long	2325700054
	.long	3209556212
	.long	3568629394
	.long	1063361434
	.long	1419097007
	.long	3206599479
	.long	3667282369
	.long	3208404843
	.long	2134057426
	.long	1068685692
	.long	1285458442
	.long	1011513657
	.long	0
	.long	1072693248
	.long	4111272116
	.long	3214484558
	.long	3215369808
	.long	3153450047
	.long	0
	.long	1070596096
	.long	1892298670
	.long	1068854538
	.long	4017792039
	.long	3212059661
	.long	3825916209
	.long	1066852238
	.long	1674419437
	.long	3209273590
	.long	2761556401
	.long	1063356717
	.long	4123887123
	.long	3206334117
	.long	39803485
	.long	3207166447
	.long	693450846
	.long	1068460666
	.long	3218699017
	.long	1011224495
	.long	0
	.long	1072693248
	.long	1769817487
	.long	3214209791
	.long	226900428
	.long	3157084142
	.long	0
	.long	1070596096
	.long	3790025843
	.long	1068852682
	.long	2975452701
	.long	3211839091
	.long	2074002222
	.long	1066849878
	.long	2835406423
	.long	3208992033
	.long	2160688582
	.long	1063352737
	.long	438342636
	.long	3206016666
	.long	598064701
	.long	3205441692
	.long	1413754136
	.long	1068048891
	.long	856972295
	.long	1010935334
	.long	0
	.long	1072693248
	.long	2796861084
	.long	3213935106
	.long	4160910146
	.long	1008936729
	.long	0
	.long	1070596096
	.long	1822957307
	.long	1068851165
	.long	2426092900
	.long	3211449386
	.long	1181651297
	.long	1066847949
	.long	11113416
	.long	3208711363
	.long	2350522541
	.long	1063349490
	.long	1110749705
	.long	3205700769
	.long	1529834450
	.long	3204806002
	.long	2134057426
	.long	1067637116
	.long	1285458442
	.long	1010465081
	.long	0
	.long	1072693248
	.long	766684177
	.long	3213435536
	.long	4183922317
	.long	3156428395
	.long	0
	.long	1070596096
	.long	2709169804
	.long	1068849985
	.long	2562652885
	.long	3211009382
	.long	194304310
	.long	1066846450
	.long	285423324
	.long	3208220240
	.long	4164698302
	.long	1063346958
	.long	490316583
	.long	3205276877
	.long	1161999017
	.long	1059948714
	.long	1413754136
	.long	1067000315
	.long	856972295
	.long	1009886758
	.long	0
	.long	1072693248
	.long	1373484687
	.long	3212886398
	.long	3463806808
	.long	3156987378
	.long	0
	.long	1070596096
	.long	699034059
	.long	1068849143
	.long	4196075177
	.long	3210399801
	.long	3624794075
	.long	1066845379
	.long	1212360239
	.long	3207661374
	.long	4225594870
	.long	1063345136
	.long	4142831282
	.long	3204654696
	.long	3419780486
	.long	1061860029
	.long	1413754136
	.long	1065951739
	.long	856972295
	.long	1008838182
	.long	0
	.long	1072693248
	.long	1048210722
	.long	3211837789
	.long	322286546
	.long	3154320084
	.long	0
	.long	1070596096
	.long	3344097458
	.long	1068848637
	.long	4078525744
	.long	3209350973
	.long	4268457311
	.long	1066844737
	.long	1752199129
	.long	3206612445
	.long	2933973771
	.long	1063344076
	.long	1552329397
	.long	3203604713
	.long	4286470131
	.long	1060555865
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1072693248
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1070596096
	.long	1431655765
	.long	1068848469
	.long	0
	.long	0
	.long	381986942
	.long	1066844524
	.long	0
	.long	0
	.long	3364860459
	.long	1063343704
	.long	0
	.long	0
	.long	3669941704
	.long	1061551801
	.long	1413754136
	.long	3213435387
	.long	856972295
	.long	3156321830
	.long	0
	.long	1072693248
	.long	1048210722
	.long	1064354141
	.long	310222654
	.long	1006836436
	.long	0
	.long	1070596096
	.long	3344097458
	.long	1068848637
	.long	4078525744
	.long	1061867325
	.long	4268457311
	.long	1066844737
	.long	1752199185
	.long	1059128797
	.long	2933974085
	.long	1063344076
	.long	1541627965
	.long	1056121065
	.long	4213857106
	.long	1060555865
	.long	1413754136
	.long	3214483963
	.long	856972295
	.long	3157370406
	.long	0
	.long	1072693248
	.long	1373484687
	.long	1065402750
	.long	3463778643
	.long	1009503730
	.long	0
	.long	1070596096
	.long	699034059
	.long	1068849143
	.long	4196075177
	.long	1062916153
	.long	3624794075
	.long	1066845379
	.long	1212360238
	.long	1060177726
	.long	4225594899
	.long	1063345136
	.long	4143059094
	.long	1057171048
	.long	3416473992
	.long	1061860029
	.long	2134057426
	.long	3215120764
	.long	1285458442
	.long	3157948729
	.long	0
	.long	1072693248
	.long	766684177
	.long	1065951888
	.long	4184103028
	.long	1008944747
	.long	0
	.long	1070596096
	.long	2709169804
	.long	1068849985
	.long	2562652885
	.long	1063525734
	.long	194304310
	.long	1066846450
	.long	285423316
	.long	1060736592
	.long	4164698149
	.long	1063346958
	.long	492080240
	.long	1057793229
	.long	1233082996
	.long	1059948714
	.long	1413754136
	.long	3215532539
	.long	856972295
	.long	3158418982
	.long	0
	.long	1072693248
	.long	2796861084
	.long	1066451458
	.long	4163800696
	.long	3156420377
	.long	0
	.long	1070596096
	.long	1822957307
	.long	1068851165
	.long	2426092900
	.long	1063965738
	.long	1181651297
	.long	1066847949
	.long	11113467
	.long	1061227715
	.long	2350524377
	.long	1063349490
	.long	1099629520
	.long	1058217121
	.long	647421766
	.long	3204806003
	.long	693450846
	.long	3215944314
	.long	3218699017
	.long	3158708143
	.long	0
	.long	1072693248
	.long	1769817487
	.long	1066726143
	.long	226750770
	.long	1009600494
	.long	0
	.long	1070596096
	.long	3790025843
	.long	1068852682
	.long	2975452701
	.long	1064355443
	.long	2074002222
	.long	1066849878
	.long	2835406424
	.long	1061508385
	.long	2160688720
	.long	1063352737
	.long	438143169
	.long	1058533018
	.long	848462310
	.long	3205441692
	.long	2134057426
	.long	3216169340
	.long	1285458442
	.long	3158997305
	.long	0
	.long	1072693248
	.long	4111272116
	.long	1067000910
	.long	3213429568
	.long	1005966399
	.long	0
	.long	1070596096
	.long	1892298670
	.long	1068854538
	.long	4017792039
	.long	1064576013
	.long	3825916209
	.long	1066852238
	.long	1674419438
	.long	1061789942
	.long	2761556545
	.long	1063356717
	.long	4123711083
	.long	1058850469
	.long	104595699
	.long	3207166447
	.long	3921389429
	.long	3216375227
	.long	3647185164
	.long	3159286466
	.long	0
	.long	1072693248
	.long	3847531748
	.long	1067275777
	.long	2487971824
	.long	3153162845
	.long	0
	.long	1070596096
	.long	2714982805
	.long	1068856732
	.long	2649074978
	.long	1064796963
	.long	4275478210
	.long	1066855031
	.long	2325700056
	.long	1062072564
	.long	3568629551
	.long	1063361434
	.long	1418922081
	.long	1059115831
	.long	3702148884
	.long	3208404843
	.long	1413754136
	.long	3216581115
	.long	856972295
	.long	3159467558
	.long	0
	.long	1072693248
	.long	1858778712
	.long	1067500564
	.long	554450480
	.long	3155137140
	.long	0
	.long	1070596096
	.long	378217452
	.long	1068859266
	.long	2507068679
	.long	1065018356
	.long	2446302718
	.long	1066858259
	.long	1361087112
	.long	1062281960
	.long	761263211
	.long	1063366881
	.long	2488165471
	.long	1059276894
	.long	3156241770
	.long	3208438003
	.long	3201086139
	.long	3216787002
	.long	4185319304
	.long	3159612138
	.long	0
	.long	1072693248
	.long	1154356116
	.long	1067638122
	.long	1840130764
	.long	1011613559
	.long	0
	.long	1070596096
	.long	2308597781
	.long	1068862139
	.long	4143524575
	.long	1065240256
	.long	2850915343
	.long	1066861923
	.long	1191668406
	.long	1062424608
	.long	2952843813
	.long	1063373033
	.long	3172428186
	.long	1059439784
	.long	2686776959
	.long	1061237060
	.long	693450846
	.long	3216992890
	.long	3218699017
	.long	3159756719
	.long	0
	.long	1072693248
	.long	3481201957
	.long	1067775754
	.long	1635996196
	.long	1010226051
	.long	0
	.long	1070596096
	.long	3471360849
	.long	1068865353
	.long	2600335524
	.long	1065407972
	.long	2613792462
	.long	1066866026
	.long	3847698425
	.long	1062568061
	.long	663967590
	.long	1063380003
	.long	493016801
	.long	1059603471
	.long	1131084659
	.long	3207199264
	.long	3387875073
	.long	3217114972
	.long	3168927931
	.long	1013177797
	.long	0
	.long	1072693248
	.long	1865079417
	.long	1067913470
	.long	2938003329
	.long	3158834385
	.long	0
	.long	1070596096
	.long	3552516355
	.long	1068868909
	.long	294617109
	.long	1065519527
	.long	76221622
	.long	1066870571
	.long	3376425737
	.long	1062712413
	.long	2723275461
	.long	1063387705
	.long	3315136619
	.long	1059769223
	.long	3585984604
	.long	3207679867
	.long	2134057426
	.long	3217217916
	.long	1285458442
	.long	3160045881
	.long	0
	.long	1072693248
	.long	2319612510
	.long	1068051277
	.long	1111303258
	.long	1011405047
	.long	0
	.long	1070596096
	.long	2666415304
	.long	1068872808
	.long	2071510147
	.long	1065631432
	.long	3395467108
	.long	1066875559
	.long	2423946865
	.long	1062857757
	.long	80612341
	.long	1063396131
	.long	2657457026
	.long	1059937281
	.long	2998464421
	.long	1061836610
	.long	880239780
	.long	3217320860
	.long	4135548218
	.long	1013033216
	.long	0
	.long	1072693248
	.long	2381996637
	.long	1068189184
	.long	639224640
	.long	3156016241
	.long	0
	.long	1070596096
	.long	1358563144
	.long	1068877051
	.long	2925458633
	.long	1065743721
	.long	492833071
	.long	1066880996
	.long	4104628456
	.long	1063004187
	.long	3054991518
	.long	1063405423
	.long	2594252722
	.long	1060106718
	.long	2125241622
	.long	3208479772
	.long	3921389429
	.long	3217423803
	.long	3647185164
	.long	3160335042
	.long	0
	.long	1072693248
	.long	4008519973
	.long	1068327199
	.long	1156786808
	.long	1009518917
	.long	0
	.long	1070596096
	.long	608713188
	.long	1068881639
	.long	3200221055
	.long	1065856427
	.long	2024185290
	.long	1066886883
	.long	3737414605
	.long	1063151800
	.long	4102026005
	.long	1063415403
	.long	3274311301
	.long	1060194804
	.long	4290805031
	.long	1061910091
	.long	2667571783
	.long	3217526747
	.long	1614402419
	.long	1012852856
	.long	0
	.long	1072693248
	.long	700360492
	.long	1068465332
	.long	4067897658
	.long	1010826293
	.long	0
	.long	1070596096
	.long	1834243786
	.long	1068886573
	.long	92010622
	.long	1065969584
	.long	2733460853
	.long	1066893225
	.long	3349627764
	.long	1063278378
	.long	2326014613
	.long	1063426279
	.long	2491557128
	.long	1060281989
	.long	2183550903
	.long	3208108974
	.long	1413754136
	.long	3217629691
	.long	856972295
	.long	3160516134
	.long	0
	.long	1072693248
	.long	3789571175
	.long	1068551266
	.long	2061680020
	.long	1011318877
	.long	0
	.long	1070596096
	.long	2598855658
	.long	1068891855
	.long	628750565
	.long	1066083224
	.long	2954249465
	.long	1066900026
	.long	1468236324
	.long	1063353514
	.long	3701994745
	.long	1063437912
	.long	310421258
	.long	1060370725
	.long	429301190
	.long	3207421006
	.long	159936490
	.long	3217732635
	.long	3547642993
	.long	1012563694
	.long	0
	.long	1072693248
	.long	2781619933
	.long	1068620462
	.long	3145355490
	.long	3159447420
	.long	0
	.long	1070596096
	.long	911495412
	.long	1068897487
	.long	290976546
	.long	1066197382
	.long	33241894
	.long	1066907291
	.long	977031819
	.long	1063429389
	.long	119357612
	.long	1063450345
	.long	1846248792
	.long	1060461000
	.long	4282034711
	.long	1061269448
	.long	3201086139
	.long	3217835578
	.long	4185319304
	.long	3160660714
	.long	0
	.long	1072693248
	.long	2901633254
	.long	1068689729
	.long	2788220944
	.long	3156803404
	.long	0
	.long	1070596096
	.long	3820542338
	.long	1068903469
	.long	288942435
	.long	1066312092
	.long	1237884341
	.long	1066915023
	.long	1606294444
	.long	1063506054
	.long	223992191
	.long	1063463656
	.long	1499707196
	.long	1060552730
	.long	3415844307
	.long	1060053511
	.long	1947268493
	.long	3217938522
	.long	1185916272
	.long	1012274533
	.long	0
	.long	1072693248
	.long	1224261120
	.long	1068759072
	.long	3122469562
	.long	1011547422
	.long	0
	.long	1070596096
	.long	1648557593
	.long	1068909805
	.long	1683250226
	.long	1066414590
	.long	2306165388
	.long	1066923228
	.long	2600821843
	.long	1063583561
	.long	3406795366
	.long	1063477798
	.long	3582359528
	.long	1060646197
	.long	3206340638
	.long	1061038571
	.long	693450846
	.long	3218041466
	.long	3218699017
	.long	3160805295
	.long	0
	.long	1072693248
	.long	3508967486
	.long	1068828494
	.long	3021154406
	.long	3159453198
	.long	0
	.long	1070596096
	.long	2356881417
	.long	1068916495
	.long	2655870529
	.long	1066472549
	.long	352865498
	.long	1066931912
	.long	3008575053
	.long	1063661963
	.long	2622089007
	.long	1063492845
	.long	3834912888
	.long	1060741366
	.long	1532107502
	.long	3207511002
	.long	4014783896
	.long	3218112076
	.long	3119156846
	.long	1011985371
	.long	0
	.long	1072693248
	.long	2730856872
	.long	1068898001
	.long	4254304408
	.long	3157194802
	.long	0
	.long	1070596096
	.long	1486032260
	.long	1068923542
	.long	1824674436
	.long	1066530837
	.long	2488677748
	.long	1066941079
	.long	1925770461
	.long	1063741314
	.long	2531500863
	.long	1063508710
	.long	3007508602
	.long	1060838619
	.long	1003260451
	.long	1061806090
	.long	3387875073
	.long	3218163548
	.long	3168927931
	.long	1014226373
	.long	0
	.long	1072693248
	.long	561239310
	.long	1068967597
	.long	210518352
	.long	1008717513
	.long	0
	.long	1070596096
	.long	3636030687
	.long	1068930947
	.long	3167463735
	.long	1066589471
	.long	73152643
	.long	1066950737
	.long	3108019154
	.long	1063821668
	.long	2100498394
	.long	1063525559
	.long	282708402
	.long	1060937705
	.long	2568921764
	.long	3207605450
	.long	2760966249
	.long	3218215020
	.long	1958126117
	.long	3162287979
	.long	0
	.long	1072693248
	.long	3078460826
	.long	1069037285
	.long	4285079825
	.long	1011989966
	.long	0
	.long	1070596096
	.long	997373752
	.long	1068938714
	.long	3045840714
	.long	1066648470
	.long	3110377520
	.long	1066960890
	.long	2323415564
	.long	1063903082
	.long	18309497
	.long	1063543250
	.long	551125427
	.long	1061039122
	.long	257413032
	.long	1061811122
	.long	2134057426
	.long	3218266492
	.long	1285458442
	.long	3161094457
	.long	0
	.long	1072693248
	.long	3593905368
	.long	1069107071
	.long	2989245764
	.long	3159385032
	.long	0
	.long	1070596096
	.long	3421947019
	.long	1068946843
	.long	3144465174
	.long	1066707852
	.long	1618007405
	.long	1066971547
	.long	584787122
	.long	1063985613
	.long	1465845750
	.long	1063561956
	.long	939979596
	.long	1061142667
	.long	1773905530
	.long	3204506663
	.long	1507148603
	.long	3218317964
	.long	1086374101
	.long	1010938257
	.long	0
	.long	1072693248
	.long	4132779814
	.long	1069176959
	.long	3895881848
	.long	3159710159
	.long	0
	.long	1070596096
	.long	4069764223
	.long	1068955338
	.long	1936654463
	.long	1066767636
	.long	27137259
	.long	1066982714
	.long	112447739
	.long	1064069319
	.long	3681308850
	.long	1063581561
	.long	3611917496
	.long	1061203836
	.long	2190408434
	.long	1061868473
	.long	880239780
	.long	3218369436
	.long	4135548218
	.long	1014081792
	.long	0
	.long	1072693248
	.long	2555242223
	.long	1069246954
	.long	1132401098
	.long	3159833783
	.long	0
	.long	1070596096
	.long	890626350
	.long	1068964202
	.long	3331100648
	.long	1066827840
	.long	30508518
	.long	1066994398
	.long	397665884
	.long	1064154260
	.long	1048931375
	.long	1063602233
	.long	4152525330
	.long	1061258067
	.long	71067649
	.long	3208005790
	.long	253330956
	.long	3218420908
	.long	991505830
	.long	3162432560
	.long	0
	.long	1072693248
	.long	1152452440
	.long	1069317060
	.long	2988872515
	.long	3159430464
	.long	0
	.long	1070596096
	.long	926263936
	.long	1068973436
	.long	3845070713
	.long	1066888484
	.long	626397172
	.long	1067006607
	.long	2777085683
	.long	1064240497
	.long	2172777379
	.long	1063623822
	.long	3188733172
	.long	1061313738
	.long	258986845
	.long	1062033866
	.long	3921389429
	.long	3218472379
	.long	3647185164
	.long	3161383618
	.long	0
	.long	1072693248
	.long	2357811071
	.long	1069387281
	.long	1815523692
	.long	1009992944
	.long	0
	.long	1070596096
	.long	3138003653
	.long	1068983043
	.long	1253835798
	.long	1066949588
	.long	2444914170
	.long	1067019349
	.long	1113961907
	.long	1064316367
	.long	1725681937
	.long	1063646503
	.long	2521427730
	.long	1061370759
	.long	2475375033
	.long	1061668474
	.long	3294480606
	.long	3218523851
	.long	1489607747
	.long	1001838107
	.long	0
	.long	1072693248
	.long	163325898
	.long	1069457623
	.long	272534188
	.long	1011389069
	.long	0
	.long	1070596096
	.long	2709640876
	.long	1068993027
	.long	3241569416
	.long	1067011170
	.long	3499770285
	.long	1067032633
	.long	2428916378
	.long	1064360877
	.long	796870323
	.long	1063670282
	.long	2401385265
	.long	1061429237
	.long	435854613
	.long	3207953615
	.long	2667571783
	.long	3218575323
	.long	1614402419
	.long	1013901432
	.long	0
	.long	1072693248
	.long	1600848691
	.long	1069528089
	.long	3799012648
	.long	1011342217
	.long	0
	.long	1070596096
	.long	3940625627
	.long	1069003390
	.long	1694054617
	.long	1067073252
	.long	3525212904
	.long	1067046468
	.long	3400170401
	.long	1064406132
	.long	2025409820
	.long	1063695095
	.long	1861695260
	.long	1061489337
	.long	2071083481
	.long	1058280175
	.long	2040662960
	.long	3218626795
	.long	4270081753
	.long	1014949642
	.long	0
	.long	1072693248
	.long	2636871548
	.long	1069573102
	.long	3635909220
	.long	1011750791
	.long	0
	.long	1070596096
	.long	3074865967
	.long	1069014136
	.long	2827594877
	.long	1067135853
	.long	4018616278
	.long	1067060863
	.long	3744630852
	.long	1064452166
	.long	967547267
	.long	1063721046
	.long	1523268212
	.long	1061551058
	.long	308558327
	.long	1057912410
	.long	1413754136
	.long	3218678267
	.long	856972295
	.long	3161564710
	.long	0
	.long	1072693248
	.long	2829232582
	.long	1069608467
	.long	1037842028
	.long	3159896230
	.long	0
	.long	1070596096
	.long	1194701220
	.long	1069025268
	.long	1485063562
	.long	1067198995
	.long	3996610205
	.long	1067075828
	.long	2865183303
	.long	1064499014
	.long	2185691567
	.long	1063748117
	.long	435906278
	.long	1061614535
	.long	1590730193
	.long	1061922957
	.long	786845313
	.long	3218729739
	.long	627570489
	.long	3158416059
	.long	0
	.long	1072693248
	.long	850612498
	.long	1069643902
	.long	3957836518
	.long	3160223109
	.long	0
	.long	1070596096
	.long	4230382869
	.long	1069036788
	.long	2678394493
	.long	1067262698
	.long	42048419
	.long	1067091374
	.long	61549345
	.long	1064546712
	.long	3326101021
	.long	1063776440
	.long	377176219
	.long	1061679763
	.long	1587637306
	.long	3206122484
	.long	159936490
	.long	3218781211
	.long	3547642993
	.long	1013612270
	.long	0
	.long	1072693248
	.long	2707383656
	.long	1069679408
	.long	554608849
	.long	1013037903
	.long	0
	.long	1070596096
	.long	1495139838
	.long	1069048702
	.long	183291521
	.long	1067326985
	.long	3540584774
	.long	1067107509
	.long	3317896890
	.long	1064595295
	.long	3844035522
	.long	1063805952
	.long	3695032775
	.long	1061746922
	.long	3749017783
	.long	3207727795
	.long	3827994963
	.long	3218832682
	.long	941734744
	.long	1014805062
	.long	0
	.long	1072693248
	.long	1615317960
	.long	1069714989
	.long	1400849280
	.long	3154963122
	.long	0
	.long	1070596096
	.long	55251407
	.long	1069061012
	.long	2380564598
	.long	1067391876
	.long	2078095393
	.long	1067124247
	.long	3759475360
	.long	1064644803
	.long	800747683
	.long	1063836662
	.long	3361780990
	.long	1061816118
	.long	3647588455
	.long	1062249457
	.long	3201086139
	.long	3218884154
	.long	4185319304
	.long	3161709290
	.long	0
	.long	1072693248
	.long	3772940467
	.long	1069750646
	.long	3998164292
	.long	3160831254
	.long	0
	.long	1070596096
	.long	3265990537
	.long	1069073721
	.long	3279971426
	.long	1067453881
	.long	2447085124
	.long	1067141598
	.long	1649985592
	.long	1064695275
	.long	3334399538
	.long	1063868754
	.long	454991464
	.long	1061887337
	.long	2110827584
	.long	3204215126
	.long	2574177316
	.long	3218935626
	.long	2889755040
	.long	3159466096
	.long	0
	.long	1072693248
	.long	2595342993
	.long	1069786383
	.long	1222183174
	.long	1011927611
	.long	0
	.long	1070596096
	.long	2192927435
	.long	1069086835
	.long	1645422722
	.long	1067486966
	.long	648269084
	.long	1067159575
	.long	2319737250
	.long	1064746750
	.long	1121494052
	.long	1063902126
	.long	897314780
	.long	1061960786
	.long	2655964904
	.long	1060259181
	.long	1947268493
	.long	3218987098
	.long	1185916272
	.long	1013323109
	.long	0
	.long	1072693248
	.long	192708373
	.long	1069822202
	.long	1013565611
	.long	1013125469
	.long	0
	.long	1070596096
	.long	1393443609
	.long	1069100357
	.long	3282624502
	.long	1067520387
	.long	3423023493
	.long	1067178189
	.long	3539283610
	.long	1064799270
	.long	1358653364
	.long	1063936902
	.long	1383848540
	.long	1062036512
	.long	1203341716
	.long	3207439782
	.long	1320359670
	.long	3219038570
	.long	1908355032
	.long	1014660481
	.long	0
	.long	1072693248
	.long	3079104278
	.long	1069858104
	.long	2593237048
	.long	1012204550
	.long	0
	.long	1070596096
	.long	1749059456
	.long	1069114291
	.long	3316773722
	.long	1067554157
	.long	2555119580
	.long	1067197455
	.long	59306268
	.long	1064852879
	.long	1944001714
	.long	1063973040
	.long	3607967027
	.long	1062114688
	.long	1550745665
	.long	1061515569
	.long	693450846
	.long	3219090042
	.long	3218699017
	.long	3161853871
	.long	0
	.long	1072693248
	.long	701484222
	.long	1069894094
	.long	1189134214
	.long	3161032728
	.long	0
	.long	1070596096
	.long	478132658
	.long	1069128642
	.long	2223654597
	.long	1067588288
	.long	706118045
	.long	1067217386
	.long	1066336370
	.long	1064907619
	.long	3310393859
	.long	1064010705
	.long	1920668857
	.long	1062195336
	.long	1238710124
	.long	3206743753
	.long	33271012
	.long	3219134917
	.long	2269574412
	.long	1015853455
	.long	0
	.long	1072693248
	.long	4098310206
	.long	1069930172
	.long	678867859
	.long	3161222246
	.long	0
	.long	1070596096
	.long	4033975882
	.long	1069143413
	.long	1577500332
	.long	1067622792
	.long	2904242360
	.long	1067237995
	.long	383406633
	.long	1064963537
	.long	511745140
	.long	1064049810
	.long	3472404675
	.long	1062243075
	.long	3498625096
	.long	1061932631
	.long	4014783896
	.long	3219160652
	.long	3119156846
	.long	1013033947
	.long	0
	.long	1072693248
	.long	2953879462
	.long	1069966343
	.long	651183718
	.long	1012253676
	.long	0
	.long	1070596096
	.long	2348802841
	.long	1069158611
	.long	4095204310
	.long	1067657681
	.long	3730066556
	.long	1067259298
	.long	2541564470
	.long	1065020679
	.long	1012137447
	.long	1064090541
	.long	3007493277
	.long	1062286104
	.long	268143184
	.long	1058732349
	.long	3701329484
	.long	3219186388
	.long	2857479637
	.long	3162758049
	.long	0
	.long	1072693248
	.long	4257115870
	.long	1070002608
	.long	39728604
	.long	1013010275
	.long	0
	.long	1070596096
	.long	3207767847
	.long	1069174239
	.long	797077063
	.long	1067692970
	.long	287980713
	.long	1067281311
	.long	1596666478
	.long	1065079095
	.long	2775385
	.long	1064132847
	.long	3823518174
	.long	1062330592
	.long	3096591665
	.long	1060290467
	.long	3387875073
	.long	3219212124
	.long	3168927931
	.long	1015274949
	.long	0
	.long	1072693248
	.long	2240974125
	.long	1070038971
	.long	3786178918
	.long	3160482292
	.long	0
	.long	1070596096
	.long	2199065489
	.long	1069190303
	.long	1118792048
	.long	1067728670
	.long	875809397
	.long	1067304048
	.long	2927616918
	.long	1065138834
	.long	3494337743
	.long	1064176822
	.long	4219965650
	.long	1062376593
	.long	4246297104
	.long	1060574506
	.long	3074420661
	.long	3219237860
	.long	1873363667
	.long	3160515403
	.long	0
	.long	1072693248
	.long	4156511795
	.long	1070075433
	.long	1791783009
	.long	1013446896
	.long	0
	.long	1070596096
	.long	4204226438
	.long	1069206807
	.long	2895376501
	.long	1067764795
	.long	3600185398
	.long	1067327526
	.long	2087275384
	.long	1065199949
	.long	347703954
	.long	1064222543
	.long	301495246
	.long	1062424178
	.long	908235914
	.long	3206713553
	.long	2760966249
	.long	3219263596
	.long	1958126117
	.long	3163336555
	.long	0
	.long	1072693248
	.long	212483252
	.long	1070111999
	.long	3496985219
	.long	3161099600
	.long	0
	.long	1070596096
	.long	1939348811
	.long	1069223758
	.long	1001569645
	.long	1067801360
	.long	764537997
	.long	1067351764
	.long	2326690580
	.long	1065262493
	.long	60750552
	.long	1064270005
	.long	2745665091
	.long	1062473435
	.long	1583079917
	.long	1060807371
	.long	2447511838
	.long	3219289332
	.long	3841595606
	.long	1014371319
	.long	0
	.long	1072693248
	.long	2234510834
	.long	1070148669
	.long	1077829152
	.long	1008793063
	.long	0
	.long	1070596096
	.long	1741599179
	.long	1069241160
	.long	288887764
	.long	1067838378
	.long	438316229
	.long	1067376778
	.long	2074161178
	.long	1065326522
	.long	2132352506
	.long	1064311981
	.long	321000208
	.long	1062524429
	.long	2575912902
	.long	1061549566
	.long	2134057426
	.long	3219315068
	.long	1285458442
	.long	3162143033
	.long	0
	.long	1072693248
	.long	719981842
	.long	1070185448
	.long	3396798827
	.long	3160592662
	.long	0
	.long	1070596096
	.long	2111724540
	.long	1069259019
	.long	2754706540
	.long	1067875863
	.long	790037588
	.long	1067402587
	.long	4097286835
	.long	1065372654
	.long	2918356845
	.long	1064337614
	.long	980543769
	.long	1062577230
	.long	1876925371
	.long	3204685716
	.long	1820603015
	.long	3219340804
	.long	3236194699
	.long	1015708874
	.long	0
	.long	1072693248
	.long	3497432648
	.long	1070222337
	.long	3403980908
	.long	1011735702
	.long	0
	.long	1070596096
	.long	2322009458
	.long	1069277341
	.long	3303184211
	.long	1067913831
	.long	3079386980
	.long	1067429210
	.long	3158396776
	.long	1065406241
	.long	500122450
	.long	1064364204
	.long	3226667103
	.long	1062631956
	.long	1843492548
	.long	1062167069
	.long	1507148603
	.long	3219366540
	.long	1086374101
	.long	1011986833
	.long	0
	.long	1072693248
	.long	1371601499
	.long	1070259341
	.long	3050542448
	.long	1012183888
	.long	0
	.long	1070596096
	.long	2434996648
	.long	1069296132
	.long	2688049294
	.long	1067952297
	.long	1586964757
	.long	1067453518
	.long	2102156419
	.long	1065440660
	.long	2974407256
	.long	1064391847
	.long	249545820
	.long	1062688662
	.long	115553579
	.long	1061128535
	.long	1193694191
	.long	3219392276
	.long	1890859349
	.long	3162902630
	.long	0
	.long	1072693248
	.long	2488075121
	.long	1070296461
	.long	1856289345
	.long	3161251229
	.long	0
	.long	1070596096
	.long	3322930631
	.long	1069315398
	.long	3277607933
	.long	1067991276
	.long	130253043
	.long	1067467675
	.long	1057196725
	.long	1065475943
	.long	3278604818
	.long	1064420531
	.long	4197429763
	.long	1062747460
	.long	1630708049
	.long	1062399550
	.long	880239780
	.long	3219418012
	.long	4135548218
	.long	1015130368
	.long	0
	.long	1072693248
	.long	2273547475
	.long	1070333701
	.long	4162052844
	.long	1011378450
	.long	0
	.long	1070596096
	.long	2392983423
	.long	1069335146
	.long	526970473
	.long	1068030785
	.long	628322840
	.long	1067482270
	.long	924033682
	.long	1065512123
	.long	854746912
	.long	1064450346
	.long	4004963071
	.long	1062808438
	.long	1468070334
	.long	1060747995
	.long	566785368
	.long	3219443748
	.long	2301849815
	.long	3161093726
	.long	0
	.long	1072693248
	.long	620845047
	.long	1070371064
	.long	314567191
	.long	1013284227
	.long	0
	.long	1070596096
	.long	2198158867
	.long	1069355382
	.long	222370857
	.long	1068070839
	.long	4028822361
	.long	1067497314
	.long	3610397327
	.long	1065549234
	.long	171354847
	.long	1064481283
	.long	693369877
	.long	1062871714
	.long	164532117
	.long	1061632559
	.long	253330956
	.long	3219469484
	.long	991505830
	.long	3163481136
	.long	0
	.long	1072693248
	.long	1894214069
	.long	1070408552
	.long	2182661802
	.long	1012022503
	.long	0
	.long	1070596096
	.long	1869133310
	.long	1069376113
	.long	1368144043
	.long	1068111455
	.long	242783364
	.long	1067512821
	.long	3692616565
	.long	1065587313
	.long	2890683560
	.long	1064513419
	.long	2407947124
	.long	1062937380
	.long	3803218812
	.long	1060561671
	.long	4234843841
	.long	3219495219
	.long	1479868884
	.long	1014082158
	.long	0
	.long	1072693248
	.long	2049842017
	.long	1070446169
	.long	2491067638
	.long	3161241370
	.long	0
	.long	1070596096
	.long	1431833764
	.long	1069397346
	.long	551048935
	.long	1068152651
	.long	2575917112
	.long	1067528800
	.long	1559777140
	.long	1065626397
	.long	743322322
	.long	1064546764
	.long	1974991899
	.long	1063005565
	.long	3459002328
	.long	1062460308
	.long	3921389429
	.long	3219520955
	.long	3647185164
	.long	3162432194
	.long	0
	.long	1072693248
	.long	1526324265
	.long	1070483918
	.long	4056333712
	.long	3157795638
	.long	0
	.long	1070596096
	.long	1830915900
	.long	1069419088
	.long	602185706
	.long	1068194444
	.long	555210277
	.long	1067545266
	.long	3690136761
	.long	1065666523
	.long	4115917104
	.long	1064581405
	.long	589230666
	.long	1063076369
	.long	3448584873
	.long	1062382364
	.long	3607935018
	.long	3219546691
	.long	4202814986
	.long	1015564293
	.long	0
	.long	1072693248
	.long	955408364
	.long	1070521802
	.long	2659929592
	.long	3160779467
	.long	0
	.long	1070596096
	.long	659174847
	.long	1069441347
	.long	2081794994
	.long	1068236852
	.long	2956616392
	.long	1067562230
	.long	1461867065
	.long	1065707733
	.long	879427925
	.long	1064617384
	.long	209598570
	.long	1063149928
	.long	59348277
	.long	1061529630
	.long	3294480606
	.long	3219572427
	.long	1489607747
	.long	1002886683
	.long	0
	.long	1072693248
	.long	1167858038
	.long	1070559824
	.long	2180017056
	.long	1010873776
	.long	0
	.long	1070596096
	.long	772792549
	.long	1069464130
	.long	3356872770
	.long	1068279894
	.long	3524844777
	.long	1067579707
	.long	1521756467
	.long	1065750067
	.long	3931969752
	.long	1064654734
	.long	2908114566
	.long	1063226363
	.long	3520733649
	.long	1061374592
	.long	2981026194
	.long	3219598163
	.long	924239062
	.long	3163047211
	.long	0
	.long	1072693248
	.long	3747221382
	.long	1070597041
	.long	1912303979
	.long	1014066411
	.long	0
	.long	1070596096
	.long	1727686788
	.long	1069487445
	.long	681802874
	.long	1068323590
	.long	2531789465
	.long	1067597711
	.long	331202638
	.long	1065793569
	.long	2627596742
	.long	1064693515
	.long	566979200
	.long	1063280948
	.long	2423252998
	.long	1061741710
	.long	2667571783
	.long	3219623899
	.long	1614402419
	.long	1014950008
	.long	0
	.long	1072693248
	.long	4148643959
	.long	1070616195
	.long	2887822437
	.long	1014214935
	.long	0
	.long	1070596096
	.long	4101770849
	.long	1069511300
	.long	1167052501
	.long	1068367958
	.long	2569727718
	.long	1067616256
	.long	1681502493
	.long	1065838283
	.long	3685967929
	.long	1064733780
	.long	2973319878
	.long	1063322263
	.long	474740638
	.long	1062549181
	.long	2354117371
	.long	3219649635
	.long	3512651629
	.long	3161564344
	.long	0
	.long	1072693248
	.long	373017919
	.long	1070635424
	.long	3647199022
	.long	1014660881
	.long	0
	.long	1070596096
	.long	933359764
	.long	1069535705
	.long	1391441088
	.long	1068413019
	.long	53106651
	.long	1067635358
	.long	4157835043
	.long	1065884256
	.long	4055868043
	.long	1064775611
	.long	1529330433
	.long	1063365236
	.long	326078913
	.long	1062325273
	.long	2040662960
	.long	3219675371
	.long	4270081753
	.long	1015998218
	.long	0
	.long	1072693248
	.long	4002124759
	.long	1070654727
	.long	3453343154
	.long	1013423881
	.long	0
	.long	1071644672
	.long	774562305
	.long	3217018021
	.long	377617675
	.long	1068458794
	.long	2495478050
	.long	1067655031
	.long	1508050213
	.long	1065931538
	.long	1921336565
	.long	1064819067
	.long	975923615
	.long	1063409946
	.long	2275792800
	.long	1060505011
	.long	1727208548
	.long	3219701107
	.long	2531251621
	.long	1013613001
	.long	0
	.long	1072693248
	.long	963359913
	.long	1070674109
	.long	2464811476
	.long	3160767877
	.long	0
	.long	1071644672
	.long	1498613974
	.long	3216992493
	.long	4138103782
	.long	1068502123
	.long	2840947366
	.long	1067675293
	.long	209537977
	.long	1065980178
	.long	3590357525
	.long	1064864171
	.long	73959555
	.long	1063456484
	.long	1512178095
	.long	1062390487
	.long	1413754136
	.long	3219726843
	.long	856972295
	.long	3162613286
	.long	0
	.long	0
	.long	0
	.long	3220176896
	.long	1994327296
	.long	1006932657
	.long	0
	.long	1073741824
	.long	0
	.long	0
	.long	266
	.long	3221225472
	.long	1431655956
	.long	1074091349
	.long	2828070733
	.long	3221924522
	.long	268536947
	.long	1074860305
	.long	4220652043
	.long	3222647048
	.long	3597047153
	.long	1075552901
	.long	1100299725
	.long	3219752579
	.long	874467977
	.long	1015419713
	.long	0
	.long	0
	.long	2668510354
	.long	3220126045
	.long	1701998120
	.long	1012206572
	.long	0
	.long	1073741824
	.long	1404962112
	.long	3215492924
	.long	3327263454
	.long	3221125605
	.long	1987673213
	.long	1074008829
	.long	231222583
	.long	3221793459
	.long	1756671652
	.long	1074731884
	.long	3921518330
	.long	3222500748
	.long	2691193109
	.long	1075341426
	.long	786845313
	.long	3219778315
	.long	627570489
	.long	3159464635
	.long	0
	.long	0
	.long	2949191430
	.long	3220076398
	.long	3989126963
	.long	3163096676
	.long	0
	.long	1073741824
	.long	2909112164
	.long	3216503445
	.long	3682557528
	.long	3221031585
	.long	4184421718
	.long	1073932454
	.long	380803270
	.long	3221673930
	.long	3399744364
	.long	1074553823
	.long	3842864962
	.long	3222371390
	.long	748075446
	.long	1075157395
	.long	473390901
	.long	3219804051
	.long	4252586071
	.long	3163191791
	.long	0
	.long	0
	.long	2308259791
	.long	3220027898
	.long	4111771200
	.long	3156765800
	.long	0
	.long	1073741824
	.long	411343470
	.long	3217132228
	.long	3760686807
	.long	3220942979
	.long	4171367426
	.long	1073861688
	.long	1878500541
	.long	3221564765
	.long	108767834
	.long	1074393698
	.long	2824899565
	.long	3222239549
	.long	3321236009
	.long	1074997439
	.long	159936490
	.long	3219829787
	.long	3547642993
	.long	1014660846
	.long	0
	.long	0
	.long	3342355758
	.long	3219980491
	.long	1666924640
	.long	3162031356
	.long	0
	.long	1073741824
	.long	2271288998
	.long	3217480250
	.long	4111284847
	.long	3220859392
	.long	2796272966
	.long	1073796048
	.long	1730314448
	.long	3221464928
	.long	1813001552
	.long	1074249474
	.long	603251437
	.long	3222036157
	.long	3079042443
	.long	1074856525
	.long	4141449374
	.long	3219855522
	.long	1579411055
	.long	3161853506
	.long	0
	.long	0
	.long	166164867
	.long	3219934128
	.long	2304492376
	.long	1012090550
	.long	0
	.long	1073741824
	.long	2798667120
	.long	3217812323
	.long	4203596829
	.long	3220780463
	.long	1473339147
	.long	1073728374
	.long	4243021652
	.long	3221373498
	.long	1638747017
	.long	1074119375
	.long	1512673116
	.long	3221855335
	.long	2338684822
	.long	1074678259
	.long	3827994963
	.long	3219881258
	.long	941734744
	.long	1015853638
	.long	0
	.long	0
	.long	3247184626
	.long	3219888759
	.long	1245319390
	.long	3161909785
	.long	0
	.long	1073741824
	.long	2381906911
	.long	3218104536
	.long	1010924947
	.long	3220705862
	.long	2481381005
	.long	1073615072
	.long	1356661294
	.long	3221289659
	.long	1527070583
	.long	1074001844
	.long	3061693405
	.long	3221694303
	.long	1993154673
	.long	1074463435
	.long	3514540551
	.long	3219906994
	.long	2102765473
	.long	1013034678
	.long	0
	.long	0
	.long	4189730214
	.long	3219844341
	.long	872357632
	.long	3158652168
	.long	0
	.long	1073741824
	.long	760634709
	.long	3218255915
	.long	2067827628
	.long	3220635284
	.long	1386756408
	.long	1073509659
	.long	404552634
	.long	3221199890
	.long	1143062644
	.long	1073895515
	.long	555342338
	.long	3221550657
	.long	3470199450
	.long	1074271418
	.long	3201086139
	.long	3219932730
	.long	4185319304
	.long	3162757866
	.long	0
	.long	0
	.long	719595600
	.long	3219800832
	.long	114970824
	.long	1014660100
	.long	0
	.long	1073741824
	.long	3811788216
	.long	3218400550
	.long	1264738948
	.long	3220568452
	.long	1958933439
	.long	1073411493
	.long	4246310163
	.long	3221058356
	.long	1367842247
	.long	1073799186
	.long	4112596176
	.long	3221422310
	.long	2100954469
	.long	1074105254
	.long	2887631728
	.long	3219958466
	.long	1841088264
	.long	1015275132
	.long	0
	.long	0
	.long	258163604
	.long	3219758190
	.long	3299889334
	.long	3162612005
	.long	0
	.long	1073741824
	.long	2865277319
	.long	3218538798
	.long	3110066496
	.long	3220505109
	.long	723074889
	.long	1073319994
	.long	1618819009
	.long	3220928086
	.long	1176243935
	.long	1073681775
	.long	1223849000
	.long	3221307454
	.long	2369288310
	.long	1073957581
	.long	2574177316
	.long	3219984202
	.long	2889755040
	.long	3160514672
	.long	0
	.long	0
	.long	2079874795
	.long	3219716377
	.long	1885564844
	.long	1014470429
	.long	0
	.long	1073741824
	.long	1656469901
	.long	3218670989
	.long	823761676
	.long	3220445021
	.long	2491445454
	.long	1073234634
	.long	1584341555
	.long	3220808039
	.long	1976688438
	.long	1073523018
	.long	3918313362
	.long	3221183550
	.long	3178717701
	.long	1073829019
	.long	2260722904
	.long	3220009938
	.long	3285965784
	.long	3163336372
	.long	0
	.long	0
	.long	997287281
	.long	3219675358
	.long	3612251184
	.long	3163317056
	.long	0
	.long	1073741824
	.long	3037674984
	.long	3218797431
	.long	2624215131
	.long	3220387969
	.long	464779768
	.long	1073154936
	.long	3700763630
	.long	3220697284
	.long	1200926910
	.long	1073378628
	.long	2934576448
	.long	3220998744
	.long	3287041894
	.long	1073681673
	.long	1947268493
	.long	3220035674
	.long	1185916272
	.long	1014371685
	.long	0
	.long	0
	.long	3978924239
	.long	3219635097
	.long	680771999
	.long	1015720881
	.long	0
	.long	1073741824
	.long	3528982175
	.long	3218918413
	.long	3018376188
	.long	3220333754
	.long	2093086652
	.long	1073080463
	.long	451029217
	.long	3220594987
	.long	1097121537
	.long	1073247142
	.long	2277789243
	.long	3220832617
	.long	149138386
	.long	1073478822
	.long	1633814081
	.long	3220061410
	.long	3941137777
	.long	3162142667
	.long	0
	.long	0
	.long	3860971117
	.long	3219595563
	.long	2891614774
	.long	3163214156
	.long	0
	.long	1073741824
	.long	80822128
	.long	3219034205
	.long	2077635078
	.long	3220282191
	.long	1552870456
	.long	1073010820
	.long	3607700650
	.long	3220500394
	.long	249538439
	.long	1073127265
	.long	2233833083
	.long	3220683068
	.long	2436456793
	.long	1073293556
	.long	1320359670
	.long	3220087146
	.long	1908355032
	.long	1015709057
	.long	0
	.long	0
	.long	343219638
	.long	3219556725
	.long	2238262736
	.long	3161731808
	.long	0
	.long	1072693248
	.long	3768132253
	.long	1071627934
	.long	3895546569
	.long	3220233108
	.long	1613056473
	.long	1072945645
	.long	3409294908
	.long	3220412831
	.long	3138966349
	.long	1073017844
	.long	3799067288
	.long	3220548255
	.long	359470498
	.long	1073133087
	.long	1006905258
	.long	3220112882
	.long	3348558652
	.long	1011988294
	.long	0
	.long	0
	.long	3841229102
	.long	3219518551
	.long	1782820117
	.long	3163516640
	.long	0
	.long	1072693248
	.long	866896685
	.long	1071521786
	.long	2743435231
	.long	3220186348
	.long	3017984361
	.long	1072884608
	.long	1491919678
	.long	3220331688
	.long	1507535811
	.long	1072917857
	.long	1966124974
	.long	3220426562
	.long	381628641
	.long	1072992589
	.long	693450846
	.long	3220138618
	.long	3218699017
	.long	3162902447
	.long	0
	.long	0
	.long	716700048
	.long	3219481016
	.long	1043841701
	.long	3163006458
	.long	0
	.long	1072693248
	.long	1097907398
	.long	1071420120
	.long	3912524876
	.long	3220106631
	.long	4107784306
	.long	1072827408
	.long	937056065
	.long	3220256414
	.long	4128945134
	.long	1072826390
	.long	1032262394
	.long	3220316565
	.long	1825923585
	.long	1072864423
	.long	379996435
	.long	3220164354
	.long	2807708551
	.long	1015130551
	.long	0
	.long	0
	.long	142288477
	.long	3219444091
	.long	1256348500
	.long	1014532780
	.long	0
	.long	1072693248
	.long	925074332
	.long	1071322730
	.long	2134728327
	.long	3220021541
	.long	1886408555
	.long	1072773770
	.long	101893267
	.long	3220186512
	.long	2020085224
	.long	1072742631
	.long	2955803861
	.long	3220217010
	.long	343310816
	.long	1072748161
	.long	33271012
	.long	3220183493
	.long	2269574412
	.long	1016902031
	.long	0
	.long	0
	.long	3684492763
	.long	3219407750
	.long	2905890526
	.long	1015709561
	.long	0
	.long	1072693248
	.long	585640690
	.long	1071229422
	.long	1068301595
	.long	3219940277
	.long	2953900489
	.long	1072723440
	.long	950237889
	.long	3220066167
	.long	109735425
	.long	1072638455
	.long	792610979
	.long	3220076689
	.long	2904190292
	.long	1072594405
	.long	4171511102
	.long	3220196360
	.long	987810900
	.long	1016106563
	.long	0
	.long	0
	.long	681498355
	.long	3219371971
	.long	3542625299
	.long	1015827699
	.long	0
	.long	1072693248
	.long	168222332
	.long	1071140014
	.long	776736583
	.long	3219862604
	.long	2405401100
	.long	1072659128
	.long	430546584
	.long	3219945234
	.long	2974083786
	.long	1072497552
	.long	849442068
	.long	3219912970
	.long	2727451532
	.long	1072409680
	.long	4014783896
	.long	3220209228
	.long	3119156846
	.long	1014082523
	.long	0
	.long	0
	.long	2298773501
	.long	3219336728
	.long	4027807132
	.long	1015090440
	.long	0
	.long	1072693248
	.long	312748411
	.long	1071054335
	.long	2723913091
	.long	3219788303
	.long	3207063211
	.long	1072570353
	.long	1845780128
	.long	3219832588
	.long	752659336
	.long	1072368142
	.long	1097207453
	.long	3219764237
	.long	407427403
	.long	1072246732
	.long	3858056690
	.long	3220222096
	.long	2007897202
	.long	3162431829
	.long	0
	.long	0
	.long	2591355529
	.long	3219302000
	.long	2351326818
	.long	3161908153
	.long	0
	.long	1072693248
	.long	3206845590
	.long	1070972224
	.long	4023349050
	.long	3219717171
	.long	3838822434
	.long	1072486918
	.long	3122427157
	.long	3219727557
	.long	2261703627
	.long	1072249178
	.long	1764870100
	.long	3219628955
	.long	3201802997
	.long	1072093301
	.long	3701329484
	.long	3220234964
	.long	2857479637
	.long	3163806625
	.long	0
	.long	0
	.long	4010047525
	.long	3219267765
	.long	3339891512
	.long	3160072694
	.long	0
	.long	1072693248
	.long	966450668
	.long	1070893532
	.long	803989702
	.long	3219649019
	.long	1934453966
	.long	1072408465
	.long	1455077850
	.long	3219629530
	.long	2709264343
	.long	1072139721
	.long	1918064558
	.long	3219505762
	.long	1476733535
	.long	1071960574
	.long	3544602278
	.long	3220247832
	.long	4139243149
	.long	3164602093
	.long	0
	.long	0
	.long	4270735453
	.long	3219234003
	.long	3565361953
	.long	1015342529
	.long	0
	.long	1072693248
	.long	2915760230
	.long	1070818115
	.long	638580392
	.long	3219583668
	.long	2252948193
	.long	1072334663
	.long	1194731830
	.long	3219537949
	.long	1353002942
	.long	1072038927
	.long	422364789
	.long	3219393446
	.long	3669634446
	.long	1071842023
	.long	3387875073
	.long	3220260700
	.long	3168927931
	.long	1016323525
	.long	0
	.long	0
	.long	419968236
	.long	3219200695
	.long	131828542
	.long	1015961466
	.long	0
	.long	1072693248
	.long	2960267235
	.long	1070745841
	.long	1037049094
	.long	3219520953
	.long	1447436626
	.long	1072265209
	.long	3197232138
	.long	3219452306
	.long	4004685084
	.long	1071946034
	.long	1412816421
	.long	3219290928
	.long	3184738590
	.long	1071740610
	.long	3231147867
	.long	3220273568
	.long	3253690381
	.long	1014950373
	.long	0
	.long	0
	.long	1505561933
	.long	3219167820
	.long	2813885200
	.long	1011162516
	.long	0
	.long	1072693248
	.long	2997217365
	.long	1070676584
	.long	2681788575
	.long	3219460719
	.long	3989126937
	.long	1072199822
	.long	1049115510
	.long	3219372139
	.long	511642058
	.long	1071860360
	.long	4034728051
	.long	3219197246
	.long	1788863965
	.long	1071639550
	.long	3074420661
	.long	3220286436
	.long	1873363667
	.long	3161563979
	.long	0
	.long	0
	.long	2606398711
	.long	3219135361
	.long	2533072598
	.long	3162554073
	.long	0
	.long	1072693248
	.long	2151361100
	.long	1070610226
	.long	1532348096
	.long	3219402822
	.long	408604530
	.long	1072138246
	.long	1039538740
	.long	3219297023
	.long	1234598850
	.long	1071781284
	.long	692507578
	.long	3219094767
	.long	4090315877
	.long	1071451983
	.long	2917693455
	.long	3220299304
	.long	676362605
	.long	3163589663
	.long	0
	.long	0
	.long	3649427950
	.long	3219078282
	.long	4113825640
	.long	3158662193
	.long	0
	.long	1072693248
	.long	2424145432
	.long	1070497215
	.long	2715404890
	.long	3219347125
	.long	2238283673
	.long	1072080240
	.long	1186867211
	.long	3219226571
	.long	660318792
	.long	1071708248
	.long	2091427915
	.long	3218937780
	.long	2591062635
	.long	1071291190
	.long	2760966249
	.long	3220312172
	.long	1958126117
	.long	3164385131
	.long	0
	.long	0
	.long	220675706
	.long	3219014926
	.long	3158451653
	.long	3161723787
	.long	0
	.long	1072693248
	.long	1565030890
	.long	1070375439
	.long	335263687
	.long	3219293502
	.long	419816567
	.long	1072025586
	.long	2098048102
	.long	3219160428
	.long	257229335
	.long	1071636821
	.long	3464457189
	.long	3218793840
	.long	2745923191
	.long	1071158056
	.long	2604239044
	.long	3220325040
	.long	1055077667
	.long	1016540488
	.long	0
	.long	0
	.long	2191706112
	.long	3218952300
	.long	333441366
	.long	3161331348
	.long	0
	.long	1072693248
	.long	3435408456
	.long	1070258830
	.long	487319719
	.long	3219241832
	.long	1713711059
	.long	1071974079
	.long	130753286
	.long	3219068219
	.long	909725130
	.long	1071511974
	.long	2762442367
	.long	3218661715
	.long	932670499
	.long	1071027067
	.long	2447511838
	.long	3220337908
	.long	3841595606
	.long	1015419895
	.long	0
	.long	0
	.long	408840118
	.long	3218890375
	.long	4048072104
	.long	3159354273
	.long	0
	.long	1072693248
	.long	2794370512
	.long	1070147203
	.long	1550993351
	.long	3219192003
	.long	1305093678
	.long	1071925532
	.long	1748211158
	.long	3218951271
	.long	3244212114
	.long	1071396455
	.long	2145666003
	.long	3218540299
	.long	1098451822
	.long	1070891096
	.long	2290784632
	.long	3220350776
	.long	2660353234
	.long	3159463173
	.long	0
	.long	0
	.long	3368427099
	.long	3218829119
	.long	2187864186
	.long	1013587140
	.long	0
	.long	1072693248
	.long	1708021028
	.long	1070040382
	.long	459972195
	.long	3219143910
	.long	3210514347
	.long	1071879770
	.long	955408873
	.long	3218841145
	.long	1911386085
	.long	1071289511
	.long	3563284038
	.long	3218428599
	.long	547420745
	.long	1070790031
	.long	2134057426
	.long	3220363644
	.long	1285458442
	.long	3163191609
	.long	0
	.long	0
	.long	2583490354
	.long	3218768505
	.long	27148750
	.long	3161491985
	.long	0
	.long	1072693248
	.long	2550940470
	.long	1069938201
	.long	2616040293
	.long	3219066585
	.long	2529278955
	.long	1071836633
	.long	2244405849
	.long	3218737335
	.long	2064814127
	.long	1071190460
	.long	1527853251
	.long	3218325722
	.long	354247476
	.long	1070708112
	.long	1977330220
	.long	3220376512
	.long	4071976381
	.long	3164168168
	.long	0
	.long	0
	.long	2390846448
	.long	3218708504
	.long	3510503948
	.long	3161709355
	.long	0
	.long	1072693248
	.long	1208409702
	.long	1069840505
	.long	1006199412
	.long	3218976755
	.long	2504246873
	.long	1071795971
	.long	1040653613
	.long	3218639377
	.long	1713268217
	.long	1071098686
	.long	462276301
	.long	3218230862
	.long	4239163357
	.long	1070603989
	.long	1820603015
	.long	3220389380
	.long	3236194699
	.long	1016757450
	.long	0
	.long	0
	.long	2528366626
	.long	3218649089
	.long	3808712102
	.long	1013396510
	.long	0
	.long	1072693248
	.long	1938261590
	.long	1069747146
	.long	4282399603
	.long	3218889831
	.long	1922568281
	.long	1071757646
	.long	2428166646
	.long	3218546841
	.long	717464824
	.long	1071013623
	.long	2205664075
	.long	3218143293
	.long	1526076492
	.long	1070438229
	.long	1663875809
	.long	3220402248
	.long	3908862373
	.long	1015853820
	.long	0
	.long	0
	.long	2200768388
	.long	3218590234
	.long	302584884
	.long	1014949287
	.long	0
	.long	1072693248
	.long	2338248308
	.long	1069657986
	.long	2617403810
	.long	3218805648
	.long	60078074
	.long	1071721530
	.long	2753634668
	.long	3218459332
	.long	626020946
	.long	1070934758
	.long	25846122
	.long	3218044977
	.long	1123428672
	.long	1070307499
	.long	1507148603
	.long	3220415116
	.long	1086374101
	.long	1013035409
	.long	0
	.long	0
	.long	4158096969
	.long	3218531913
	.long	3224512900
	.long	3160902954
	.long	0
	.long	1072693248
	.long	4240313622
	.long	1069572894
	.long	3745046477
	.long	3218724046
	.long	1955120038
	.long	1071687503
	.long	2661530988
	.long	3218376484
	.long	2615081511
	.long	1070861626
	.long	3172686029
	.long	3217895196
	.long	3468537113
	.long	1070173373
	.long	1350421397
	.long	3220427984
	.long	1218191675
	.long	3162757684
	.long	0
	.long	0
	.long	1606100959
	.long	3218474103
	.long	3385789004
	.long	3161114183
	.long	0
	.long	1072693248
	.long	2255302008
	.long	1069435977
	.long	54085333
	.long	3218644877
	.long	2504974906
	.long	1071655456
	.long	4169947836
	.long	3218297958
	.long	2891827398
	.long	1070793803
	.long	1030744944
	.long	3217756428
	.long	2378245768
	.long	1070036506
	.long	1193694191
	.long	3220440852
	.long	1890859349
	.long	3163951206
	.long	0
	.long	0
	.long	3782220567
	.long	3218416778
	.long	2488119274
	.long	3162237132
	.long	0
	.long	1072693248
	.long	91372748
	.long	1069281344
	.long	318776714
	.long	3218567997
	.long	1791786485
	.long	1071605902
	.long	245863368
	.long	3218223442
	.long	3306699316
	.long	1070730899
	.long	492155895
	.long	3217627699
	.long	504590813
	.long	1069961586
	.long	1036966986
	.long	3220453720
	.long	1122344434
	.long	1016974413
	.long	0
	.long	0
	.long	1412148370
	.long	3218359917
	.long	2689483850
	.long	1013993587
	.long	0
	.long	1072693248
	.long	1375255072
	.long	1069134151
	.long	1493266594
	.long	3218493272
	.long	487968719
	.long	1071549129
	.long	4010395236
	.long	3218152642
	.long	2875570533
	.long	1070672565
	.long	2596803943
	.long	3217508124
	.long	2843491945
	.long	1069843051
	.long	880239780
	.long	3220466588
	.long	4135548218
	.long	1016178944
	.long	0
	.long	0
	.long	895247324
	.long	3218303496
	.long	3064008070
	.long	1013805381
	.long	0
	.long	1072693248
	.long	729983844
	.long	1068994194
	.long	3631919317
	.long	3218420574
	.long	2031343772
	.long	1071495745
	.long	1505593195
	.long	3218085291
	.long	519428997
	.long	1070618477
	.long	3710689661
	.long	3217396902
	.long	3156076115
	.long	1069741343
	.long	723512574
	.long	3220479456
	.long	2825204233
	.long	1014372050
	.long	0
	.long	0
	.long	2664093517
	.long	3218247493
	.long	3574482624
	.long	3157273734
	.long	0
	.long	1072693248
	.long	2465054196
	.long	1068861279
	.long	3495559184
	.long	3218349782
	.long	2180799467
	.long	1071445588
	.long	3760731789
	.long	3217962527
	.long	3108073739
	.long	1070540584
	.long	1955989624
	.long	3217293302
	.long	166797605
	.long	1069654305
	.long	566785368
	.long	3220492324
	.long	2301849815
	.long	3162142302
	.long	0
	.long	0
	.long	1026922473
	.long	3218191888
	.long	4275547180
	.long	3161242805
	.long	0
	.long	1072693248
	.long	865263972
	.long	1068735226
	.long	1189261043
	.long	3218280780
	.long	602895285
	.long	1071398507
	.long	457371263
	.long	3217840141
	.long	1846858242
	.long	1070447677
	.long	3060498116
	.long	3217196657
	.long	4267893365
	.long	1069596604
	.long	410058162
	.long	3220505192
	.long	4004709614
	.long	3163734243
	.long	0
	.long	0
	.long	1197423420
	.long	3218136659
	.long	176395860
	.long	3160912588
	.long	0
	.long	1072693248
	.long	3614211808
	.long	1068615863
	.long	2638984897
	.long	3218213456
	.long	1695604536
	.long	1071354361
	.long	2372043566
	.long	3217723241
	.long	136909605
	.long	1070361652
	.long	3640039447
	.long	3217106358
	.long	4088615056
	.long	1069449448
	.long	253330956
	.long	3220518060
	.long	991505830
	.long	3164529712
	.long	0
	.long	0
	.long	1971725998
	.long	3218081786
	.long	2628057466
	.long	3160778219
	.long	0
	.long	1072693248
	.long	3850616508
	.long	1068503032
	.long	715485429
	.long	3218147706
	.long	2831870887
	.long	1071313021
	.long	276112813
	.long	3217611421
	.long	4071473627
	.long	1070282053
	.long	3678820506
	.long	3217012537
	.long	927889203
	.long	1069392578
	.long	96603751
	.long	3220530928
	.long	2021697954
	.long	1016395907
	.long	0
	.long	0
	.long	4068253482
	.long	3217974755
	.long	3098800802
	.long	3161416360
	.long	0
	.long	1072693248
	.long	3555466384
	.long	1068294223
	.long	3714292293
	.long	3218083427
	.long	58004546
	.long	1071274368
	.long	2889916307
	.long	3217504297
	.long	2725223785
	.long	1070208483
	.long	1960889150
	.long	3216854085
	.long	2172838405
	.long	1069259864
	.long	4234843841
	.long	3220543795
	.long	1479868884
	.long	1015130734
	.long	0
	.long	0
	.long	3947580658
	.long	3217866316
	.long	4153464200
	.long	3160712468
	.long	0
	.long	1072693248
	.long	1306970432
	.long	1068093810
	.long	3320658510
	.long	3217961305
	.long	2638377394
	.long	1071238289
	.long	1218095092
	.long	3217401513
	.long	506153465
	.long	1070140566
	.long	2096772766
	.long	3216705262
	.long	2058303373
	.long	1069148484
	.long	4078116635
	.long	3220556663
	.long	39665264
	.long	3161092265
	.long	0
	.long	0
	.long	4023957346
	.long	3217758473
	.long	2652051644
	.long	3159900932
	.long	0
	.long	1072693248
	.long	1548193280
	.long	1067905620
	.long	2849460080
	.long	3217838064
	.long	586099284
	.long	1071204684
	.long	2426038050
	.long	3217302731
	.long	1209882939
	.long	1070077962
	.long	204929670
	.long	3216565209
	.long	2951398611
	.long	1069077118
	.long	3921389429
	.long	3220569531
	.long	3647185164
	.long	3163480770
	.long	0
	.long	0
	.long	3763679576
	.long	3217651189
	.long	448385104
	.long	3157192355
	.long	0
	.long	1072693248
	.long	64931152
	.long	1067729411
	.long	3593250320
	.long	3217717209
	.long	1127364104
	.long	1071173457
	.long	21320967
	.long	3217207636
	.long	4036384111
	.long	1070020367
	.long	3394693850
	.long	3216433130
	.long	4171153724
	.long	1068939101
	.long	3764662223
	.long	3220582399
	.long	3105356094
	.long	3164312749
	.long	0
	.long	0
	.long	1425758568
	.long	3217544428
	.long	1224254624
	.long	3159340630
	.long	0
	.long	1072693248
	.long	2086713664
	.long	1067564956
	.long	1688303979
	.long	3217598569
	.long	938619730
	.long	1071144522
	.long	526120622
	.long	3217115928
	.long	3819295674
	.long	1069967496
	.long	2059466203
	.long	3216308301
	.long	4034680199
	.long	1068903334
	.long	3607935018
	.long	3220595267
	.long	4202814986
	.long	1016612869
	.long	0
	.long	0
	.long	2429422902
	.long	3217438153
	.long	3297520416
	.long	1012522944
	.long	0
	.long	1072693248
	.long	528565440
	.long	1067373727
	.long	2395999282
	.long	3217481977
	.long	1512953454
	.long	1071117799
	.long	2702142222
	.long	3217023483
	.long	3515899579
	.long	1069919100
	.long	2767905791
	.long	3216190039
	.long	3127036520
	.long	1068808800
	.long	3451207812
	.long	3220608135
	.long	1547135652
	.long	1015564659
	.long	0
	.long	0
	.long	1959743063
	.long	3217332330
	.long	607720656
	.long	3161041674
	.long	0
	.long	1072693248
	.long	225834432
	.long	1067090614
	.long	732340398
	.long	3217367274
	.long	3221152670
	.long	1071093215
	.long	2810588359
	.long	3216851955
	.long	612200454
	.long	1069874948
	.long	645427044
	.long	3216077719
	.long	1966523063
	.long	1068694214
	.long	3294480606
	.long	3220621003
	.long	1489607747
	.long	1003935259
	.long	0
	.long	0
	.long	1940260416
	.long	3217226924
	.long	4144841612
	.long	3159543701
	.long	0
	.long	1072693248
	.long	1135818128
	.long	1066829850
	.long	867512158
	.long	3217254304
	.long	3821819258
	.long	1071070704
	.long	2890864845
	.long	3216685596
	.long	918459421
	.long	1069834828
	.long	797471869
	.long	3215958923
	.long	4093823457
	.long	1068635569
	.long	3137753400
	.long	3220633871
	.long	3579918397
	.long	3163046845
	.long	0
	.long	0
	.long	2242453104
	.long	3217121901
	.long	1143544320
	.long	1011877960
	.long	0
	.long	1072693248
	.long	1016752608
	.long	1066591110
	.long	1270001991
	.long	3217142917
	.long	1410785275
	.long	1071050206
	.long	1103929722
	.long	3216523923
	.long	3268015214
	.long	1069798554
	.long	957665123
	.long	3215754628
	.long	1067716675
	.long	1068610012
	.long	2981026194
	.long	3220646739
	.long	924239062
	.long	3164095787
	.long	0
	.long	0
	.long	982366697
	.long	3217003288
	.long	1160619554
	.long	1010902188
	.long	0
	.long	1072693248
	.long	404678272
	.long	1066346403
	.long	1419607756
	.long	3217032967
	.long	1800406832
	.long	1071031665
	.long	3144361055
	.long	3216366473
	.long	1744148039
	.long	1069765960
	.long	3211598528
	.long	3215558937
	.long	405031618
	.long	1068609434
	.long	2824298989
	.long	3220659607
	.long	2088964722
	.long	1016829832
	.long	0
	.long	0
	.long	931121154
	.long	3216794574
	.long	1121583044
	.long	1012154237
	.long	0
	.long	1072693248
	.long	709357472
	.long	1065955296
	.long	29972165
	.long	3216817456
	.long	254360647
	.long	1071015033
	.long	3732687627
	.long	3216212805
	.long	3224370681
	.long	1069736900
	.long	202592280
	.long	3215370879
	.long	1420621447
	.long	1068491719
	.long	2667571783
	.long	3220672475
	.long	1614402419
	.long	1015998584
	.long	0
	.long	0
	.long	2354785698
	.long	3216586427
	.long	959264268
	.long	3159915811
	.long	0
	.long	1072693248
	.long	2828230112
	.long	1065606626
	.long	3693284264
	.long	3216602456
	.long	1996210616
	.long	1071000265
	.long	3053541878
	.long	3216062494
	.long	1133334567
	.long	1069711236
	.long	977677585
	.long	3215189563
	.long	347473669
	.long	1068436198
	.long	2510844577
	.long	3220685343
	.long	498468876
	.long	1013614463
	.long	0
	.long	0
	.long	1972895212
	.long	3216378783
	.long	3207896256
	.long	3158122047
	.long	0
	.long	1072693248
	.long	4107853760
	.long	1065246718
	.long	2229347510
	.long	3216389497
	.long	4172638464
	.long	1070987323
	.long	212469442
	.long	3215847669
	.long	1989165661
	.long	1069688852
	.long	4131454738
	.long	3215014128
	.long	3452987325
	.long	1068469531
	.long	2354117371
	.long	3220698211
	.long	3512651629
	.long	3162612920
	.long	0
	.long	0
	.long	3439278129
	.long	3216171577
	.long	1073008748
	.long	3159914989
	.long	0
	.long	1072693248
	.long	2934847296
	.long	1064716672
	.long	2625704508
	.long	3216178311
	.long	3692939630
	.long	1070976174
	.long	3072165553
	.long	3215558041
	.long	223733198
	.long	1069669653
	.long	1158528288
	.long	3214753465
	.long	3197803998
	.long	1068357327
	.long	2197390165
	.long	3220711079
	.long	3038089327
	.long	3163878824
	.long	0
	.long	0
	.long	3930011523
	.long	3215946901
	.long	3574565577
	.long	1010859792
	.long	0
	.long	1072693248
	.long	930814464
	.long	1064233867
	.long	3218126771
	.long	3215954682
	.long	873740287
	.long	1070966789
	.long	1580060562
	.long	3215272745
	.long	1505280051
	.long	1069653548
	.long	2221338643
	.long	3214421240
	.long	2641116540
	.long	1068312400
	.long	2040662960
	.long	3220723947
	.long	4270081753
	.long	1017046794
	.long	0
	.long	0
	.long	2319126882
	.long	3215533863
	.long	2132624928
	.long	3153789373
	.long	0
	.long	1072693248
	.long	2962101888
	.long	1063503197
	.long	3966050657
	.long	3215537841
	.long	3178024141
	.long	1070959142
	.long	1146830997
	.long	3214991029
	.long	1630003424
	.long	1069640468
	.long	2204651989
	.long	3214096041
	.long	796867276
	.long	1068243248
	.long	1883935754
	.long	3220736815
	.long	2988318241
	.long	1016251326
	.long	0
	.long	0
	.long	3919601072
	.long	3215121322
	.long	2384023593
	.long	1011053048
	.long	0
	.long	1072693248
	.long	1803946752
	.long	1062615560
	.long	2247013512
	.long	3215122999
	.long	3277043521
	.long	1070953215
	.long	578207576
	.long	3214490301
	.long	1619551293
	.long	1069630354
	.long	2648603598
	.long	3213667381
	.long	2728840488
	.long	1068193529
	.long	1727208548
	.long	3220749683
	.long	2531251621
	.long	1014661577
	.long	0
	.long	0
	.long	612012528
	.long	3214484294
	.long	549859680
	.long	1005659865
	.long	0
	.long	1072693248
	.long	2064191488
	.long	1061404484
	.long	4042886527
	.long	3214485286
	.long	149923164
	.long	1070948993
	.long	104448276
	.long	3213936806
	.long	2663431480
	.long	1069623160
	.long	4100741337
	.long	3213036454
	.long	3253086979
	.long	1068220130
	.long	1570481342
	.long	3220762551
	.long	2595802427
	.long	3161852775
	.long	0
	.long	0
	.long	75380366
	.long	3213435470
	.long	3015416698
	.long	1008346766
	.long	0
	.long	1072693248
	.long	2911696896
	.long	1059306942
	.long	451505506
	.long	3213435718
	.long	3089290734
	.long	1070946463
	.long	3705470162
	.long	3212886822
	.long	2568117130
	.long	1069618857
	.long	643302148
	.long	3211985100
	.long	500114129
	.long	1068202863
	.long	0
	.long	1090469888
	.long	0
	.long	1090469888
	.long	0
	.long	1090469888
	.long	0
	.long	1090469888
	.long	0
	.long	1090469888
	.long	0
	.long	1090469888
	.long	0
	.long	1090469888
	.long	0
	.long	1090469888
	.long	0
	.long	2147483648
	.long	0
	.long	2147483648
	.long	0
	.long	2147483648
	.long	0
	.long	2147483648
	.long	0
	.long	2147483648
	.long	0
	.long	2147483648
	.long	0
	.long	2147483648
	.long	0
	.long	2147483648
	.long	1841940611
	.long	1071931184
	.long	1841940611
	.long	1071931184
	.long	1841940611
	.long	1071931184
	.long	1841940611
	.long	1071931184
	.long	1841940611
	.long	1071931184
	.long	1841940611
	.long	1071931184
	.long	1841940611
	.long	1071931184
	.long	1841940611
	.long	1071931184
	.long	1413742592
	.long	1073291771
	.long	1413742592
	.long	1073291771
	.long	1413742592
	.long	1073291771
	.long	1413742592
	.long	1073291771
	.long	1413742592
	.long	1073291771
	.long	1413742592
	.long	1073291771
	.long	1413742592
	.long	1073291771
	.long	1413742592
	.long	1073291771
	.long	1280049152
	.long	1032227875
	.long	1280049152
	.long	1032227875
	.long	1280049152
	.long	1032227875
	.long	1280049152
	.long	1032227875
	.long	1280049152
	.long	1032227875
	.long	1280049152
	.long	1032227875
	.long	1280049152
	.long	1032227875
	.long	1280049152
	.long	1032227875
	.long	57701189
	.long	992578094
	.long	57701189
	.long	992578094
	.long	57701189
	.long	992578094
	.long	57701189
	.long	992578094
	.long	57701189
	.long	992578094
	.long	57701189
	.long	992578094
	.long	57701189
	.long	992578094
	.long	57701189
	.long	992578094
	.long	1413754136
	.long	1073291771
	.long	1413754136
	.long	1073291771
	.long	1413754136
	.long	1073291771
	.long	1413754136
	.long	1073291771
	.long	1413754136
	.long	1073291771
	.long	1413754136
	.long	1073291771
	.long	1413754136
	.long	1073291771
	.long	1413754136
	.long	1073291771
	.long	856972294
	.long	1016178214
	.long	856972294
	.long	1016178214
	.long	856972294
	.long	1016178214
	.long	856972294
	.long	1016178214
	.long	856972294
	.long	1016178214
	.long	856972294
	.long	1016178214
	.long	856972294
	.long	1016178214
	.long	856972294
	.long	1016178214
	.long	688016905
	.long	961289425
	.long	688016905
	.long	961289425
	.long	688016905
	.long	961289425
	.long	688016905
	.long	961289425
	.long	688016905
	.long	961289425
	.long	688016905
	.long	961289425
	.long	688016905
	.long	961289425
	.long	688016905
	.long	961289425
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	3191264137
	.long	3217058711
	.long	3191264137
	.long	3217058711
	.long	3191264137
	.long	3217058711
	.long	3191264137
	.long	3217058711
	.long	3191264137
	.long	3217058711
	.long	3191264137
	.long	3217058711
	.long	3191264137
	.long	3217058711
	.long	3191264137
	.long	3217058711
	.long	3706049281
	.long	1063713903
	.long	3706049281
	.long	1063713903
	.long	3706049281
	.long	1063713903
	.long	3706049281
	.long	1063713903
	.long	3706049281
	.long	1063713903
	.long	3706049281
	.long	1063713903
	.long	3706049281
	.long	1063713903
	.long	3706049281
	.long	1063713903
	.long	3458873344
	.long	3202311037
	.long	3458873344
	.long	3202311037
	.long	3458873344
	.long	3202311037
	.long	3458873344
	.long	3202311037
	.long	3458873344
	.long	3202311037
	.long	3458873344
	.long	3202311037
	.long	3458873344
	.long	3202311037
	.long	3458873344
	.long	3202311037
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	0
	.long	1072693248
	.long	879804142
	.long	3218967329
	.long	879804142
	.long	3218967329
	.long	879804142
	.long	3218967329
	.long	879804142
	.long	3218967329
	.long	879804142
	.long	3218967329
	.long	879804142
	.long	3218967329
	.long	879804142
	.long	3218967329
	.long	879804142
	.long	3218967329
	.long	2328005206
	.long	1066919862
	.long	2328005206
	.long	1066919862
	.long	2328005206
	.long	1066919862
	.long	2328005206
	.long	1066919862
	.long	2328005206
	.long	1066919862
	.long	2328005206
	.long	1066919862
	.long	2328005206
	.long	1066919862
	.long	2328005206
	.long	1066919862
	.long	62689958
	.long	3207287387
	.long	62689958
	.long	3207287387
	.long	62689958
	.long	3207287387
	.long	62689958
	.long	3207287387
	.long	62689958
	.long	3207287387
	.long	62689958
	.long	3207287387
	.long	62689958
	.long	3207287387
	.long	62689958
	.long	3207287387
	.type	__svml_dtan_data_internal,@object
	.size	__svml_dtan_data_internal,29248
	.align 64
__svml_dtan_reduction_data_internal:
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	2
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	5
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	10
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	20
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	40
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	81
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	162
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	325
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	651
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1303
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	2607
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	5215
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	10430
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	20860
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	41721
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	83443
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	166886
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	333772
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	667544
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1335088
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	2670176
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	5340353
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	10680707
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	21361414
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	42722829
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	85445659
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	170891318
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	341782637
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	683565275
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1367130551
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	2734261102
	.long	0
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1173554908
	.long	1
	.long	0
	.long	0
	.long	0
	.long	0
	.long	2347109817
	.long	2
	.long	0
	.long	0
	.long	0
	.long	0
	.long	399252338
	.long	5
	.long	0
	.long	0
	.long	0
	.long	0
	.long	798504676
	.long	10
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1597009353
	.long	20
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3194018707
	.long	40
	.long	0
	.long	0
	.long	0
	.long	0
	.long	2093070119
	.long	81
	.long	0
	.long	0
	.long	0
	.long	0
	.long	4186140238
	.long	162
	.long	0
	.long	0
	.long	0
	.long	0
	.long	4077313180
	.long	325
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3859659065
	.long	651
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3424350834
	.long	1303
	.long	0
	.long	0
	.long	0
	.long	0
	.long	2553734372
	.long	2607
	.long	0
	.long	0
	.long	0
	.long	0
	.long	812501448
	.long	5215
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1625002897
	.long	10430
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3250005794
	.long	20860
	.long	0
	.long	0
	.long	0
	.long	0
	.long	2205044292
	.long	41721
	.long	0
	.long	0
	.long	0
	.long	0
	.long	115121288
	.long	83443
	.long	0
	.long	0
	.long	0
	.long	0
	.long	230242576
	.long	166886
	.long	0
	.long	0
	.long	0
	.long	0
	.long	460485152
	.long	333772
	.long	0
	.long	0
	.long	0
	.long	0
	.long	920970305
	.long	667544
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1841940610
	.long	1335088
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3683881221
	.long	2670176
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3072795146
	.long	5340353
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1850622997
	.long	10680707
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3701245994
	.long	21361414
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3107524692
	.long	42722829
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1920082089
	.long	85445659
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3840164178
	.long	170891318
	.long	0
	.long	0
	.long	0
	.long	0
	.long	3385361061
	.long	341782637
	.long	0
	.long	0
	.long	0
	.long	0
	.long	2475754826
	.long	683565275
	.long	0
	.long	0
	.long	0
	.long	0
	.long	656542356
	.long	1367130551
	.long	0
	.long	0
	.long	0
	.long	0
	.long	1313084713
	.long	2734261102
	.long	0
	.long	0
	.long	1
	.long	0
	.long	2626169427
	.long	1173554908
	.long	0
	.long	0
	.long	2
	.long	0
	.long	957371559
	.long	2347109817
	.long	0
	.long	0
	.long	5
	.long	0
	.long	1914743119
	.long	399252338
	.long	0
	.long	0
	.long	10
	.long	0
	.long	3829486239
	.long	798504676
	.long	0
	.long	0
	.long	20
	.long	0
	.long	3364005183
	.long	1597009353
	.long	0
	.long	0
	.long	40
	.long	0
	.long	2433043071
	.long	3194018707
	.long	0
	.long	0
	.long	81
	.long	0
	.long	571118846
	.long	2093070119
	.long	0
	.long	0
	.long	162
	.long	0
	.long	1142237692
	.long	4186140238
	.long	0
	.long	0
	.long	325
	.long	0
	.long	2284475384
	.long	4077313180
	.long	0
	.long	0
	.long	651
	.long	0
	.long	273983472
	.long	3859659065
	.long	0
	.long	0
	.long	1303
	.long	0
	.long	547966945
	.long	3424350834
	.long	0
	.long	0
	.long	2607
	.long	0
	.long	1095933890
	.long	2553734372
	.long	0
	.long	0
	.long	5215
	.long	0
	.long	2191867780
	.long	812501448
	.long	0
	.long	0
	.long	10430
	.long	0
	.long	88768265
	.long	1625002897
	.long	0
	.long	0
	.long	20860
	.long	0
	.long	177536531
	.long	3250005794
	.long	0
	.long	0
	.long	41721
	.long	0
	.long	355073063
	.long	2205044292
	.long	0
	.long	0
	.long	83443
	.long	0
	.long	710146126
	.long	115121288
	.long	0
	.long	0
	.long	166886
	.long	0
	.long	1420292253
	.long	230242576
	.long	0
	.long	0
	.long	333772
	.long	0
	.long	2840584506
	.long	460485152
	.long	0
	.long	0
	.long	667544
	.long	0
	.long	1386201717
	.long	920970305
	.long	0
	.long	0
	.long	1335088
	.long	0
	.long	2772403434
	.long	1841940610
	.long	0
	.long	0
	.long	2670176
	.long	0
	.long	1249839573
	.long	3683881221
	.long	0
	.long	0
	.long	5340353
	.long	0
	.long	2499679147
	.long	3072795146
	.long	0
	.long	0
	.long	10680707
	.long	0
	.long	704390999
	.long	1850622997
	.long	0
	.long	0
	.long	21361414
	.long	0
	.long	1408781999
	.long	3701245994
	.long	0
	.long	0
	.long	42722829
	.long	0
	.long	2817563999
	.long	3107524692
	.long	0
	.long	0
	.long	85445659
	.long	0
	.long	1340160702
	.long	1920082089
	.long	0
	.long	0
	.long	170891318
	.long	0
	.long	2680321405
	.long	3840164178
	.long	0
	.long	0
	.long	341782637
	.long	0
	.long	1065675514
	.long	3385361061
	.long	0
	.long	0
	.long	683565275
	.long	0
	.long	2131351028
	.long	2475754826
	.long	0
	.long	0
	.long	1367130551
	.long	0
	.long	4262702056
	.long	656542356
	.long	0
	.long	0
	.long	2734261102
	.long	0
	.long	4230436817
	.long	1313084713
	.long	0
	.long	0
	.long	1173554908
	.long	1
	.long	4165906339
	.long	2626169427
	.long	0
	.long	0
	.long	2347109817
	.long	2
	.long	4036845383
	.long	957371559
	.long	0
	.long	0
	.long	399252338
	.long	5
	.long	3778723471
	.long	1914743119
	.long	0
	.long	0
	.long	798504676
	.long	10
	.long	3262479647
	.long	3829486239
	.long	0
	.long	0
	.long	1597009353
	.long	20
	.long	2229991998
	.long	3364005183
	.long	0
	.long	0
	.long	3194018707
	.long	40
	.long	165016701
	.long	2433043071
	.long	0
	.long	0
	.long	2093070119
	.long	81
	.long	330033402
	.long	571118846
	.long	0
	.long	0
	.long	4186140238
	.long	162
	.long	660066805
	.long	1142237692
	.long	0
	.long	0
	.long	4077313180
	.long	325
	.long	1320133610
	.long	2284475384
	.long	0
	.long	0
	.long	3859659065
	.long	651
	.long	2640267220
	.long	273983472
	.long	0
	.long	0
	.long	3424350834
	.long	1303
	.long	985567145
	.long	547966945
	.long	0
	.long	0
	.long	2553734372
	.long	2607
	.long	1971134291
	.long	1095933890
	.long	0
	.long	0
	.long	812501448
	.long	5215
	.long	3942268582
	.long	2191867780
	.long	0
	.long	0
	.long	1625002897
	.long	10430
	.long	3589569869
	.long	88768265
	.long	0
	.long	0
	.long	3250005794
	.long	20860
	.long	2884172442
	.long	177536531
	.long	0
	.long	0
	.long	2205044292
	.long	41721
	.long	1473377588
	.long	355073063
	.long	0
	.long	0
	.long	115121288
	.long	83443
	.long	2946755177
	.long	710146126
	.long	0
	.long	0
	.long	230242576
	.long	166886
	.long	1598543059
	.long	1420292253
	.long	0
	.long	0
	.long	460485152
	.long	333772
	.long	3197086118
	.long	2840584506
	.long	0
	.long	0
	.long	920970305
	.long	667544
	.long	2099204941
	.long	1386201717
	.long	0
	.long	0
	.long	1841940610
	.long	1335088
	.long	4198409883
	.long	2772403434
	.long	0
	.long	0
	.long	3683881221
	.long	2670176
	.long	4101852471
	.long	1249839573
	.long	0
	.long	0
	.long	3072795146
	.long	5340353
	.long	3908737646
	.long	2499679147
	.long	0
	.long	0
	.long	1850622997
	.long	10680707
	.long	3522507997
	.long	704390999
	.long	0
	.long	0
	.long	3701245994
	.long	21361414
	.long	2750048699
	.long	1408781999
	.long	0
	.long	0
	.long	3107524692
	.long	42722829
	.long	1205130103
	.long	2817563999
	.long	0
	.long	0
	.long	1920082089
	.long	85445659
	.long	2410260206
	.long	1340160702
	.long	0
	.long	0
	.long	3840164178
	.long	170891318
	.long	525553116
	.long	2680321405
	.long	0
	.long	0
	.long	3385361061
	.long	341782637
	.long	1051106232
	.long	1065675514
	.long	0
	.long	0
	.long	2475754826
	.long	683565275
	.long	2102212464
	.long	2131351028
	.long	0
	.long	0
	.long	656542356
	.long	1367130551
	.long	4204424928
	.long	4262702056
	.long	0
	.long	0
	.long	1313084713
	.long	2734261102
	.long	4113882560
	.long	4230436817
	.long	1
	.long	0
	.long	2626169427
	.long	1173554908
	.long	3932797825
	.long	4165906339
	.long	2
	.long	0
	.long	957371559
	.long	2347109817
	.long	3570628355
	.long	4036845383
	.long	5
	.long	0
	.long	1914743119
	.long	399252338
	.long	2846289414
	.long	3778723471
	.long	10
	.long	0
	.long	3829486239
	.long	798504676
	.long	1397611533
	.long	3262479647
	.long	20
	.long	0
	.long	3364005183
	.long	1597009353
	.long	2795223067
	.long	2229991998
	.long	40
	.long	0
	.long	2433043071
	.long	3194018707
	.long	1295478838
	.long	165016701
	.long	81
	.long	0
	.long	571118846
	.long	2093070119
	.long	2590957677
	.long	330033402
	.long	162
	.long	0
	.long	1142237692
	.long	4186140238
	.long	886948059
	.long	660066805
	.long	325
	.long	0
	.long	2284475384
	.long	4077313180
	.long	1773896118
	.long	1320133610
	.long	651
	.long	0
	.long	273983472
	.long	3859659065
	.long	3547792237
	.long	2640267220
	.long	1303
	.long	0
	.long	547966945
	.long	3424350834
	.long	2800617179
	.long	985567145
	.long	2607
	.long	0
	.long	1095933890
	.long	2553734372
	.long	1306267062
	.long	1971134291
	.long	5215
	.long	0
	.long	2191867780
	.long	812501448
	.long	2612534124
	.long	3942268582
	.long	10430
	.long	0
	.long	88768265
	.long	1625002897
	.long	930100952
	.long	3589569869
	.long	20860
	.long	0
	.long	177536531
	.long	3250005794
	.long	1860201905
	.long	2884172442
	.long	41721
	.long	0
	.long	355073063
	.long	2205044292
	.long	3720403810
	.long	1473377588
	.long	83443
	.long	0
	.long	710146126
	.long	115121288
	.long	3145840325
	.long	2946755177
	.long	166886
	.long	0
	.long	1420292253
	.long	230242576
	.long	1996713354
	.long	1598543059
	.long	333772
	.long	0
	.long	2840584506
	.long	460485152
	.long	3993426708
	.long	3197086118
	.long	667544
	.long	0
	.long	1386201717
	.long	920970305
	.long	3691886121
	.long	2099204941
	.long	1335088
	.long	0
	.long	2772403434
	.long	1841940610
	.long	3088804946
	.long	4198409883
	.long	2670176
	.long	0
	.long	1249839573
	.long	3683881221
	.long	1882642597
	.long	4101852471
	.long	5340353
	.long	0
	.long	2499679147
	.long	3072795146
	.long	3765285194
	.long	3908737646
	.long	10680707
	.long	0
	.long	704390999
	.long	1850622997
	.long	3235603093
	.long	3522507997
	.long	21361414
	.long	0
	.long	1408781999
	.long	3701245994
	.long	2176238891
	.long	2750048699
	.long	42722829
	.long	0
	.long	2817563999
	.long	3107524692
	.long	57510486
	.long	1205130103
	.long	85445659
	.long	0
	.long	1340160702
	.long	1920082089
	.long	115020972
	.long	2410260206
	.long	170891318
	.long	0
	.long	2680321405
	.long	3840164178
	.long	230041945
	.long	525553116
	.long	341782637
	.long	0
	.long	1065675514
	.long	3385361061
	.long	460083891
	.long	1051106232
	.long	683565275
	.long	0
	.long	2131351028
	.long	2475754826
	.long	920167782
	.long	2102212464
	.long	1367130551
	.long	0
	.long	4262702056
	.long	656542356
	.long	1840335564
	.long	4204424928
	.long	2734261102
	.long	0
	.long	4230436817
	.long	1313084713
	.long	3680671129
	.long	4113882560
	.long	1173554908
	.long	1
	.long	4165906339
	.long	2626169427
	.long	3066374962
	.long	3932797825
	.long	2347109817
	.long	2
	.long	4036845383
	.long	957371559
	.long	1837782628
	.long	3570628355
	.long	399252338
	.long	5
	.long	3778723471
	.long	1914743119
	.long	3675565257
	.long	2846289414
	.long	798504676
	.long	10
	.long	3262479647
	.long	3829486239
	.long	3056163219
	.long	1397611533
	.long	1597009353
	.long	20
	.long	2229991998
	.long	3364005183
	.long	1817359143
	.long	2795223067
	.long	3194018707
	.long	40
	.long	165016701
	.long	2433043071
	.long	3634718287
	.long	1295478838
	.long	2093070119
	.long	81
	.long	330033402
	.long	571118846
	.long	2974469278
	.long	2590957677
	.long	4186140238
	.long	162
	.long	660066805
	.long	1142237692
	.long	1653971260
	.long	886948059
	.long	4077313180
	.long	325
	.long	1320133610
	.long	2284475384
	.long	3307942520
	.long	1773896118
	.long	3859659065
	.long	651
	.long	2640267220
	.long	273983472
	.long	2320917745
	.long	3547792237
	.long	3424350834
	.long	1303
	.long	985567145
	.long	547966945
	.long	346868194
	.long	2800617179
	.long	2553734372
	.long	2607
	.long	1971134291
	.long	1095933890
	.long	693736388
	.long	1306267062
	.long	812501448
	.long	5215
	.long	3942268582
	.long	2191867780
	.long	1387472776
	.long	2612534124
	.long	1625002897
	.long	10430
	.long	3589569869
	.long	88768265
	.long	2774945552
	.long	930100952
	.long	3250005794
	.long	20860
	.long	2884172442
	.long	177536531
	.long	1254923809
	.long	1860201905
	.long	2205044292
	.long	41721
	.long	1473377588
	.long	355073063
	.long	2509847619
	.long	3720403810
	.long	115121288
	.long	83443
	.long	2946755177
	.long	710146126
	.long	724727943
	.long	3145840325
	.long	230242576
	.long	166886
	.long	1598543059
	.long	1420292253
	.long	1449455886
	.long	1996713354
	.long	460485152
	.long	333772
	.long	3197086118
	.long	2840584506
	.long	2898911772
	.long	3993426708
	.long	920970305
	.long	667544
	.long	2099204941
	.long	1386201717
	.long	1502856249
	.long	3691886121
	.long	1841940610
	.long	1335088
	.long	4198409883
	.long	2772403434
	.long	3005712498
	.long	3088804946
	.long	3683881221
	.long	2670176
	.long	4101852471
	.long	1249839573
	.long	1716457700
	.long	1882642597
	.long	3072795146
	.long	5340353
	.long	3908737646
	.long	2499679147
	.long	3432915400
	.long	3765285194
	.long	1850622997
	.long	10680707
	.long	3522507997
	.long	704390999
	.long	2570863504
	.long	3235603093
	.long	3701245994
	.long	21361414
	.long	2750048699
	.long	1408781999
	.long	846759712
	.long	2176238891
	.long	3107524692
	.long	42722829
	.long	1205130103
	.long	2817563999
	.long	1693519425
	.long	57510486
	.long	1920082089
	.long	85445659
	.long	2410260206
	.long	1340160702
	.long	3387038850
	.long	115020972
	.long	3840164178
	.long	170891318
	.long	525553116
	.long	2680321405
	.long	2479110404
	.long	230041945
	.long	3385361061
	.long	341782637
	.long	1051106232
	.long	1065675514
	.long	663253512
	.long	460083891
	.long	2475754826
	.long	683565275
	.long	2102212464
	.long	2131351028
	.long	1326507024
	.long	920167782
	.long	656542356
	.long	1367130551
	.long	4204424928
	.long	4262702056
	.long	2653014048
	.long	1840335564
	.long	1313084713
	.long	2734261102
	.long	4113882560
	.long	4230436817
	.long	1011060801
	.long	3680671129
	.long	2626169427
	.long	1173554908
	.long	3932797825
	.long	4165906339
	.long	2022121603
	.long	3066374962
	.long	957371559
	.long	2347109817
	.long	3570628355
	.long	4036845383
	.long	4044243207
	.long	1837782628
	.long	1914743119
	.long	399252338
	.long	2846289414
	.long	3778723471
	.long	3793519119
	.long	3675565257
	.long	3829486239
	.long	798504676
	.long	1397611533
	.long	3262479647
	.long	3292070943
	.long	3056163219
	.long	3364005183
	.long	1597009353
	.long	2795223067
	.long	2229991998
	.long	2289174591
	.long	1817359143
	.long	2433043071
	.long	3194018707
	.long	1295478838
	.long	165016701
	.long	283381887
	.long	3634718287
	.long	571118846
	.long	2093070119
	.long	2590957677
	.long	330033402
	.long	566763775
	.long	2974469278
	.long	1142237692
	.long	4186140238
	.long	886948059
	.long	660066805
	.long	1133527550
	.long	1653971260
	.long	2284475384
	.long	4077313180
	.long	1773896118
	.long	1320133610
	.long	2267055100
	.long	3307942520
	.long	273983472
	.long	3859659065
	.long	3547792237
	.long	2640267220
	.long	239142905
	.long	2320917745
	.long	547966945
	.long	3424350834
	.long	2800617179
	.long	985567145
	.long	478285810
	.long	346868194
	.long	1095933890
	.long	2553734372
	.long	1306267062
	.long	1971134291
	.long	956571621
	.long	693736388
	.long	2191867780
	.long	812501448
	.long	2612534124
	.long	3942268582
	.long	1913143242
	.long	1387472776
	.long	88768265
	.long	1625002897
	.long	930100952
	.long	3589569869
	.long	3826286484
	.long	2774945552
	.long	177536531
	.long	3250005794
	.long	1860201905
	.long	2884172442
	.long	3357605672
	.long	1254923809
	.long	355073063
	.long	2205044292
	.long	3720403810
	.long	1473377588
	.long	2420244049
	.long	2509847619
	.long	710146126
	.long	115121288
	.long	3145840325
	.long	2946755177
	.long	545520802
	.long	724727943
	.long	1420292253
	.long	230242576
	.long	1996713354
	.long	1598543059
	.long	1091041605
	.long	1449455886
	.long	2840584506
	.long	460485152
	.long	3993426708
	.long	3197086118
	.long	2182083211
	.long	2898911772
	.long	1386201717
	.long	920970305
	.long	3691886121
	.long	2099204941
	.long	69199126
	.long	1502856249
	.long	2772403434
	.long	1841940610
	.long	3088804946
	.long	4198409883
	.long	138398252
	.long	3005712498
	.long	1249839573
	.long	3683881221
	.long	1882642597
	.long	4101852471
	.long	276796504
	.long	1716457700
	.long	2499679147
	.long	3072795146
	.long	3765285194
	.long	3908737646
	.long	553593009
	.long	3432915400
	.long	704390999
	.long	1850622997
	.long	3235603093
	.long	3522507997
	.long	1107186019
	.long	2570863504
	.long	1408781999
	.long	3701245994
	.long	2176238891
	.long	2750048699
	.long	2214372039
	.long	846759712
	.long	2817563999
	.long	3107524692
	.long	57510486
	.long	1205130103
	.long	133776782
	.long	1693519425
	.long	1340160702
	.long	1920082089
	.long	115020972
	.long	2410260206
	.long	267553565
	.long	3387038850
	.long	2680321405
	.long	3840164178
	.long	230041945
	.long	525553116
	.long	535107130
	.long	2479110404
	.long	1065675514
	.long	3385361061
	.long	460083891
	.long	1051106232
	.long	1070214261
	.long	663253512
	.long	2131351028
	.long	2475754826
	.long	920167782
	.long	2102212464
	.long	2140428522
	.long	1326507024
	.long	4262702056
	.long	656542356
	.long	1840335564
	.long	4204424928
	.long	4280857045
	.long	2653014048
	.long	4230436817
	.long	1313084713
	.long	3680671129
	.long	4113882560
	.long	4266746795
	.long	1011060801
	.long	4165906339
	.long	2626169427
	.long	3066374962
	.long	3932797825
	.long	4238526295
	.long	2022121603
	.long	4036845383
	.long	957371559
	.long	1837782628
	.long	3570628355
	.long	4182085295
	.long	4044243207
	.long	3778723471
	.long	1914743119
	.long	3675565257
	.long	2846289414
	.long	4069203294
	.long	3793519119
	.long	3262479647
	.long	3829486239
	.long	3056163219
	.long	1397611533
	.long	3843439293
	.long	3292070943
	.long	2229991998
	.long	3364005183
	.long	1817359143
	.long	2795223067
	.long	3391911291
	.long	2289174591
	.long	165016701
	.long	2433043071
	.long	3634718287
	.long	1295478838
	.long	2488855287
	.long	283381887
	.long	330033402
	.long	571118846
	.long	2974469278
	.long	2590957677
	.long	682743279
	.long	566763775
	.long	660066805
	.long	1142237692
	.long	1653971260
	.long	886948059
	.long	1365486558
	.long	1133527550
	.long	1320133610
	.long	2284475384
	.long	3307942520
	.long	1773896118
	.long	2730973117
	.long	2267055100
	.long	2640267220
	.long	273983472
	.long	2320917745
	.long	3547792237
	.long	1166978938
	.long	239142905
	.long	985567145
	.long	547966945
	.long	346868194
	.long	2800617179
	.long	2333957877
	.long	478285810
	.long	1971134291
	.long	1095933890
	.long	693736388
	.long	1306267062
	.long	372948459
	.long	956571621
	.long	3942268582
	.long	2191867780
	.long	1387472776
	.long	2612534124
	.long	745896919
	.long	1913143242
	.long	3589569869
	.long	88768265
	.long	2774945552
	.long	930100952
	.long	1491793838
	.long	3826286484
	.long	2884172442
	.long	177536531
	.long	1254923809
	.long	1860201905
	.long	2983587677
	.long	3357605672
	.long	1473377588
	.long	355073063
	.long	2509847619
	.long	3720403810
	.long	1672208059
	.long	2420244049
	.long	2946755177
	.long	710146126
	.long	724727943
	.long	3145840325
	.long	3344416119
	.long	545520802
	.long	1598543059
	.long	1420292253
	.long	1449455886
	.long	1996713354
	.long	2393864943
	.long	1091041605
	.long	3197086118
	.long	2840584506
	.long	2898911772
	.long	3993426708
	.long	492762590
	.long	2182083211
	.long	2099204941
	.long	1386201717
	.long	1502856249
	.long	3691886121
	.long	985525180
	.long	69199126
	.long	4198409883
	.long	2772403434
	.long	3005712498
	.long	3088804946
	.long	1971050360
	.long	138398252
	.long	4101852471
	.long	1249839573
	.long	1716457700
	.long	1882642597
	.long	3942100721
	.long	276796504
	.long	3908737646
	.long	2499679147
	.long	3432915400
	.long	3765285194
	.long	3589234146
	.long	553593009
	.long	3522507997
	.long	704390999
	.long	2570863504
	.long	3235603093
	.long	2883500997
	.long	1107186019
	.long	2750048699
	.long	1408781999
	.long	846759712
	.long	2176238891
	.long	1472034698
	.long	2214372039
	.long	1205130103
	.long	2817563999
	.long	1693519425
	.long	57510486
	.long	2944069397
	.long	133776782
	.long	2410260206
	.long	1340160702
	.long	3387038850
	.long	115020972
	.long	1593171499
	.long	267553565
	.long	525553116
	.long	2680321405
	.long	2479110404
	.long	230041945
	.long	3186342998
	.long	535107130
	.long	1051106232
	.long	1065675514
	.long	663253512
	.long	460083891
	.long	2077718700
	.long	1070214261
	.long	2102212464
	.long	2131351028
	.long	1326507024
	.long	920167782
	.long	4155437400
	.long	2140428522
	.long	4204424928
	.long	4262702056
	.long	2653014048
	.long	1840335564
	.long	4015907504
	.long	4280857045
	.long	4113882560
	.long	4230436817
	.long	1011060801
	.long	3680671129
	.long	3736847713
	.long	4266746795
	.long	3932797825
	.long	4165906339
	.long	2022121603
	.long	3066374962
	.long	3178728131
	.long	4238526295
	.long	3570628355
	.long	4036845383
	.long	4044243207
	.long	1837782628
	.long	2062488966
	.long	4182085295
	.long	2846289414
	.long	3778723471
	.long	3793519119
	.long	3675565257
	.long	4124977933
	.long	4069203294
	.long	1397611533
	.long	3262479647
	.long	3292070943
	.long	3056163219
	.long	3954988571
	.long	3843439293
	.long	2795223067
	.long	2229991998
	.long	2289174591
	.long	1817359143
	.long	3615009846
	.long	3391911291
	.long	1295478838
	.long	165016701
	.long	283381887
	.long	3634718287
	.long	2935052397
	.long	2488855287
	.long	2590957677
	.long	330033402
	.long	566763775
	.long	2974469278
	.long	1575137499
	.long	682743279
	.long	886948059
	.long	660066805
	.long	1133527550
	.long	1653971260
	.long	3150274999
	.long	1365486558
	.long	1773896118
	.long	1320133610
	.long	2267055100
	.long	3307942520
	.long	2005582702
	.long	2730973117
	.long	3547792237
	.long	2640267220
	.long	239142905
	.long	2320917745
	.long	4011165404
	.long	1166978938
	.long	2800617179
	.long	985567145
	.long	478285810
	.long	346868194
	.long	3727363513
	.long	2333957877
	.long	1306267062
	.long	1971134291
	.long	956571621
	.long	693736388
	.long	3159759730
	.long	372948459
	.long	2612534124
	.long	3942268582
	.long	1913143242
	.long	1387472776
	.long	2024552164
	.long	745896919
	.long	930100952
	.long	3589569869
	.long	3826286484
	.long	2774945552
	.long	4049104329
	.long	1491793838
	.long	1860201905
	.long	2884172442
	.long	3357605672
	.long	1254923809
	.long	3803241362
	.long	2983587677
	.long	3720403810
	.long	1473377588
	.long	2420244049
	.long	2509847619
	.long	3311515428
	.long	1672208059
	.long	3145840325
	.long	2946755177
	.long	545520802
	.long	724727943
	.long	2328063560
	.long	3344416119
	.long	1996713354
	.long	1598543059
	.long	1091041605
	.long	1449455886
	.long	361159825
	.long	2393864943
	.long	3993426708
	.long	3197086118
	.long	2182083211
	.long	2898911772
	.long	722319651
	.long	492762590
	.long	3691886121
	.long	2099204941
	.long	69199126
	.long	1502856249
	.long	1444639302
	.long	985525180
	.long	3088804946
	.long	4198409883
	.long	138398252
	.long	3005712498
	.long	2889278605
	.long	1971050360
	.long	1882642597
	.long	4101852471
	.long	276796504
	.long	1716457700
	.long	1483589915
	.long	3942100721
	.long	3765285194
	.long	3908737646
	.long	553593009
	.long	3432915400
	.long	2967179831
	.long	3589234146
	.long	3235603093
	.long	3522507997
	.long	1107186019
	.long	2570863504
	.long	1639392366
	.long	2883500997
	.long	2176238891
	.long	2750048699
	.long	2214372039
	.long	846759712
	.long	3278784732
	.long	1472034698
	.long	57510486
	.long	1205130103
	.long	133776782
	.long	1693519425
	.long	2262602168
	.long	2944069397
	.long	115020972
	.long	2410260206
	.long	267553565
	.long	3387038850
	.long	230237041
	.long	1593171499
	.long	230041945
	.long	525553116
	.long	535107130
	.long	2479110404
	.long	460474083
	.long	3186342998
	.long	460083891
	.long	1051106232
	.long	1070214261
	.long	663253512
	.long	920948167
	.long	2077718700
	.long	920167782
	.long	2102212464
	.long	2140428522
	.long	1326507024
	.long	1841896334
	.long	4155437400
	.long	1840335564
	.long	4204424928
	.long	4280857045
	.long	2653014048
	.long	3683792669
	.long	4015907504
	.long	3680671129
	.long	4113882560
	.long	4266746795
	.long	1011060801
	.long	3072618042
	.long	3736847713
	.long	3066374962
	.long	3932797825
	.long	4238526295
	.long	2022121603
	.long	1850268788
	.long	3178728131
	.long	1837782628
	.long	3570628355
	.long	4182085295
	.long	4044243207
	.long	3700537577
	.long	2062488966
	.long	3675565257
	.long	2846289414
	.long	4069203294
	.long	3793519119
	.long	3106107858
	.long	4124977933
	.long	3056163219
	.long	1397611533
	.long	3843439293
	.long	3292070943
	.long	1917248420
	.long	3954988571
	.long	1817359143
	.long	2795223067
	.long	3391911291
	.long	2289174591
	.long	3834496840
	.long	3615009846
	.long	3634718287
	.long	1295478838
	.long	2488855287
	.long	283381887
	.long	3374026384
	.long	2935052397
	.long	2974469278
	.long	2590957677
	.long	682743279
	.long	566763775
	.long	2453085473
	.long	1575137499
	.long	1653971260
	.long	886948059
	.long	1365486558
	.long	1133527550
	.long	611203650
	.long	3150274999
	.long	3307942520
	.long	1773896118
	.long	2730973117
	.long	2267055100
	.long	1222407300
	.long	2005582702
	.long	2320917745
	.long	3547792237
	.long	1166978938
	.long	239142905
	.long	2444814601
	.long	4011165404
	.long	346868194
	.long	2800617179
	.long	2333957877
	.long	478285810
	.long	594661906
	.long	3727363513
	.long	693736388
	.long	1306267062
	.long	372948459
	.long	956571621
	.long	1189323812
	.long	3159759730
	.long	1387472776
	.long	2612534124
	.long	745896919
	.long	1913143242
	.long	2378647625
	.long	2024552164
	.long	2774945552
	.long	930100952
	.long	1491793838
	.long	3826286484
	.long	462327955
	.long	4049104329
	.long	1254923809
	.long	1860201905
	.long	2983587677
	.long	3357605672
	.long	924655910
	.long	3803241362
	.long	2509847619
	.long	3720403810
	.long	1672208059
	.long	2420244049
	.long	1849311821
	.long	3311515428
	.long	724727943
	.long	3145840325
	.long	3344416119
	.long	545520802
	.long	3698623643
	.long	2328063560
	.long	1449455886
	.long	1996713354
	.long	2393864943
	.long	1091041605
	.long	3102279991
	.long	361159825
	.long	2898911772
	.long	3993426708
	.long	492762590
	.long	2182083211
	.long	1909592686
	.long	722319651
	.long	1502856249
	.long	3691886121
	.long	985525180
	.long	69199126
	.long	3819185373
	.long	1444639302
	.long	3005712498
	.long	3088804946
	.long	1971050360
	.long	138398252
	.long	3343403450
	.long	2889278605
	.long	1716457700
	.long	1882642597
	.long	3942100721
	.long	276796504
	.long	2391839604
	.long	1483589915
	.long	3432915400
	.long	3765285194
	.long	3589234146
	.long	553593009
	.long	488711913
	.long	2967179831
	.long	2570863504
	.long	3235603093
	.long	2883500997
	.long	1107186019
	.long	977423826
	.long	1639392366
	.long	846759712
	.long	2176238891
	.long	1472034698
	.long	2214372039
	.long	1954847653
	.long	3278784732
	.long	1693519425
	.long	57510486
	.long	2944069397
	.long	133776782
	.long	3909695307
	.long	2262602168
	.long	3387038850
	.long	115020972
	.long	1593171499
	.long	267553565
	.long	3524423319
	.long	230237041
	.long	2479110404
	.long	230041945
	.long	3186342998
	.long	535107130
	.long	2753879342
	.long	460474083
	.long	663253512
	.long	460083891
	.long	2077718700
	.long	1070214261
	.long	1212791388
	.long	920948167
	.long	1326507024
	.long	920167782
	.long	4155437400
	.long	2140428522
	.long	2425582776
	.long	1841896334
	.long	2653014048
	.long	1840335564
	.long	4015907504
	.long	4280857045
	.long	556198256
	.long	3683792669
	.long	1011060801
	.long	3680671129
	.long	3736847713
	.long	4266746795
	.long	1112396512
	.long	3072618042
	.long	2022121603
	.long	3066374962
	.long	3178728131
	.long	4238526295
	.long	2224793024
	.long	1850268788
	.long	4044243207
	.long	1837782628
	.long	2062488966
	.long	4182085295
	.long	154618752
	.long	3700537577
	.long	3793519119
	.long	3675565257
	.long	4124977933
	.long	4069203294
	.long	309237504
	.long	3106107858
	.long	3292070943
	.long	3056163219
	.long	3954988571
	.long	3843439293
	.long	618475008
	.long	1917248420
	.long	2289174591
	.long	1817359143
	.long	3615009846
	.long	3391911291
	.long	1236950016
	.long	3834496840
	.long	283381887
	.long	3634718287
	.long	2935052397
	.long	2488855287
	.long	2473900033
	.long	3374026384
	.long	566763775
	.long	2974469278
	.long	1575137499
	.long	682743279
	.long	652832771
	.long	2453085473
	.long	1133527550
	.long	1653971260
	.long	3150274999
	.long	1365486558
	.long	1305665542
	.long	611203650
	.long	2267055100
	.long	3307942520
	.long	2005582702
	.long	2730973117
	.long	2611331084
	.long	1222407300
	.long	239142905
	.long	2320917745
	.long	4011165404
	.long	1166978938
	.long	927694873
	.long	2444814601
	.long	478285810
	.long	346868194
	.long	3727363513
	.long	2333957877
	.long	1855389746
	.long	594661906
	.long	956571621
	.long	693736388
	.long	3159759730
	.long	372948459
	.long	3710779492
	.long	1189323812
	.long	1913143242
	.long	1387472776
	.long	2024552164
	.long	745896919
	.long	3126591689
	.long	2378647625
	.long	3826286484
	.long	2774945552
	.long	4049104329
	.long	1491793838
	.long	1958216082
	.long	462327955
	.long	3357605672
	.long	1254923809
	.long	3803241362
	.long	2983587677
	.long	3916432164
	.long	924655910
	.long	2420244049
	.long	2509847619
	.long	3311515428
	.long	1672208059
	.long	3537897033
	.long	1849311821
	.long	545520802
	.long	724727943
	.long	2328063560
	.long	3344416119
	.long	2780826770
	.long	3698623643
	.long	1091041605
	.long	1449455886
	.long	361159825
	.long	2393864943
	.long	1266686244
	.long	3102279991
	.long	2182083211
	.long	2898911772
	.long	722319651
	.long	492762590
	.long	2533372489
	.long	1909592686
	.long	69199126
	.long	1502856249
	.long	1444639302
	.long	985525180
	.long	771777682
	.long	3819185373
	.long	138398252
	.long	3005712498
	.long	2889278605
	.long	1971050360
	.long	1543555365
	.long	3343403450
	.long	276796504
	.long	1716457700
	.long	1483589915
	.long	3942100721
	.long	3087110731
	.long	2391839604
	.long	553593009
	.long	3432915400
	.long	2967179831
	.long	3589234146
	.long	1879254167
	.long	488711913
	.long	1107186019
	.long	2570863504
	.long	1639392366
	.long	2883500997
	.long	3758508334
	.long	977423826
	.long	2214372039
	.long	846759712
	.long	3278784732
	.long	1472034698
	.long	3222049373
	.long	1954847653
	.long	133776782
	.long	1693519425
	.long	2262602168
	.long	2944069397
	.long	2149131451
	.long	3909695307
	.long	267553565
	.long	3387038850
	.long	230237041
	.long	1593171499
	.long	3295607
	.long	3524423319
	.long	535107130
	.long	2479110404
	.long	460474083
	.long	3186342998
	.long	6591214
	.long	2753879342
	.long	1070214261
	.long	663253512
	.long	920948167
	.long	2077718700
	.long	13182429
	.long	1212791388
	.long	2140428522
	.long	1326507024
	.long	1841896334
	.long	4155437400
	.long	26364858
	.long	2425582776
	.long	4280857045
	.long	2653014048
	.long	3683792669
	.long	4015907504
	.long	52729717
	.long	556198256
	.long	4266746795
	.long	1011060801
	.long	3072618042
	.long	3736847713
	.long	105459434
	.long	1112396512
	.long	4238526295
	.long	2022121603
	.long	1850268788
	.long	3178728131
	.long	210918868
	.long	2224793024
	.long	4182085295
	.long	4044243207
	.long	3700537577
	.long	2062488966
	.long	421837736
	.long	154618752
	.long	4069203294
	.long	3793519119
	.long	3106107858
	.long	4124977933
	.long	843675472
	.long	309237504
	.long	3843439293
	.long	3292070943
	.long	1917248420
	.long	3954988571
	.long	1687350944
	.long	618475008
	.long	3391911291
	.long	2289174591
	.long	3834496840
	.long	3615009846
	.long	3374701889
	.long	1236950016
	.long	2488855287
	.long	283381887
	.long	3374026384
	.long	2935052397
	.long	2454436482
	.long	2473900033
	.long	682743279
	.long	566763775
	.long	2453085473
	.long	1575137499
	.long	613905668
	.long	652832771
	.long	1365486558
	.long	1133527550
	.long	611203650
	.long	3150274999
	.long	1227811337
	.long	1305665542
	.long	2730973117
	.long	2267055100
	.long	1222407300
	.long	2005582702
	.long	2455622675
	.long	2611331084
	.long	1166978938
	.long	239142905
	.long	2444814601
	.long	4011165404
	.long	616278055
	.long	927694873
	.long	2333957877
	.long	478285810
	.long	594661906
	.long	3727363513
	.long	1232556110
	.long	1855389746
	.long	372948459
	.long	956571621
	.long	1189323812
	.long	3159759730
	.long	2465112221
	.long	3710779492
	.long	745896919
	.long	1913143242
	.long	2378647625
	.long	2024552164
	.long	635257146
	.long	3126591689
	.long	1491793838
	.long	3826286484
	.long	462327955
	.long	4049104329
	.long	1270514292
	.long	1958216082
	.long	2983587677
	.long	3357605672
	.long	924655910
	.long	3803241362
	.long	2541028584
	.long	3916432164
	.long	1672208059
	.long	2420244049
	.long	1849311821
	.long	3311515428
	.long	787089873
	.long	3537897033
	.long	3344416119
	.long	545520802
	.long	3698623643
	.long	2328063560
	.long	1574179747
	.long	2780826770
	.long	2393864943
	.long	1091041605
	.long	3102279991
	.long	361159825
	.long	3148359494
	.long	1266686244
	.long	492762590
	.long	2182083211
	.long	1909592686
	.long	722319651
	.long	2001751692
	.long	2533372489
	.long	985525180
	.long	69199126
	.long	3819185373
	.long	1444639302
	.long	4003503385
	.long	771777682
	.long	1971050360
	.long	138398252
	.long	3343403450
	.long	2889278605
	.long	3712039474
	.long	1543555365
	.long	3942100721
	.long	276796504
	.long	2391839604
	.long	1483589915
	.long	3129111652
	.long	3087110731
	.long	3589234146
	.long	553593009
	.long	488711913
	.long	2967179831
	.long	1963256009
	.long	1879254167
	.long	2883500997
	.long	1107186019
	.long	977423826
	.long	1639392366
	.long	3926512018
	.long	3758508334
	.long	1472034698
	.long	2214372039
	.long	1954847653
	.long	3278784732
	.long	3558056740
	.long	3222049373
	.long	2944069397
	.long	133776782
	.long	3909695307
	.long	2262602168
	.long	2821146184
	.long	2149131451
	.long	1593171499
	.long	267553565
	.long	3524423319
	.long	230237041
	.long	1347325072
	.long	3295607
	.long	3186342998
	.long	535107130
	.long	2753879342
	.long	460474083
	.long	2694650145
	.long	6591214
	.long	2077718700
	.long	1070214261
	.long	1212791388
	.long	920948167
	.long	1094332995
	.long	13182429
	.long	4155437400
	.long	2140428522
	.long	2425582776
	.long	1841896334
	.long	2188665991
	.long	26364858
	.long	4015907504
	.long	4280857045
	.long	556198256
	.long	3683792669
	.long	82364686
	.long	52729717
	.long	3736847713
	.long	4266746795
	.long	1112396512
	.long	3072618042
	.long	164729372
	.long	105459434
	.long	3178728131
	.long	4238526295
	.long	2224793024
	.long	1850268788
	.long	329458745
	.long	210918868
	.long	2062488966
	.long	4182085295
	.long	154618752
	.long	3700537577
	.long	658917491
	.long	421837736
	.long	4124977933
	.long	4069203294
	.long	309237504
	.long	3106107858
	.long	1317834983
	.long	843675472
	.long	3954988571
	.long	3843439293
	.long	618475008
	.long	1917248420
	.long	2635669967
	.long	1687350944
	.long	3615009846
	.long	3391911291
	.long	1236950016
	.long	3834496840
	.long	976372639
	.long	3374701889
	.long	2935052397
	.long	2488855287
	.long	2473900033
	.long	3374026384
	.long	1952745279
	.long	2454436482
	.long	1575137499
	.long	682743279
	.long	652832771
	.long	2453085473
	.long	3905490559
	.long	613905668
	.long	3150274999
	.long	1365486558
	.long	1305665542
	.long	611203650
	.long	3516013822
	.long	1227811337
	.long	2005582702
	.long	2730973117
	.long	2611331084
	.long	1222407300
	.long	2737060348
	.long	2455622675
	.long	4011165404
	.long	1166978938
	.long	927694873
	.long	2444814601
	.long	1179153400
	.long	616278055
	.long	3727363513
	.long	2333957877
	.long	1855389746
	.long	594661906
	.long	2358306800
	.long	1232556110
	.long	3159759730
	.long	372948459
	.long	3710779492
	.long	1189323812
	.long	421646305
	.long	2465112221
	.long	2024552164
	.long	745896919
	.long	3126591689
	.long	2378647625
	.long	843292611
	.long	635257146
	.long	4049104329
	.long	1491793838
	.long	1958216082
	.long	462327955
	.long	1686585223
	.long	1270514292
	.long	3803241362
	.long	2983587677
	.long	3916432164
	.long	924655910
	.long	3373170446
	.long	2541028584
	.long	3311515428
	.long	1672208059
	.long	3537897033
	.long	1849311821
	.long	2451373597
	.long	787089873
	.long	2328063560
	.long	3344416119
	.long	2780826770
	.long	3698623643
	.long	607779899
	.long	1574179747
	.long	361159825
	.long	2393864943
	.long	1266686244
	.long	3102279991
	.long	1215559799
	.long	3148359494
	.long	722319651
	.long	492762590
	.long	2533372489
	.long	1909592686
	.long	2431119599
	.long	2001751692
	.long	1444639302
	.long	985525180
	.long	771777682
	.long	3819185373
	.long	567271902
	.long	4003503385
	.long	2889278605
	.long	1971050360
	.long	1543555365
	.long	3343403450
	.long	1134543805
	.long	3712039474
	.long	1483589915
	.long	3942100721
	.long	3087110731
	.long	2391839604
	.long	2269087610
	.long	3129111652
	.long	2967179831
	.long	3589234146
	.long	1879254167
	.long	488711913
	.long	243207925
	.long	1963256009
	.long	1639392366
	.long	2883500997
	.long	3758508334
	.long	977423826
	.long	486415851
	.long	3926512018
	.long	3278784732
	.long	1472034698
	.long	3222049373
	.long	1954847653
	.long	972831702
	.long	3558056740
	.long	2262602168
	.long	2944069397
	.long	2149131451
	.long	3909695307
	.long	1945663404
	.long	2821146184
	.long	230237041
	.long	1593171499
	.long	3295607
	.long	3524423319
	.long	3891326808
	.long	1347325072
	.long	460474083
	.long	3186342998
	.long	6591214
	.long	2753879342
	.long	3487686321
	.long	2694650145
	.long	920948167
	.long	2077718700
	.long	13182429
	.long	1212791388
	.long	2680405347
	.long	1094332995
	.long	1841896334
	.long	4155437400
	.long	26364858
	.long	2425582776
	.long	1065843399
	.long	2188665991
	.long	3683792669
	.long	4015907504
	.long	52729717
	.long	556198256
	.long	2131686798
	.long	82364686
	.long	3072618042
	.long	3736847713
	.long	105459434
	.long	1112396512
	.long	4263373596
	.long	164729372
	.long	1850268788
	.long	3178728131
	.long	210918868
	.long	2224793024
	.long	4231779897
	.long	329458745
	.long	3700537577
	.long	2062488966
	.long	421837736
	.long	154618752
	.long	4168592498
	.long	658917491
	.long	3106107858
	.long	4124977933
	.long	843675472
	.long	309237504
	.long	4042217701
	.long	1317834983
	.long	1917248420
	.long	3954988571
	.long	1687350944
	.long	618475008
	.long	3789468107
	.long	2635669967
	.long	3834496840
	.long	3615009846
	.long	3374701889
	.long	1236950016
	.long	3283968918
	.long	976372639
	.long	3374026384
	.long	2935052397
	.long	2454436482
	.long	2473900033
	.long	2272970540
	.long	1952745279
	.long	2453085473
	.long	1575137499
	.long	613905668
	.long	652832771
	.long	250973784
	.long	3905490559
	.long	611203650
	.long	3150274999
	.long	1227811337
	.long	1305665542
	.long	501947569
	.long	3516013822
	.long	1222407300
	.long	2005582702
	.long	2455622675
	.long	2611331084
	.long	1003895138
	.long	2737060348
	.long	2444814601
	.long	4011165404
	.long	616278055
	.long	927694873
	.long	2007790276
	.long	1179153400
	.long	594661906
	.long	3727363513
	.long	1232556110
	.long	1855389746
	.long	4015580553
	.long	2358306800
	.long	1189323812
	.long	3159759730
	.long	2465112221
	.long	3710779492
	.long	3736193810
	.long	421646305
	.long	2378647625
	.long	2024552164
	.long	635257146
	.long	3126591689
	.long	3177420325
	.long	843292611
	.long	462327955
	.long	4049104329
	.long	1270514292
	.long	1958216082
	.long	2059873354
	.long	1686585223
	.long	924655910
	.long	3803241362
	.long	2541028584
	.long	3916432164
	.long	4119746708
	.long	3373170446
	.long	1849311821
	.long	3311515428
	.long	787089873
	.long	3537897033
	.long	3944526121
	.long	2451373597
	.long	3698623643
	.long	2328063560
	.long	1574179747
	.long	2780826770
	.long	3594084947
	.long	607779899
	.long	3102279991
	.long	361159825
	.long	3148359494
	.long	1266686244
	.long	2893202598
	.long	1215559799
	.long	1909592686
	.long	722319651
	.long	2001751692
	.long	2533372489
	.long	1491437901
	.long	2431119599
	.long	3819185373
	.long	1444639302
	.long	4003503385
	.long	771777682
	.long	2982875802
	.long	567271902
	.long	3343403450
	.long	2889278605
	.long	3712039474
	.long	1543555365
	.long	1670784308
	.long	1134543805
	.long	2391839604
	.long	1483589915
	.long	3129111652
	.long	3087110731
	.long	3341568617
	.long	2269087610
	.long	488711913
	.long	2967179831
	.long	1963256009
	.long	1879254167
	.long	2388169939
	.long	243207925
	.long	977423826
	.long	1639392366
	.long	3926512018
	.long	3758508334
	.long	481372583
	.long	486415851
	.long	1954847653
	.long	3278784732
	.long	3558056740
	.long	3222049373
	.long	962745166
	.long	972831702
	.long	3909695307
	.long	2262602168
	.long	2821146184
	.long	2149131451
	.long	1925490332
	.long	1945663404
	.long	3524423319
	.long	230237041
	.long	1347325072
	.long	3295607
	.long	3850980665
	.long	3891326808
	.long	2753879342
	.long	460474083
	.long	2694650145
	.long	6591214
	.long	3406994035
	.long	3487686321
	.long	1212791388
	.long	920948167
	.long	1094332995
	.long	13182429
	.long	2519020775
	.long	2680405347
	.long	2425582776
	.long	1841896334
	.long	2188665991
	.long	26364858
	.long	743074255
	.long	1065843399
	.long	556198256
	.long	3683792669
	.long	82364686
	.long	52729717
	.long	1486148511
	.long	2131686798
	.long	1112396512
	.long	3072618042
	.long	164729372
	.long	105459434
	.long	2972297022
	.long	4263373596
	.long	2224793024
	.long	1850268788
	.long	329458745
	.long	210918868
	.long	1649626749
	.long	4231779897
	.long	154618752
	.long	3700537577
	.long	658917491
	.long	421837736
	.long	3299253499
	.long	4168592498
	.long	309237504
	.long	3106107858
	.long	1317834983
	.long	843675472
	.long	2303539703
	.long	4042217701
	.long	618475008
	.long	1917248420
	.long	2635669967
	.long	1687350944
	.long	312112110
	.long	3789468107
	.long	1236950016
	.long	3834496840
	.long	976372639
	.long	3374701889
	.long	624224221
	.long	3283968918
	.long	2473900033
	.long	3374026384
	.long	1952745279
	.long	2454436482
	.long	1248448442
	.long	2272970540
	.long	652832771
	.long	2453085473
	.long	3905490559
	.long	613905668
	.long	2496896884
	.long	250973784
	.long	1305665542
	.long	611203650
	.long	3516013822
	.long	1227811337
	.long	698826472
	.long	501947569
	.long	2611331084
	.long	1222407300
	.long	2737060348
	.long	2455622675
	.long	1397652945
	.long	1003895138
	.long	927694873
	.long	2444814601
	.long	1179153400
	.long	616278055
	.long	2795305890
	.long	2007790276
	.long	1855389746
	.long	594661906
	.long	2358306800
	.long	1232556110
	.long	1295644484
	.long	4015580553
	.long	3710779492
	.long	1189323812
	.long	421646305
	.long	2465112221
	.long	2591288968
	.long	3736193810
	.long	3126591689
	.long	2378647625
	.long	843292611
	.long	635257146
	.long	887610640
	.long	3177420325
	.long	1958216082
	.long	462327955
	.long	1686585223
	.long	1270514292
	.long	1775221280
	.long	2059873354
	.long	3916432164
	.long	924655910
	.long	3373170446
	.long	2541028584
	.long	3550442561
	.long	4119746708
	.long	3537897033
	.long	1849311821
	.long	2451373597
	.long	787089873
	.long	2805917826
	.long	3944526121
	.long	2780826770
	.long	3698623643
	.long	607779899
	.long	1574179747
	.long	1316868356
	.long	3594084947
	.long	1266686244
	.long	3102279991
	.long	1215559799
	.long	3148359494
	.long	2633736712
	.long	2893202598
	.long	2533372489
	.long	1909592686
	.long	2431119599
	.long	2001751692
	.long	972506129
	.long	1491437901
	.long	771777682
	.long	3819185373
	.long	567271902
	.long	4003503385
	.long	1945012259
	.long	2982875802
	.long	1543555365
	.long	3343403450
	.long	1134543805
	.long	3712039474
	.long	3890024518
	.long	1670784308
	.long	3087110731
	.long	2391839604
	.long	2269087610
	.long	3129111652
	.long	3485081741
	.long	3341568617
	.long	1879254167
	.long	488711913
	.long	243207925
	.long	1963256009
	.long	2675196186
	.long	2388169939
	.long	3758508334
	.long	977423826
	.long	486415851
	.long	3926512018
	.long	1055425077
	.long	481372583
	.long	3222049373
	.long	1954847653
	.long	972831702
	.long	3558056740
	.long	2110850155
	.long	962745166
	.long	2149131451
	.long	3909695307
	.long	1945663404
	.long	2821146184
	.long	4221700311
	.long	1925490332
	.long	3295607
	.long	3524423319
	.long	3891326808
	.long	1347325072
	.long	4148433327
	.long	3850980665
	.long	6591214
	.long	2753879342
	.long	3487686321
	.long	2694650145
	.long	4001899359
	.long	3406994035
	.long	13182429
	.long	1212791388
	.long	2680405347
	.long	1094332995
	.long	3708831422
	.long	2519020775
	.long	26364858
	.long	2425582776
	.long	1065843399
	.long	2188665991
	.long	3122695549
	.long	743074255
	.long	52729717
	.long	556198256
	.long	2131686798
	.long	82364686
	.long	1950423802
	.long	1486148511
	.long	105459434
	.long	1112396512
	.long	4263373596
	.long	164729372
	.long	3900847605
	.long	2972297022
	.long	210918868
	.long	2224793024
	.long	4231779897
	.long	329458745
	.long	3506727914
	.long	1649626749
	.long	421837736
	.long	154618752
	.long	4168592498
	.long	658917491
	.long	2718488532
	.long	3299253499
	.long	843675472
	.long	309237504
	.long	4042217701
	.long	1317834983
	.long	1142009769
	.long	2303539703
	.long	1687350944
	.long	618475008
	.long	3789468107
	.long	2635669967
	.long	2284019538
	.long	312112110
	.long	3374701889
	.long	1236950016
	.long	3283968918
	.long	976372639
	.long	273071781
	.long	624224221
	.long	2454436482
	.long	2473900033
	.long	2272970540
	.long	1952745279
	.long	546143563
	.long	1248448442
	.long	613905668
	.long	652832771
	.long	250973784
	.long	3905490559
	.long	1092287127
	.long	2496896884
	.long	1227811337
	.long	1305665542
	.long	501947569
	.long	3516013822
	.long	2184574254
	.long	698826472
	.long	2455622675
	.long	2611331084
	.long	1003895138
	.long	2737060348
	.long	74181213
	.long	1397652945
	.long	616278055
	.long	927694873
	.long	2007790276
	.long	1179153400
	.long	148362426
	.long	2795305890
	.long	1232556110
	.long	1855389746
	.long	4015580553
	.long	2358306800
	.long	296724853
	.long	1295644484
	.long	2465112221
	.long	3710779492
	.long	3736193810
	.long	421646305
	.long	593449707
	.long	2591288968
	.long	635257146
	.long	3126591689
	.long	3177420325
	.long	843292611
	.long	1186899415
	.long	887610640
	.long	1270514292
	.long	1958216082
	.long	2059873354
	.long	1686585223
	.long	2373798830
	.long	1775221280
	.long	2541028584
	.long	3916432164
	.long	4119746708
	.long	3373170446
	.long	452630365
	.long	3550442561
	.long	787089873
	.long	3537897033
	.long	3944526121
	.long	2451373597
	.long	905260731
	.long	2805917826
	.long	1574179747
	.long	2780826770
	.long	3594084947
	.long	607779899
	.long	1810521462
	.long	1316868356
	.long	3148359494
	.long	1266686244
	.long	2893202598
	.long	1215559799
	.long	3621042925
	.long	2633736712
	.long	2001751692
	.long	2533372489
	.long	1491437901
	.long	2431119599
	.long	2947118554
	.long	972506129
	.long	4003503385
	.long	771777682
	.long	2982875802
	.long	567271902
	.long	1599269812
	.long	1945012259
	.long	3712039474
	.long	1543555365
	.long	1670784308
	.long	1134543805
	.long	3198539624
	.long	3890024518
	.long	3129111652
	.long	3087110731
	.long	3341568617
	.long	2269087610
	.long	2102111953
	.long	3485081741
	.long	1963256009
	.long	1879254167
	.long	2388169939
	.long	243207925
	.long	4204223906
	.long	2675196186
	.long	3926512018
	.long	3758508334
	.long	481372583
	.long	486415851
	.long	4113480516
	.long	1055425077
	.long	3558056740
	.long	3222049373
	.long	962745166
	.long	972831702
	.long	3931993737
	.long	2110850155
	.long	2821146184
	.long	2149131451
	.long	1925490332
	.long	1945663404
	.long	3569020178
	.long	4221700311
	.long	1347325072
	.long	3295607
	.long	3850980665
	.long	3891326808
	.long	2843073060
	.long	4148433327
	.long	2694650145
	.long	6591214
	.long	3406994035
	.long	3487686321
	.long	1391178824
	.long	4001899359
	.long	1094332995
	.long	13182429
	.long	2519020775
	.long	2680405347
	.long	2782357648
	.long	3708831422
	.long	2188665991
	.long	26364858
	.long	743074255
	.long	1065843399
	.long	1269748001
	.long	3122695549
	.long	82364686
	.long	52729717
	.long	1486148511
	.long	2131686798
	.long	2539496002
	.long	1950423802
	.long	164729372
	.long	105459434
	.long	2972297022
	.long	4263373596
	.long	784024708
	.long	3900847605
	.long	329458745
	.long	210918868
	.long	1649626749
	.long	4231779897
	.long	1568049417
	.long	3506727914
	.long	658917491
	.long	421837736
	.long	3299253499
	.long	4168592498
	.long	3136098835
	.long	2718488532
	.long	1317834983
	.long	843675472
	.long	2303539703
	.long	4042217701
	.long	1977230375
	.long	1142009769
	.long	2635669967
	.long	1687350944
	.long	312112110
	.long	3789468107
	.long	3954460750
	.long	2284019538
	.long	976372639
	.long	3374701889
	.long	624224221
	.long	3283968918
	.long	3613954205
	.long	273071781
	.long	1952745279
	.long	2454436482
	.long	1248448442
	.long	2272970540
	.long	2932941114
	.long	546143563
	.long	3905490559
	.long	613905668
	.long	2496896884
	.long	250973784
	.long	1570914932
	.long	1092287127
	.long	3516013822
	.long	1227811337
	.long	698826472
	.long	501947569
	.long	3141829865
	.long	2184574254
	.long	2737060348
	.long	2455622675
	.long	1397652945
	.long	1003895138
	.long	1988692435
	.long	74181213
	.long	1179153400
	.long	616278055
	.long	2795305890
	.long	2007790276
	.long	3977384870
	.long	148362426
	.long	2358306800
	.long	1232556110
	.long	1295644484
	.long	4015580553
	.long	3659802444
	.long	296724853
	.long	421646305
	.long	2465112221
	.long	2591288968
	.long	3736193810
	.long	3024637593
	.long	593449707
	.long	843292611
	.long	635257146
	.long	887610640
	.long	3177420325
	.long	1754307891
	.long	1186899415
	.long	1686585223
	.long	1270514292
	.long	1775221280
	.long	2059873354
	.long	3508615783
	.long	2373798830
	.long	3373170446
	.long	2541028584
	.long	3550442561
	.long	4119746708
	.long	2722264270
	.long	452630365
	.long	2451373597
	.long	787089873
	.long	2805917826
	.long	3944526121
	.long	1149561244
	.long	905260731
	.long	607779899
	.long	1574179747
	.long	1316868356
	.long	3594084947
	.long	2299122488
	.long	1810521462
	.long	1215559799
	.long	3148359494
	.long	2633736712
	.long	2893202598
	.long	303277681
	.long	3621042925
	.long	2431119599
	.long	2001751692
	.long	972506129
	.long	1491437901
	.long	606555363
	.long	2947118554
	.long	567271902
	.long	4003503385
	.long	1945012259
	.long	2982875802
	.long	1213110727
	.long	1599269812
	.long	1134543805
	.long	3712039474
	.long	3890024518
	.long	1670784308
	.long	2426221454
	.long	3198539624
	.long	2269087610
	.long	3129111652
	.long	3485081741
	.long	3341568617
	.long	557475612
	.long	2102111953
	.long	243207925
	.long	1963256009
	.long	2675196186
	.long	2388169939
	.long	1114951224
	.long	4204223906
	.long	486415851
	.long	3926512018
	.long	1055425077
	.long	481372583
	.long	2229902448
	.long	4113480516
	.long	972831702
	.long	3558056740
	.long	2110850155
	.long	962745166
	.long	164837600
	.long	3931993737
	.long	1945663404
	.long	2821146184
	.long	4221700311
	.long	1925490332
	.long	329675200
	.long	3569020178
	.long	3891326808
	.long	1347325072
	.long	4148433327
	.long	3850980665
	.long	659350401
	.long	2843073060
	.long	3487686321
	.long	2694650145
	.long	4001899359
	.long	3406994035
	.long	1318700802
	.long	1391178824
	.long	2680405347
	.long	1094332995
	.long	3708831422
	.long	2519020775
	.long	2637401604
	.long	2782357648
	.long	1065843399
	.long	2188665991
	.long	3122695549
	.long	743074255
	.long	979835913
	.long	1269748001
	.long	2131686798
	.long	82364686
	.long	1950423802
	.long	1486148511
	.long	1959671827
	.long	2539496002
	.long	4263373596
	.long	164729372
	.long	3900847605
	.long	2972297022
	.long	3919343654
	.long	784024708
	.long	4231779897
	.long	329458745
	.long	3506727914
	.long	1649626749
	.long	3543720013
	.long	1568049417
	.long	4168592498
	.long	658917491
	.long	2718488532
	.long	3299253499
	.long	2792472730
	.long	3136098835
	.long	4042217701
	.long	1317834983
	.long	1142009769
	.long	2303539703
	.long	1289978165
	.long	1977230375
	.long	3789468107
	.long	2635669967
	.long	2284019538
	.long	312112110
	.long	2579956331
	.long	3954460750
	.long	3283968918
	.long	976372639
	.long	273071781
	.long	624224221
	.long	864945366
	.long	3613954205
	.long	2272970540
	.long	1952745279
	.long	546143563
	.long	1248448442
	.long	1729890733
	.long	2932941114
	.long	250973784
	.long	3905490559
	.long	1092287127
	.long	2496896884
	.long	3459781466
	.long	1570914932
	.long	501947569
	.long	3516013822
	.long	2184574254
	.long	698826472
	.long	2624595636
	.long	3141829865
	.long	1003895138
	.long	2737060348
	.long	74181213
	.long	1397652945
	.long	954223976
	.long	1988692435
	.long	2007790276
	.long	1179153400
	.long	148362426
	.long	2795305890
	.long	1908447953
	.long	3977384870
	.long	4015580553
	.long	2358306800
	.long	296724853
	.long	1295644484
	.long	3816895906
	.long	3659802444
	.long	3736193810
	.long	421646305
	.long	593449707
	.long	2591288968
	.long	3338824517
	.long	3024637593
	.long	3177420325
	.long	843292611
	.long	1186899415
	.long	887610640
	.long	2382681739
	.long	1754307891
	.long	2059873354
	.long	1686585223
	.long	2373798830
	.long	1775221280
	.long	470396183
	.long	3508615783
	.long	4119746708
	.long	3373170446
	.long	452630365
	.long	3550442561
	.long	940792367
	.long	2722264270
	.long	3944526121
	.long	2451373597
	.long	905260731
	.long	2805917826
	.long	1881584735
	.long	1149561244
	.long	3594084947
	.long	607779899
	.long	1810521462
	.long	1316868356
	.long	3763169470
	.long	2299122488
	.long	2893202598
	.long	1215559799
	.long	3621042925
	.long	2633736712
	.long	3231371645
	.long	303277681
	.long	1491437901
	.long	2431119599
	.long	2947118554
	.long	972506129
	.long	2167775995
	.long	606555363
	.long	2982875802
	.long	567271902
	.long	1599269812
	.long	1945012259
	.long	40584695
	.long	1213110727
	.long	1670784308
	.long	1134543805
	.long	3198539624
	.long	3890024518
	.long	81169391
	.long	2426221454
	.long	3341568617
	.long	2269087610
	.long	2102111953
	.long	3485081741
	.long	162338783
	.long	557475612
	.long	2388169939
	.long	243207925
	.long	4204223906
	.long	2675196186
	.long	324677567
	.long	1114951224
	.long	481372583
	.long	486415851
	.long	4113480516
	.long	1055425077
	.long	649355134
	.long	2229902448
	.long	962745166
	.long	972831702
	.long	3931993737
	.long	2110850155
	.long	1298710268
	.long	164837600
	.long	1925490332
	.long	1945663404
	.long	3569020178
	.long	4221700311
	.long	2597420537
	.long	329675200
	.long	3850980665
	.long	3891326808
	.long	2843073060
	.long	4148433327
	.long	899873778
	.long	659350401
	.long	3406994035
	.long	3487686321
	.long	1391178824
	.long	4001899359
	.long	1799747556
	.long	1318700802
	.long	2519020775
	.long	2680405347
	.long	2782357648
	.long	3708831422
	.long	3599495112
	.long	2637401604
	.long	743074255
	.long	1065843399
	.long	1269748001
	.long	3122695549
	.long	2904022928
	.long	979835913
	.long	1486148511
	.long	2131686798
	.long	2539496002
	.long	1950423802
	.long	1513078560
	.long	1959671827
	.long	2972297022
	.long	4263373596
	.long	784024708
	.long	3900847605
	.long	3026157121
	.long	3919343654
	.long	1649626749
	.long	4231779897
	.long	1568049417
	.long	3506727914
	.long	1757346946
	.long	3543720013
	.long	3299253499
	.long	4168592498
	.long	3136098835
	.long	2718488532
	.long	3514693892
	.long	2792472730
	.long	2303539703
	.long	4042217701
	.long	1977230375
	.long	1142009769
	.long	2734420489
	.long	1289978165
	.long	312112110
	.long	3789468107
	.long	3954460750
	.long	2284019538
	.long	1173873683
	.long	2579956331
	.long	624224221
	.long	3283968918
	.long	3613954205
	.long	273071781
	.long	2347747367
	.long	864945366
	.long	1248448442
	.long	2272970540
	.long	2932941114
	.long	546143563
	.long	400527438
	.long	1729890733
	.long	2496896884
	.long	250973784
	.long	1570914932
	.long	1092287127
	.long	801054876
	.long	3459781466
	.long	698826472
	.long	501947569
	.long	3141829865
	.long	2184574254
	.long	1602109753
	.long	2624595636
	.long	1397652945
	.long	1003895138
	.long	1988692435
	.long	74181213
	.long	3204219507
	.long	954223976
	.long	2795305890
	.long	2007790276
	.long	3977384870
	.long	148362426
	.long	2113471718
	.long	1908447953
	.long	1295644484
	.long	4015580553
	.long	3659802444
	.long	296724853
	.long	4226943436
	.long	3816895906
	.long	2591288968
	.long	3736193810
	.long	3024637593
	.long	593449707
	.long	4158919577
	.long	3338824517
	.long	887610640
	.long	3177420325
	.long	1754307891
	.long	1186899415
	.long	4022871858
	.long	2382681739
	.long	1775221280
	.long	2059873354
	.long	3508615783
	.long	2373798830
	.long	3750776420
	.long	470396183
	.long	3550442561
	.long	4119746708
	.long	2722264270
	.long	452630365
	.long	3206585544
	.long	940792367
	.long	2805917826
	.long	3944526121
	.long	1149561244
	.long	905260731
	.long	2118203793
	.long	1881584735
	.long	1316868356
	.long	3594084947
	.long	2299122488
	.long	1810521462
	.long	4236407587
	.long	3763169470
	.long	2633736712
	.long	2893202598
	.long	303277681
	.long	3621042925
	.long	4177847879
	.long	3231371645
	.long	972506129
	.long	1491437901
	.long	606555363
	.long	2947118554
	.long	4060728462
	.long	2167775995
	.long	1945012259
	.long	2982875802
	.long	1213110727
	.long	1599269812
	.long	3826489629
	.long	40584695
	.long	3890024518
	.long	1670784308
	.long	2426221454
	.long	3198539624
	.long	3358011962
	.long	81169391
	.long	3485081741
	.long	3341568617
	.long	557475612
	.long	2102111953
	.long	2421056629
	.long	162338783
	.long	2675196186
	.long	2388169939
	.long	1114951224
	.long	4204223906
	.long	547145963
	.long	324677567
	.long	1055425077
	.long	481372583
	.long	2229902448
	.long	4113480516
	.long	1094291926
	.long	649355134
	.long	2110850155
	.long	962745166
	.long	164837600
	.long	3931993737
	.long	2188583852
	.long	1298710268
	.long	4221700311
	.long	1925490332
	.long	329675200
	.long	3569020178
	.long	82200408
	.long	2597420537
	.long	4148433327
	.long	3850980665
	.long	659350401
	.long	2843073060
	.long	164400817
	.long	899873778
	.long	4001899359
	.long	3406994035
	.long	1318700802
	.long	1391178824
	.long	328801635
	.long	1799747556
	.long	3708831422
	.long	2519020775
	.long	2637401604
	.long	2782357648
	.long	657603271
	.long	3599495112
	.long	3122695549
	.long	743074255
	.long	979835913
	.long	1269748001
	.long	1315206542
	.long	2904022928
	.long	1950423802
	.long	1486148511
	.long	1959671827
	.long	2539496002
	.long	2630413084
	.long	1513078560
	.long	3900847605
	.long	2972297022
	.long	3919343654
	.long	784024708
	.long	965858873
	.long	3026157121
	.long	3506727914
	.long	1649626749
	.long	3543720013
	.long	1568049417
	.long	1931717747
	.long	1757346946
	.long	2718488532
	.long	3299253499
	.long	2792472730
	.long	3136098835
	.long	3863435494
	.long	3514693892
	.long	1142009769
	.long	2303539703
	.long	1289978165
	.long	1977230375
	.long	3431903692
	.long	2734420489
	.long	2284019538
	.long	312112110
	.long	2579956331
	.long	3954460750
	.long	2568840088
	.long	1173873683
	.long	273071781
	.long	624224221
	.long	864945366
	.long	3613954205
	.long	842712880
	.long	2347747367
	.long	546143563
	.long	1248448442
	.long	1729890733
	.long	2932941114
	.long	1685425760
	.long	400527438
	.long	1092287127
	.long	2496896884
	.long	3459781466
	.long	1570914932
	.long	3370851521
	.long	801054876
	.long	2184574254
	.long	698826472
	.long	2624595636
	.long	3141829865
	.long	2446735747
	.long	1602109753
	.long	74181213
	.long	1397652945
	.long	954223976
	.long	1988692435
	.long	598504198
	.long	3204219507
	.long	148362426
	.long	2795305890
	.long	1908447953
	.long	3977384870
	.long	1197008397
	.long	2113471718
	.long	296724853
	.long	1295644484
	.long	3816895906
	.long	3659802444
	.long	2394016794
	.long	4226943436
	.long	593449707
	.long	2591288968
	.long	3338824517
	.long	3024637593
	.long	493066293
	.long	4158919577
	.long	1186899415
	.long	887610640
	.long	2382681739
	.long	1754307891
	.long	986132586
	.long	4022871858
	.long	2373798830
	.long	1775221280
	.long	470396183
	.long	3508615783
	.long	1972265172
	.long	3750776420
	.long	452630365
	.long	3550442561
	.long	940792367
	.long	2722264270
	.long	3944530345
	.long	3206585544
	.long	905260731
	.long	2805917826
	.long	1881584735
	.long	1149561244
	.long	3594093395
	.long	2118203793
	.long	1810521462
	.long	1316868356
	.long	3763169470
	.long	2299122488
	.long	2893219494
	.long	4236407587
	.long	3621042925
	.long	2633736712
	.long	3231371645
	.long	303277681
	.long	1491471692
	.long	4177847879
	.long	2947118554
	.long	972506129
	.long	2167775995
	.long	606555363
	.long	2982943385
	.long	4060728462
	.long	1599269812
	.long	1945012259
	.long	40584695
	.long	1213110727
	.long	1670919475
	.long	3826489629
	.long	3198539624
	.long	3890024518
	.long	81169391
	.long	2426221454
	.long	3341838951
	.long	3358011962
	.long	2102111953
	.long	3485081741
	.long	162338783
	.long	557475612
	.long	2388710606
	.long	2421056629
	.long	4204223906
	.long	2675196186
	.long	324677567
	.long	1114951224
	.long	482453916
	.long	547145963
	.long	4113480516
	.long	1055425077
	.long	649355134
	.long	2229902448
	.long	964907833
	.long	1094291926
	.long	3931993737
	.long	2110850155
	.long	1298710268
	.long	164837600
	.long	1929815667
	.long	2188583852
	.long	3569020178
	.long	4221700311
	.long	2597420537
	.long	329675200
	.long	3859631335
	.long	82200408
	.long	2843073060
	.long	4148433327
	.long	899873778
	.long	659350401
	.long	3424295375
	.long	164400817
	.long	1391178824
	.long	4001899359
	.long	1799747556
	.long	1318700802
	.long	2553623455
	.long	328801635
	.long	2782357648
	.long	3708831422
	.long	3599495112
	.long	2637401604
	.long	812279614
	.long	657603271
	.long	1269748001
	.long	3122695549
	.long	2904022928
	.long	979835913
	.long	1624559229
	.long	1315206542
	.long	2539496002
	.long	1950423802
	.long	1513078560
	.long	1959671827
	.long	3249118458
	.long	2630413084
	.long	784024708
	.long	3900847605
	.long	3026157121
	.long	3919343654
	.long	2203269620
	.long	965858873
	.long	1568049417
	.long	3506727914
	.long	1757346946
	.long	3543720013
	.long	111571945
	.long	1931717747
	.long	3136098835
	.long	2718488532
	.long	3514693892
	.long	2792472730
	.long	223143890
	.long	3863435494
	.long	1977230375
	.long	1142009769
	.long	2734420489
	.long	1289978165
	.long	446287780
	.long	3431903692
	.long	3954460750
	.long	2284019538
	.long	1173873683
	.long	2579956331
	.long	892575561
	.long	2568840088
	.long	3613954205
	.long	273071781
	.long	2347747367
	.long	864945366
	.long	1785151123
	.long	842712880
	.long	2932941114
	.long	546143563
	.long	400527438
	.long	1729890733
	.long	3570302247
	.long	1685425760
	.long	1570914932
	.long	1092287127
	.long	801054876
	.long	3459781466
	.long	2845637198
	.long	3370851521
	.long	3141829865
	.long	2184574254
	.long	1602109753
	.long	2624595636
	.long	1396307100
	.long	2446735747
	.long	1988692435
	.long	74181213
	.long	3204219507
	.long	954223976
	.long	2792614201
	.long	598504198
	.long	3977384870
	.long	148362426
	.long	2113471718
	.long	1908447953
	.long	1290261106
	.long	1197008397
	.long	3659802444
	.long	296724853
	.long	4226943436
	.long	3816895906
	.long	2580522212
	.long	2394016794
	.long	3024637593
	.long	593449707
	.long	4158919577
	.long	3338824517
	.long	866077128
	.long	493066293
	.long	1754307891
	.long	1186899415
	.long	4022871858
	.long	2382681739
	.long	1732154256
	.long	986132586
	.long	3508615783
	.long	2373798830
	.long	3750776420
	.long	470396183
	.long	3464308513
	.long	1972265172
	.long	2722264270
	.long	452630365
	.long	3206585544
	.long	940792367
	.long	2633649730
	.long	3944530345
	.long	1149561244
	.long	905260731
	.long	2118203793
	.long	1881584735
	.long	972332164
	.long	3594093395
	.long	2299122488
	.long	1810521462
	.long	4236407587
	.long	3763169470
	.long	1944664328
	.long	2893219494
	.long	303277681
	.long	3621042925
	.long	4177847879
	.long	3231371645
	.long	3889328657
	.long	1491471692
	.long	606555363
	.long	2947118554
	.long	4060728462
	.long	2167775995
	.long	3483690018
	.long	2982943385
	.long	1213110727
	.long	1599269812
	.long	3826489629
	.long	40584695
	.long	2672412741
	.long	1670919475
	.long	2426221454
	.long	3198539624
	.long	3358011962
	.long	81169391
	.long	1049858187
	.long	3341838951
	.long	557475612
	.long	2102111953
	.long	2421056629
	.long	162338783
	.long	2099716375
	.long	2388710606
	.long	1114951224
	.long	4204223906
	.long	547145963
	.long	324677567
	.long	4199432751
	.long	482453916
	.long	2229902448
	.long	4113480516
	.long	1094291926
	.long	649355134
	.long	4103898207
	.long	964907833
	.long	164837600
	.long	3931993737
	.long	2188583852
	.long	1298710268
	.long	3912829119
	.long	1929815667
	.long	329675200
	.long	3569020178
	.long	82200408
	.long	2597420537
	.long	3530690942
	.long	3859631335
	.long	659350401
	.long	2843073060
	.long	164400817
	.long	899873778
	.long	2766414588
	.long	3424295375
	.long	1318700802
	.long	1391178824
	.long	328801635
	.long	1799747556
	.long	1237861880
	.long	2553623455
	.long	2637401604
	.long	2782357648
	.long	657603271
	.long	3599495112
	.long	2475723761
	.long	812279614
	.long	979835913
	.long	1269748001
	.long	1315206542
	.long	2904022928
	.long	656480226
	.long	1624559229
	.long	1959671827
	.long	2539496002
	.long	2630413084
	.long	1513078560
	.long	1312960453
	.long	3249118458
	.long	3919343654
	.long	784024708
	.long	965858873
	.long	3026157121
	.long	2625920907
	.long	2203269620
	.long	3543720013
	.long	1568049417
	.long	1931717747
	.long	1757346946
	.long	956874519
	.long	111571945
	.long	2792472730
	.long	3136098835
	.long	3863435494
	.long	3514693892
	.long	1913749038
	.long	223143890
	.long	1289978165
	.long	1977230375
	.long	3431903692
	.long	2734420489
	.long	3827498077
	.long	446287780
	.long	2579956331
	.long	3954460750
	.long	2568840088
	.long	1173873683
	.long	3360028859
	.long	892575561
	.long	864945366
	.long	3613954205
	.long	842712880
	.long	2347747367
	.long	2425090423
	.long	1785151123
	.long	1729890733
	.long	2932941114
	.long	1685425760
	.long	400527438
	.long	555213551
	.long	3570302247
	.long	3459781466
	.long	1570914932
	.long	3370851521
	.long	801054876
	.long	1110427102
	.long	2845637198
	.long	2624595636
	.long	3141829865
	.long	2446735747
	.long	1602109753
	.long	2220854205
	.long	1396307100
	.long	954223976
	.long	1988692435
	.long	598504198
	.long	3204219507
	.long	146741115
	.long	2792614201
	.long	1908447953
	.long	3977384870
	.long	1197008397
	.long	2113471718
	.long	293482231
	.long	1290261106
	.long	3816895906
	.long	3659802444
	.long	2394016794
	.long	4226943436
	.long	586964463
	.long	2580522212
	.long	3338824517
	.long	3024637593
	.long	493066293
	.long	4158919577
	.long	1173928927
	.long	866077128
	.long	2382681739
	.long	1754307891
	.long	986132586
	.long	4022871858
	.long	2347857855
	.long	1732154256
	.long	470396183
	.long	3508615783
	.long	1972265172
	.long	3750776420
	.long	400748414
	.long	3464308513
	.long	940792367
	.long	2722264270
	.long	3944530345
	.long	3206585544
	.long	801496828
	.long	2633649730
	.long	1881584735
	.long	1149561244
	.long	3594093395
	.long	2118203793
	.long	1602993657
	.long	972332164
	.long	3763169470
	.long	2299122488
	.long	2893219494
	.long	4236407587
	.long	3205987314
	.long	1944664328
	.long	3231371645
	.long	303277681
	.long	1491471692
	.long	4177847879
	.long	2117007332
	.long	3889328657
	.long	2167775995
	.long	606555363
	.long	2982943385
	.long	4060728462
	.long	4234014665
	.long	3483690018
	.long	40584695
	.long	1213110727
	.long	1670919475
	.long	3826489629
	.long	4173062034
	.long	2672412741
	.long	81169391
	.long	2426221454
	.long	3341838951
	.long	3358011962
	.long	4051156773
	.long	1049858187
	.long	162338783
	.long	557475612
	.long	2388710606
	.long	2421056629
	.long	3807346250
	.long	2099716375
	.long	324677567
	.long	1114951224
	.long	482453916
	.long	547145963
	.long	3319725204
	.long	4199432751
	.long	649355134
	.long	2229902448
	.long	964907833
	.long	1094291926
	.long	2344483112
	.long	4103898207
	.long	1298710268
	.long	164837600
	.long	1929815667
	.long	2188583852
	.long	393998928
	.long	3912829119
	.long	2597420537
	.long	329675200
	.long	3859631335
	.long	82200408
	.long	787997856
	.long	3530690942
	.long	899873778
	.long	659350401
	.long	3424295375
	.long	164400817
	.long	1575995713
	.long	2766414588
	.long	1799747556
	.long	1318700802
	.long	2553623455
	.long	328801635
	.long	3151991427
	.long	1237861880
	.long	3599495112
	.long	2637401604
	.long	812279614
	.long	657603271
	.long	2009015559
	.long	2475723761
	.long	2904022928
	.long	979835913
	.long	1624559229
	.long	1315206542
	.long	4018031118
	.long	656480226
	.long	1513078560
	.long	1959671827
	.long	3249118458
	.long	2630413084
	.long	3741094941
	.long	1312960453
	.long	3026157121
	.long	3919343654
	.long	2203269620
	.long	965858873
	.long	3187222587
	.long	2625920907
	.long	1757346946
	.long	3543720013
	.long	111571945
	.long	1931717747
	.long	2079477878
	.long	956874519
	.long	3514693892
	.long	2792472730
	.long	223143890
	.long	3863435494
	.long	4158955756
	.long	1913749038
	.long	2734420489
	.long	1289978165
	.long	446287780
	.long	3431903692
	.long	4022944216
	.long	3827498077
	.long	1173873683
	.long	2579956331
	.long	892575561
	.long	2568840088
	.long	3750921137
	.long	3360028859
	.long	2347747367
	.long	864945366
	.long	1785151123
	.long	842712880
	.long	3206874979
	.long	2425090423
	.long	400527438
	.long	1729890733
	.long	3570302247
	.long	1685425760
	.long	2118782663
	.long	555213551
	.long	801054876
	.long	3459781466
	.long	2845637198
	.long	3370851521
	.long	4237565327
	.long	1110427102
	.long	1602109753
	.long	2624595636
	.long	1396307100
	.long	2446735747
	.long	4180163359
	.long	2220854205
	.long	3204219507
	.long	954223976
	.long	2792614201
	.long	598504198
	.long	4065359423
	.long	146741115
	.long	2113471718
	.long	1908447953
	.long	1290261106
	.long	1197008397
	.long	3835751551
	.long	293482231
	.long	4226943436
	.long	3816895906
	.long	2580522212
	.long	2394016794
	.long	3376535807
	.long	586964463
	.long	4158919577
	.long	3338824517
	.long	866077128
	.long	493066293
	.long	2458104319
	.long	1173928927
	.long	4022871858
	.long	2382681739
	.long	1732154256
	.long	986132586
	.long	621241343
	.long	2347857855
	.long	3750776420
	.long	470396183
	.long	3464308513
	.long	1972265172
	.long	1242482686
	.long	400748414
	.long	3206585544
	.long	940792367
	.long	2633649730
	.long	3944530345
	.long	2484965372
	.long	801496828
	.long	2118203793
	.long	1881584735
	.long	972332164
	.long	3594093395
	.long	674963448
	.long	1602993657
	.long	4236407587
	.long	3763169470
	.long	1944664328
	.long	2893219494
	.long	1349926897
	.long	3205987314
	.long	4177847879
	.long	3231371645
	.long	3889328657
	.long	1491471692
	.long	2699853794
	.long	2117007332
	.long	4060728462
	.long	2167775995
	.long	3483690018
	.long	2982943385
	.long	1104740292
	.long	4234014665
	.long	3826489629
	.long	40584695
	.long	2672412741
	.long	1670919475
	.long	2209480585
	.long	4173062034
	.long	3358011962
	.long	81169391
	.long	1049858187
	.long	3341838951
	.long	123993874
	.long	4051156773
	.long	2421056629
	.long	162338783
	.long	2099716375
	.long	2388710606
	.long	247987749
	.long	3807346250
	.long	547145963
	.long	324677567
	.long	4199432751
	.long	482453916
	.long	495975499
	.long	3319725204
	.long	1094291926
	.long	649355134
	.long	4103898207
	.long	964907833
	.long	991950999
	.long	2344483112
	.long	2188583852
	.long	1298710268
	.long	3912829119
	.long	1929815667
	.long	1983901999
	.long	393998928
	.long	82200408
	.long	2597420537
	.long	3530690942
	.long	3859631335
	.long	3967803999
	.long	787997856
	.long	164400817
	.long	899873778
	.long	2766414588
	.long	3424295375
	.long	3640640703
	.long	1575995713
	.long	328801635
	.long	1799747556
	.long	1237861880
	.long	2553623455
	.long	2986314111
	.long	3151991427
	.long	657603271
	.long	3599495112
	.long	2475723761
	.long	812279614
	.long	1677660927
	.long	2009015559
	.long	1315206542
	.long	2904022928
	.long	656480226
	.long	1624559229
	.long	3355321855
	.long	4018031118
	.long	2630413084
	.long	1513078560
	.long	1312960453
	.long	3249118458
	.long	2415676415
	.long	3741094941
	.long	965858873
	.long	3026157121
	.long	2625920907
	.long	2203269620
	.long	536385535
	.long	3187222587
	.long	1931717747
	.long	1757346946
	.long	956874519
	.long	111571945
	.long	1072771071
	.long	2079477878
	.long	3863435494
	.long	3514693892
	.long	1913749038
	.long	223143890
	.long	2145542143
	.long	4158955756
	.long	3431903692
	.long	2734420489
	.long	3827498077
	.long	446287780
	.long	4291084286
	.long	4022944216
	.long	2568840088
	.long	1173873683
	.long	3360028859
	.long	892575561
	.long	4287201277
	.long	3750921137
	.long	842712880
	.long	2347747367
	.long	2425090423
	.long	1785151123
	.long	4279435259
	.long	3206874979
	.long	1685425760
	.long	400527438
	.long	555213551
	.long	3570302247
	.long	4263903223
	.long	2118782663
	.long	3370851521
	.long	801054876
	.long	1110427102
	.long	2845637198
	.long	4232839151
	.long	4237565327
	.long	2446735747
	.long	1602109753
	.long	2220854205
	.long	1396307100
	.long	4170711006
	.long	4180163359
	.long	598504198
	.long	3204219507
	.long	146741115
	.long	2792614201
	.long	4046454716
	.long	4065359423
	.long	1197008397
	.long	2113471718
	.long	293482231
	.long	1290261106
	.long	3797942136
	.long	3835751551
	.long	2394016794
	.long	4226943436
	.long	586964463
	.long	2580522212
	.long	3300916976
	.long	3376535807
	.long	493066293
	.long	4158919577
	.long	1173928927
	.long	866077128
	.long	2306866656
	.long	2458104319
	.long	986132586
	.long	4022871858
	.long	2347857855
	.long	1732154256
	.long	318766016
	.long	621241343
	.long	1972265172
	.long	3750776420
	.long	400748414
	.long	3464308513
	.long	637532033
	.long	1242482686
	.long	3944530345
	.long	3206585544
	.long	801496828
	.long	2633649730
	.long	1275064066
	.long	2484965372
	.long	3594093395
	.long	2118203793
	.long	1602993657
	.long	972332164
	.long	2550128133
	.long	674963448
	.long	2893219494
	.long	4236407587
	.long	3205987314
	.long	1944664328
	.long	805288971
	.long	1349926897
	.long	1491471692
	.long	4177847879
	.long	2117007332
	.long	3889328657
	.long	1610577942
	.long	2699853794
	.long	2982943385
	.long	4060728462
	.long	4234014665
	.long	3483690018
	.long	3221155884
	.long	1104740292
	.long	1670919475
	.long	3826489629
	.long	4173062034
	.long	2672412741
	.long	2147344473
	.long	2209480585
	.long	3341838951
	.long	3358011962
	.long	4051156773
	.long	1049858187
	.long	4294688947
	.long	123993874
	.long	2388710606
	.long	2421056629
	.long	3807346250
	.long	2099716375
	.long	4294410598
	.long	247987749
	.long	482453916
	.long	547145963
	.long	3319725204
	.long	4199432751
	.long	4293853900
	.long	495975499
	.long	964907833
	.long	1094291926
	.long	2344483112
	.long	4103898207
	.long	4292740504
	.long	991950999
	.long	1929815667
	.long	2188583852
	.long	393998928
	.long	3912829119
	.long	4290513712
	.long	1983901999
	.long	3859631335
	.long	82200408
	.long	787997856
	.long	3530690942
	.long	4286060128
	.long	3967803999
	.long	3424295375
	.long	164400817
	.long	1575995713
	.long	2766414588
	.long	4277152960
	.long	3640640703
	.long	2553623455
	.long	328801635
	.long	3151991427
	.long	1237861880
	.long	4259338624
	.long	2986314111
	.long	812279614
	.long	657603271
	.long	2009015559
	.long	2475723761
	.long	4223709953
	.long	1677660927
	.long	1624559229
	.long	1315206542
	.long	4018031118
	.long	656480226
	.long	4152452611
	.long	3355321855
	.long	3249118458
	.long	2630413084
	.long	3741094941
	.long	1312960453
	.long	4009937927
	.long	2415676415
	.long	2203269620
	.long	965858873
	.long	3187222587
	.long	2625920907
	.long	3724908559
	.long	536385535
	.long	111571945
	.long	1931717747
	.long	2079477878
	.long	956874519
	.long	3154849823
	.long	1072771071
	.long	223143890
	.long	3863435494
	.long	4158955756
	.long	1913749038
	.long	2014732351
	.long	2145542143
	.long	446287780
	.long	3431903692
	.long	4022944216
	.long	3827498077
	.long	4029464703
	.long	4291084286
	.long	892575561
	.long	2568840088
	.long	3750921137
	.long	3360028859
	.long	3763962110
	.long	4287201277
	.long	1785151123
	.long	842712880
	.long	3206874979
	.long	2425090423
	.long	3232956925
	.long	4279435259
	.long	3570302247
	.long	1685425760
	.long	2118782663
	.long	555213551
	.long	2170946555
	.long	4263903223
	.long	2845637198
	.long	3370851521
	.long	4237565327
	.long	1110427102
	.long	46925815
	.long	4232839151
	.long	1396307100
	.long	2446735747
	.long	4180163359
	.long	2220854205
	.long	93851631
	.long	4170711006
	.long	2792614201
	.long	598504198
	.long	4065359423
	.long	146741115
	.long	187703262
	.long	4046454716
	.long	1290261106
	.long	1197008397
	.long	3835751551
	.long	293482231
	.long	375406524
	.long	3797942136
	.long	2580522212
	.long	2394016794
	.long	3376535807
	.long	586964463
	.long	750813049
	.long	3300916976
	.long	866077128
	.long	493066293
	.long	2458104319
	.long	1173928927
	.long	1501626098
	.long	2306866656
	.long	1732154256
	.long	986132586
	.long	621241343
	.long	2347857855
	.long	3003252197
	.long	318766016
	.long	3464308513
	.long	1972265172
	.long	1242482686
	.long	400748414
	.long	1711537099
	.long	637532033
	.long	2633649730
	.long	3944530345
	.long	2484965372
	.long	801496828
	.long	3423074199
	.long	1275064066
	.long	972332164
	.long	3594093395
	.long	674963448
	.long	1602993657
	.long	2551181103
	.long	2550128133
	.long	1944664328
	.long	2893219494
	.long	1349926897
	.long	3205987314
	.long	807394910
	.long	805288971
	.long	3889328657
	.long	1491471692
	.long	2699853794
	.long	2117007332
	.long	1614789820
	.long	1610577942
	.long	3483690018
	.long	2982943385
	.long	1104740292
	.long	4234014665
	.long	3229579640
	.long	3221155884
	.long	2672412741
	.long	1670919475
	.long	2209480585
	.long	4173062034
	.long	2164191985
	.long	2147344473
	.long	1049858187
	.long	3341838951
	.long	123993874
	.long	4051156773
	.long	33416674
	.long	4294688947
	.long	2099716375
	.long	2388710606
	.long	247987749
	.long	3807346250
	.long	66833348
	.long	4294410598
	.long	4199432751
	.long	482453916
	.long	495975499
	.long	3319725204
	.long	133666696
	.long	4293853900
	.long	4103898207
	.long	964907833
	.long	991950999
	.long	2344483112
	.long	267333393
	.long	4292740504
	.long	3912829119
	.long	1929815667
	.long	1983901999
	.long	393998928
	.long	534666787
	.long	4290513712
	.long	3530690942
	.long	3859631335
	.long	3967803999
	.long	787997856
	.long	1069333574
	.long	4286060128
	.long	2766414588
	.long	3424295375
	.long	3640640703
	.long	1575995713
	.long	2138667148
	.long	4277152960
	.long	1237861880
	.long	2553623455
	.long	2986314111
	.long	3151991427
	.long	4277334296
	.long	4259338624
	.long	2475723761
	.long	812279614
	.long	1677660927
	.long	2009015559
	.long	4259701297
	.long	4223709953
	.long	656480226
	.long	1624559229
	.long	3355321855
	.long	4018031118
	.long	4224435298
	.long	4152452611
	.long	1312960453
	.long	3249118458
	.long	2415676415
	.long	3741094941
	.long	4153903301
	.long	4009937927
	.long	2625920907
	.long	2203269620
	.long	536385535
	.long	3187222587
	.long	4012839307
	.long	3724908559
	.long	956874519
	.long	111571945
	.long	1072771071
	.long	2079477878
	.long	3730711318
	.long	3154849823
	.long	1913749038
	.long	223143890
	.long	2145542143
	.long	4158955756
	.long	3166455341
	.long	2014732351
	.long	3827498077
	.long	446287780
	.long	4291084286
	.long	4022944216
	.long	2037943386
	.long	4029464703
	.long	3360028859
	.long	892575561
	.long	4287201277
	.long	3750921137
	.long	4075886773
	.long	3763962110
	.long	2425090423
	.long	1785151123
	.long	4279435259
	.long	3206874979
	.long	3856806251
	.long	3232956925
	.long	555213551
	.long	3570302247
	.long	4263903223
	.long	2118782663
	.long	3418645206
	.long	2170946555
	.long	1110427102
	.long	2845637198
	.long	4232839151
	.long	4237565327
	.long	2542323117
	.long	46925815
	.long	2220854205
	.long	1396307100
	.long	4170711006
	.long	4180163359
	.long	789678938
	.long	93851631
	.long	146741115
	.long	2792614201
	.long	4046454716
	.long	4065359423
	.long	1579357876
	.long	187703262
	.long	293482231
	.long	1290261106
	.long	3797942136
	.long	3835751551
	.long	3158715752
	.long	375406524
	.long	586964463
	.long	2580522212
	.long	3300916976
	.long	3376535807
	.long	2022464208
	.long	750813049
	.long	1173928927
	.long	866077128
	.long	2306866656
	.long	2458104319
	.long	4044928416
	.long	1501626098
	.long	2347857855
	.long	1732154256
	.long	318766016
	.long	621241343
	.long	3794889537
	.long	3003252197
	.long	400748414
	.long	3464308513
	.long	637532033
	.long	1242482686
	.long	3294811778
	.long	1711537099
	.long	801496828
	.long	2633649730
	.long	1275064066
	.long	2484965372
	.long	2294656261
	.long	3423074199
	.long	1602993657
	.long	972332164
	.long	2550128133
	.long	674963448
	.long	294345226
	.long	2551181103
	.long	3205987314
	.long	1944664328
	.long	805288971
	.long	1349926897
	.long	588690452
	.long	807394910
	.long	2117007332
	.long	3889328657
	.long	1610577942
	.long	2699853794
	.long	1177380905
	.long	1614789820
	.long	4234014665
	.long	3483690018
	.long	3221155884
	.long	1104740292
	.long	2354761811
	.long	3229579640
	.long	4173062034
	.long	2672412741
	.long	2147344473
	.long	2209480585
	.long	414556326
	.long	2164191985
	.long	4051156773
	.long	1049858187
	.long	4294688947
	.long	123993874
	.long	829112653
	.long	33416674
	.long	3807346250
	.long	2099716375
	.long	4294410598
	.long	247987749
	.long	1658225307
	.long	66833348
	.long	3319725204
	.long	4199432751
	.long	4293853900
	.long	495975499
	.long	3316450614
	.long	133666696
	.long	2344483112
	.long	4103898207
	.long	4292740504
	.long	991950999
	.long	2337933933
	.long	267333393
	.long	393998928
	.long	3912829119
	.long	4290513712
	.long	1983901999
	.long	380900570
	.long	534666787
	.long	787997856
	.long	3530690942
	.long	4286060128
	.long	3967803999
	.long	761801140
	.long	1069333574
	.long	1575995713
	.long	2766414588
	.long	4277152960
	.long	3640640703
	.long	1523602280
	.long	2138667148
	.long	3151991427
	.long	1237861880
	.long	4259338624
	.long	2986314111
	.long	3047204561
	.long	4277334296
	.long	2009015559
	.long	2475723761
	.long	4223709953
	.long	1677660927
	.long	1799441827
	.long	4259701297
	.long	4018031118
	.long	656480226
	.long	4152452611
	.long	3355321855
	.long	3598883655
	.long	4224435298
	.long	3741094941
	.long	1312960453
	.long	4009937927
	.long	2415676415
	.long	2902800015
	.long	4153903301
	.long	3187222587
	.long	2625920907
	.long	3724908559
	.long	536385535
	.long	1510632735
	.long	4012839307
	.long	2079477878
	.long	956874519
	.long	3154849823
	.long	1072771071
	.long	3021265470
	.long	3730711318
	.long	4158955756
	.long	1913749038
	.long	2014732351
	.long	2145542143
	.long	1747563645
	.long	3166455341
	.long	4022944216
	.long	3827498077
	.long	4029464703
	.long	4291084286
	.long	3495127291
	.long	2037943386
	.long	3750921137
	.long	3360028859
	.long	3763962110
	.long	4287201277
	.long	2695287286
	.long	4075886773
	.long	3206874979
	.long	2425090423
	.long	3232956925
	.long	4279435259
	.long	1095607277
	.long	3856806251
	.long	2118782663
	.long	555213551
	.long	2170946555
	.long	4263903223
	.long	2191214555
	.long	3418645206
	.long	4237565327
	.long	1110427102
	.long	46925815
	.long	4232839151
	.long	87461814
	.long	2542323117
	.long	4180163359
	.long	2220854205
	.long	93851631
	.long	4170711006
	.long	174923629
	.long	789678938
	.long	4065359423
	.long	146741115
	.long	187703262
	.long	4046454716
	.long	349847258
	.long	1579357876
	.long	3835751551
	.long	293482231
	.long	375406524
	.long	3797942136
	.long	699694516
	.long	3158715752
	.long	3376535807
	.long	586964463
	.long	750813049
	.long	3300916976
	.long	1399389033
	.long	2022464208
	.long	2458104319
	.long	1173928927
	.long	1501626098
	.long	2306866656
	.long	2798778067
	.long	4044928416
	.long	621241343
	.long	2347857855
	.long	3003252197
	.long	318766016
	.long	1302588838
	.long	3794889537
	.long	1242482686
	.long	400748414
	.long	1711537099
	.long	637532033
	.long	2605177677
	.long	3294811778
	.long	2484965372
	.long	801496828
	.long	3423074199
	.long	1275064066
	.long	915388059
	.long	2294656261
	.long	674963448
	.long	1602993657
	.long	2551181103
	.long	2550128133
	.long	1830776118
	.long	294345226
	.long	1349926897
	.long	3205987314
	.long	807394910
	.long	805288971
	.long	3661552236
	.long	588690452
	.long	2699853794
	.long	2117007332
	.long	1614789820
	.long	1610577942
	.long	3028137177
	.long	1177380905
	.long	1104740292
	.long	4234014665
	.long	3229579640
	.long	3221155884
	.long	1761307059
	.long	2354761811
	.long	2209480585
	.long	4173062034
	.long	2164191985
	.long	2147344473
	.long	3522614119
	.long	414556326
	.long	123993874
	.long	4051156773
	.long	33416674
	.long	4294688947
	.long	2750260943
	.long	829112653
	.long	247987749
	.long	3807346250
	.long	66833348
	.long	4294410598
	.long	1205554591
	.long	1658225307
	.long	495975499
	.long	3319725204
	.long	133666696
	.long	4293853900
	.long	2411109183
	.long	3316450614
	.long	991950999
	.long	2344483112
	.long	267333393
	.long	4292740504
	.long	527251070
	.long	2337933933
	.long	1983901999
	.long	393998928
	.long	534666787
	.long	4290513712
	.long	1054502141
	.long	380900570
	.long	3967803999
	.long	787997856
	.long	1069333574
	.long	4286060128
	.long	2109004283
	.long	761801140
	.long	3640640703
	.long	1575995713
	.long	2138667148
	.long	4277152960
	.long	4218008566
	.long	1523602280
	.long	2986314111
	.long	3151991427
	.long	4277334296
	.long	4259338624
	.long	4141049836
	.long	3047204561
	.long	1677660927
	.long	2009015559
	.long	4259701297
	.long	4223709953
	.long	3987132377
	.long	1799441827
	.long	3355321855
	.long	4018031118
	.long	4224435298
	.long	4152452611
	.long	3679297459
	.long	3598883655
	.long	2415676415
	.long	3741094941
	.long	4153903301
	.long	4009937927
	.long	3063627623
	.long	2902800015
	.long	536385535
	.long	3187222587
	.long	4012839307
	.long	3724908559
	.long	1832287951
	.long	1510632735
	.long	1072771071
	.long	2079477878
	.long	3730711318
	.long	3154849823
	.long	3664575902
	.long	3021265470
	.long	2145542143
	.long	4158955756
	.long	3166455341
	.long	2014732351
	.long	3034184508
	.long	1747563645
	.long	4291084286
	.long	4022944216
	.long	2037943386
	.long	4029464703
	.long	1773401721
	.long	3495127291
	.long	4287201277
	.long	3750921137
	.long	4075886773
	.long	3763962110
	.long	3546803442
	.long	2695287286
	.long	4279435259
	.long	3206874979
	.long	3856806251
	.long	3232956925
	.long	2798639588
	.long	1095607277
	.long	4263903223
	.long	2118782663
	.long	3418645206
	.long	2170946555
	.long	1302311881
	.long	2191214555
	.long	4232839151
	.long	4237565327
	.long	2542323117
	.long	46925815
	.long	2604623763
	.long	87461814
	.long	4170711006
	.long	4180163359
	.long	789678938
	.long	93851631
	.long	914280231
	.long	174923629
	.long	4046454716
	.long	4065359423
	.long	1579357876
	.long	187703262
	.long	1828560463
	.long	349847258
	.long	3797942136
	.long	3835751551
	.long	3158715752
	.long	375406524
	.long	3657120927
	.long	699694516
	.long	3300916976
	.long	3376535807
	.long	2022464208
	.long	750813049
	.long	3019274558
	.long	1399389033
	.long	2306866656
	.long	2458104319
	.long	4044928416
	.long	1501626098
	.long	1743581820
	.long	2798778067
	.long	318766016
	.long	621241343
	.long	3794889537
	.long	3003252197
	.long	3487163641
	.long	1302588838
	.long	637532033
	.long	1242482686
	.long	3294811778
	.long	1711537099
	.long	2679359986
	.long	2605177677
	.long	1275064066
	.long	2484965372
	.long	2294656261
	.long	3423074199
	.long	1063752677
	.long	915388059
	.long	2550128133
	.long	674963448
	.long	294345226
	.long	2551181103
	.long	2127505355
	.long	1830776118
	.long	805288971
	.long	1349926897
	.long	588690452
	.long	807394910
	.long	4255010710
	.long	3661552236
	.long	1610577942
	.long	2699853794
	.long	1177380905
	.long	1614789820
	.long	4215054124
	.long	3028137177
	.long	3221155884
	.long	1104740292
	.long	2354761811
	.long	3229579640
	.long	4135140952
	.long	1761307059
	.long	2147344473
	.long	2209480585
	.long	414556326
	.long	2164191985
	.long	3975314608
	.long	3522614119
	.long	4294688947
	.long	123993874
	.long	829112653
	.long	33416674
	.long	3655661921
	.long	2750260943
	.long	4294410598
	.long	247987749
	.long	1658225307
	.long	66833348
	.long	3016356546
	.long	1205554591
	.long	4293853900
	.long	495975499
	.long	3316450614
	.long	133666696
	.long	1737745796
	.long	2411109183
	.long	4292740504
	.long	991950999
	.long	2337933933
	.long	267333393
	.long	3475491593
	.long	527251070
	.long	4290513712
	.long	1983901999
	.long	380900570
	.long	534666787
	.long	2656015891
	.long	1054502141
	.long	4286060128
	.long	3967803999
	.long	761801140
	.long	1069333574
	.long	1017064486
	.long	2109004283
	.long	4277152960
	.long	3640640703
	.long	1523602280
	.long	2138667148
	.long	2034128973
	.long	4218008566
	.long	4259338624
	.long	2986314111
	.long	3047204561
	.long	4277334296
	.long	4068257947
	.long	4141049836
	.long	4223709953
	.long	1677660927
	.long	1799441827
	.long	4259701297
	.long	3841548598
	.long	3987132377
	.long	4152452611
	.long	3355321855
	.long	3598883655
	.long	4224435298
	.long	3388129901
	.long	3679297459
	.long	4009937927
	.long	2415676415
	.long	2902800015
	.long	4153903301
	.long	2481292507
	.long	3063627623
	.long	3724908559
	.long	536385535
	.long	1510632735
	.long	4012839307
	.long	667617719
	.long	1832287951
	.long	3154849823
	.long	1072771071
	.long	3021265470
	.long	3730711318
	.long	1335235438
	.long	3664575902
	.long	2014732351
	.long	2145542143
	.long	1747563645
	.long	3166455341
	.long	2670470877
	.long	3034184508
	.long	4029464703
	.long	4291084286
	.long	3495127291
	.long	2037943386
	.long	1045974458
	.long	1773401721
	.long	3763962110
	.long	4287201277
	.long	2695287286
	.long	4075886773
	.long	2091948916
	.long	3546803442
	.long	3232956925
	.long	4279435259
	.long	1095607277
	.long	3856806251
	.long	4183897833
	.long	2798639588
	.long	2170946555
	.long	4263903223
	.long	2191214555
	.long	3418645206
	.long	4072828371
	.long	1302311881
	.long	46925815
	.long	4232839151
	.long	87461814
	.long	2542323117
	.long	3850689447
	.long	2604623763
	.long	93851631
	.long	4170711006
	.long	174923629
	.long	789678938
	.long	3406411599
	.long	914280231
	.long	187703262
	.long	4046454716
	.long	349847258
	.long	1579357876
	.long	2517855902
	.long	1828560463
	.long	375406524
	.long	3797942136
	.long	699694516
	.long	3158715752
	.long	740744509
	.long	3657120927
	.long	750813049
	.long	3300916976
	.long	1399389033
	.long	2022464208
	.long	1481489018
	.long	3019274558
	.long	1501626098
	.long	2306866656
	.long	2798778067
	.long	4044928416
	.long	2962978036
	.long	1743581820
	.long	3003252197
	.long	318766016
	.long	1302588838
	.long	3794889537
	.long	1630988776
	.long	3487163641
	.long	1711537099
	.long	637532033
	.long	2605177677
	.long	3294811778
	.long	3261977553
	.long	2679359986
	.long	3423074199
	.long	1275064066
	.long	915388059
	.long	2294656261
	.long	2228987811
	.long	1063752677
	.long	2551181103
	.long	2550128133
	.long	1830776118
	.long	294345226
	.long	163008326
	.long	2127505355
	.long	807394910
	.long	805288971
	.long	3661552236
	.long	588690452
	.long	326016652
	.long	4255010710
	.long	1614789820
	.long	1610577942
	.long	3028137177
	.long	1177380905
	.long	652033304
	.long	4215054124
	.long	3229579640
	.long	3221155884
	.long	1761307059
	.long	2354761811
	.long	1304066609
	.long	4135140952
	.long	2164191985
	.long	2147344473
	.long	3522614119
	.long	414556326
	.long	2608133219
	.long	3975314608
	.long	33416674
	.long	4294688947
	.long	2750260943
	.long	829112653
	.long	921299143
	.long	3655661921
	.long	66833348
	.long	4294410598
	.long	1205554591
	.long	1658225307
	.long	1842598287
	.long	3016356546
	.long	133666696
	.long	4293853900
	.long	2411109183
	.long	3316450614
	.long	3685196575
	.long	1737745796
	.long	267333393
	.long	4292740504
	.long	527251070
	.long	2337933933
	.long	3075425855
	.long	3475491593
	.long	534666787
	.long	4290513712
	.long	1054502141
	.long	380900570
	.long	1855884414
	.long	2656015891
	.long	1069333574
	.long	4286060128
	.long	2109004283
	.long	761801140
	.long	3711768829
	.long	1017064486
	.long	2138667148
	.long	4277152960
	.long	4218008566
	.long	1523602280
	.long	3128570363
	.long	2034128973
	.long	4277334296
	.long	4259338624
	.long	4141049836
	.long	3047204561
	.long	1962173430
	.long	4068257947
	.long	4259701297
	.long	4223709953
	.long	3987132377
	.long	1799441827
	.long	3924346860
	.long	3841548598
	.long	4224435298
	.long	4152452611
	.long	3679297459
	.long	3598883655
	.long	3553726425
	.long	3388129901
	.long	4153903301
	.long	4009937927
	.long	3063627623
	.long	2902800015
	.long	2812485555
	.long	2481292507
	.long	4012839307
	.long	3724908559
	.long	1832287951
	.long	1510632735
	.long	1330003814
	.long	667617719
	.long	3730711318
	.long	3154849823
	.long	3664575902
	.long	3021265470
	.long	2660007629
	.long	1335235438
	.long	3166455341
	.long	2014732351
	.long	3034184508
	.long	1747563645
	.long	1025047962
	.long	2670470877
	.long	2037943386
	.long	4029464703
	.long	1773401721
	.long	3495127291
	.long	2050095924
	.long	1045974458
	.long	4075886773
	.long	3763962110
	.long	3546803442
	.long	2695287286
	.long	4100191849
	.long	2091948916
	.long	3856806251
	.long	3232956925
	.long	2798639588
	.long	1095607277
	.long	3905416403
	.long	4183897833
	.long	3418645206
	.long	2170946555
	.long	1302311881
	.long	2191214555
	.long	3515865511
	.long	4072828371
	.long	2542323117
	.long	46925815
	.long	2604623763
	.long	87461814
	.long	2736763727
	.long	3850689447
	.long	789678938
	.long	93851631
	.long	914280231
	.long	174923629
	.long	1178560158
	.long	3406411599
	.long	1579357876
	.long	187703262
	.long	1828560463
	.long	349847258
	.long	2357120316
	.long	2517855902
	.long	3158715752
	.long	375406524
	.long	3657120927
	.long	699694516
	.long	419273337
	.long	740744509
	.long	2022464208
	.long	750813049
	.long	3019274558
	.long	1399389033
	.long	838546674
	.long	1481489018
	.long	4044928416
	.long	1501626098
	.long	1743581820
	.long	2798778067
	.long	1677093349
	.long	2962978036
	.long	3794889537
	.long	3003252197
	.long	3487163641
	.long	1302588838
	.long	3354186699
	.long	1630988776
	.long	3294811778
	.long	1711537099
	.long	2679359986
	.long	2605177677
	.long	2413406103
	.long	3261977553
	.long	2294656261
	.long	3423074199
	.long	1063752677
	.long	915388059
	.long	531844911
	.long	2228987811
	.long	294345226
	.long	2551181103
	.long	2127505355
	.long	1830776118
	.long	1063689823
	.long	163008326
	.long	588690452
	.long	807394910
	.long	4255010710
	.long	3661552236
	.long	2127379647
	.long	326016652
	.long	1177380905
	.long	1614789820
	.long	4215054124
	.long	3028137177
	.long	4254759295
	.long	652033304
	.long	2354761811
	.long	3229579640
	.long	4135140952
	.long	1761307059
	.long	4214551295
	.long	1304066609
	.long	414556326
	.long	2164191985
	.long	3975314608
	.long	3522614119
	.long	4134135294
	.long	2608133219
	.long	829112653
	.long	33416674
	.long	3655661921
	.long	2750260943
	.long	3973303293
	.long	921299143
	.long	1658225307
	.long	66833348
	.long	3016356546
	.long	1205554591
	.long	3651639290
	.long	1842598287
	.long	3316450614
	.long	133666696
	.long	1737745796
	.long	2411109183
	.long	3008311285
	.long	3685196575
	.long	2337933933
	.long	267333393
	.long	3475491593
	.long	527251070
	.long	1721655274
	.long	3075425855
	.long	380900570
	.long	534666787
	.long	2656015891
	.long	1054502141
	.long	3443310548
	.long	1855884414
	.long	761801140
	.long	1069333574
	.long	1017064486
	.long	2109004283
	.long	2591653800
	.long	3711768829
	.long	1523602280
	.long	2138667148
	.long	2034128973
	.long	4218008566
	.long	888340305
	.long	3128570363
	.long	3047204561
	.long	4277334296
	.long	4068257947
	.long	4141049836
	.long	1776680610
	.long	1962173430
	.long	1799441827
	.long	4259701297
	.long	3841548598
	.long	3987132377
	.long	3553361221
	.long	3924346860
	.long	3598883655
	.long	4224435298
	.long	3388129901
	.long	3679297459
	.long	2811755147
	.long	3553726425
	.long	2902800015
	.long	4153903301
	.long	2481292507
	.long	3063627623
	.long	1328542998
	.long	2812485555
	.long	1510632735
	.long	4012839307
	.long	667617719
	.long	1832287951
	.long	2657085997
	.long	1330003814
	.long	3021265470
	.long	3730711318
	.long	1335235438
	.long	3664575902
	.long	1019204698
	.long	2660007629
	.long	1747563645
	.long	3166455341
	.long	2670470877
	.long	3034184508
	.long	2038409397
	.long	1025047962
	.long	3495127291
	.long	2037943386
	.long	1045974458
	.long	1773401721
	.long	4076818795
	.long	2050095924
	.long	2695287286
	.long	4075886773
	.long	2091948916
	.long	3546803442
	.long	3858670295
	.long	4100191849
	.long	1095607277
	.long	3856806251
	.long	4183897833
	.long	2798639588
	.long	3422373294
	.long	3905416403
	.long	2191214555
	.long	3418645206
	.long	4072828371
	.long	1302311881
	.long	2549779293
	.long	3515865511
	.long	87461814
	.long	2542323117
	.long	3850689447
	.long	2604623763
	.long	804591290
	.long	2736763727
	.long	174923629
	.long	789678938
	.long	3406411599
	.long	914280231
	.long	1609182581
	.long	1178560158
	.long	349847258
	.long	1579357876
	.long	2517855902
	.long	1828560463
	.long	3218365162
	.long	2357120316
	.long	699694516
	.long	3158715752
	.long	740744509
	.long	3657120927
	.long	2141763028
	.long	419273337
	.long	1399389033
	.long	2022464208
	.long	1481489018
	.long	3019274558
	.long	4283526057
	.long	838546674
	.long	2798778067
	.long	4044928416
	.long	2962978036
	.long	1743581820
	.long	4272084818
	.long	1677093349
	.long	1302588838
	.long	3794889537
	.long	1630988776
	.long	3487163641
	.long	4249202340
	.long	3354186699
	.long	2605177677
	.long	3294811778
	.long	3261977553
	.long	2679359986
	.long	4203437385
	.long	2413406103
	.long	915388059
	.long	2294656261
	.long	2228987811
	.long	1063752677
	.long	4111907475
	.long	531844911
	.long	1830776118
	.long	294345226
	.long	163008326
	.long	2127505355
	.long	3928847655
	.long	1063689823
	.long	3661552236
	.long	588690452
	.long	326016652
	.long	4255010710
	.long	3562728015
	.long	2127379647
	.long	3028137177
	.long	1177380905
	.long	652033304
	.long	4215054124
	.long	2830488734
	.long	4254759295
	.long	1761307059
	.long	2354761811
	.long	1304066609
	.long	4135140952
	.long	1366010173
	.long	4214551295
	.long	3522614119
	.long	414556326
	.long	2608133219
	.long	3975314608
	.long	2732020347
	.long	4134135294
	.long	2750260943
	.long	829112653
	.long	921299143
	.long	3655661921
	.long	1169073399
	.long	3973303293
	.long	1205554591
	.long	1658225307
	.long	1842598287
	.long	3016356546
	.long	2338146798
	.long	3651639290
	.long	2411109183
	.long	3316450614
	.long	3685196575
	.long	1737745796
	.long	381326301
	.long	3008311285
	.long	527251070
	.long	2337933933
	.long	3075425855
	.long	3475491593
	.long	762652602
	.long	1721655274
	.long	1054502141
	.long	380900570
	.long	1855884414
	.long	2656015891
	.long	1525305205
	.long	3443310548
	.long	2109004283
	.long	761801140
	.long	3711768829
	.long	1017064486
	.long	3050610411
	.long	2591653800
	.long	4218008566
	.long	1523602280
	.long	3128570363
	.long	2034128973
	.long	1806253526
	.long	888340305
	.long	4141049836
	.long	3047204561
	.long	1962173430
	.long	4068257947
	.long	3612507052
	.long	1776680610
	.long	3987132377
	.long	1799441827
	.long	3924346860
	.long	3841548598
	.long	2930046808
	.long	3553361221
	.long	3679297459
	.long	3598883655
	.long	3553726425
	.long	3388129901
	.long	1565126321
	.long	2811755147
	.long	3063627623
	.long	2902800015
	.long	2812485555
	.long	2481292507
	.long	3130252643
	.long	1328542998
	.long	1832287951
	.long	1510632735
	.long	1330003814
	.long	667617719
	.long	1965537991
	.long	2657085997
	.long	3664575902
	.long	3021265470
	.long	2660007629
	.long	1335235438
	.long	3931075983
	.long	1019204698
	.long	3034184508
	.long	1747563645
	.long	1025047962
	.long	2670470877
	.long	3567184671
	.long	2038409397
	.long	1773401721
	.long	3495127291
	.long	2050095924
	.long	1045974458
	.long	2839402047
	.long	4076818795
	.long	3546803442
	.long	2695287286
	.long	4100191849
	.long	2091948916
	.long	1383836798
	.long	3858670295
	.long	2798639588
	.long	1095607277
	.long	3905416403
	.long	4183897833
	.long	2767673597
	.long	3422373294
	.long	1302311881
	.long	2191214555
	.long	3515865511
	.long	4072828371
	.long	1240379898
	.long	2549779293
	.long	2604623763
	.long	87461814
	.long	2736763727
	.long	3850689447
	.long	2480759797
	.long	804591290
	.long	914280231
	.long	174923629
	.long	1178560158
	.long	3406411599
	.long	666552299
	.long	1609182581
	.long	1828560463
	.long	349847258
	.long	2357120316
	.long	2517855902
	.long	1333104599
	.long	3218365162
	.long	3657120927
	.long	699694516
	.long	419273337
	.long	740744509
	.long	2666209199
	.long	2141763028
	.long	3019274558
	.long	1399389033
	.long	838546674
	.long	1481489018
	.long	1037451103
	.long	4283526057
	.long	1743581820
	.long	2798778067
	.long	1677093349
	.long	2962978036
	.long	2074902206
	.long	4272084818
	.long	3487163641
	.long	1302588838
	.long	3354186699
	.long	1630988776
	.long	4149804412
	.long	4249202340
	.long	2679359986
	.long	2605177677
	.long	2413406103
	.long	3261977553
	.long	4004641529
	.long	4203437385
	.long	1063752677
	.long	915388059
	.long	531844911
	.long	2228987811
	.long	3714315762
	.long	4111907475
	.long	2127505355
	.long	1830776118
	.long	1063689823
	.long	163008326
	.long	3133664229
	.long	3928847655
	.long	4255010710
	.long	3661552236
	.long	2127379647
	.long	326016652
	.long	1972361163
	.long	3562728015
	.long	4215054124
	.long	3028137177
	.long	4254759295
	.long	652033304
	.long	3944722327
	.long	2830488734
	.long	4135140952
	.long	1761307059
	.long	4214551295
	.long	1304066609
	.long	3594477359
	.long	1366010173
	.long	3975314608
	.long	3522614119
	.long	4134135294
	.long	2608133219
	.long	2893987423
	.long	2732020347
	.long	3655661921
	.long	2750260943
	.long	3973303293
	.long	921299143
	.long	1493007550
	.long	1169073399
	.long	3016356546
	.long	1205554591
	.long	3651639290
	.long	1842598287
	.long	2986015100
	.long	2338146798
	.long	1737745796
	.long	2411109183
	.long	3008311285
	.long	3685196575
	.long	1677062904
	.long	381326301
	.long	3475491593
	.long	527251070
	.long	1721655274
	.long	3075425855
	.long	3354125809
	.long	762652602
	.long	2656015891
	.long	1054502141
	.long	3443310548
	.long	1855884414
	.long	2413284322
	.long	1525305205
	.long	1017064486
	.long	2109004283
	.long	2591653800
	.long	3711768829
	.long	531601349
	.long	3050610411
	.long	2034128973
	.long	4218008566
	.long	888340305
	.long	3128570363
	.long	1063202699
	.long	1806253526
	.long	4068257947
	.long	4141049836
	.long	1776680610
	.long	1962173430
	.long	2126405399
	.long	3612507052
	.long	3841548598
	.long	3987132377
	.long	3553361221
	.long	3924346860
	.long	4252810799
	.long	2930046808
	.long	3388129901
	.long	3679297459
	.long	2811755147
	.long	3553726425
	.long	4210654302
	.long	1565126321
	.long	2481292507
	.long	3063627623
	.long	1328542998
	.long	2812485555
	.long	4126341309
	.long	3130252643
	.long	667617719
	.long	1832287951
	.long	2657085997
	.long	1330003814
	.long	3957715323
	.long	1965537991
	.long	1335235438
	.long	3664575902
	.long	1019204698
	.long	2660007629
	.long	3620463350
	.long	3931075983
	.long	2670470877
	.long	3034184508
	.long	2038409397
	.long	1025047962
	.long	2945959404
	.long	3567184671
	.long	1045974458
	.long	1773401721
	.long	4076818795
	.long	2050095924
	.long	1596951513
	.long	2839402047
	.long	2091948916
	.long	3546803442
	.long	3858670295
	.long	4100191849
	.long	3193903027
	.long	1383836798
	.long	4183897833
	.long	2798639588
	.long	3422373294
	.long	3905416403
	.long	2092838759
	.long	2767673597
	.long	4072828371
	.long	1302311881
	.long	2549779293
	.long	3515865511
	.long	4185677519
	.long	1240379898
	.long	3850689447
	.long	2604623763
	.long	804591290
	.long	2736763727
	.long	4076387742
	.long	2480759797
	.long	3406411599
	.long	914280231
	.long	1609182581
	.long	1178560158
	.long	3857808189
	.long	666552299
	.long	2517855902
	.long	1828560463
	.long	3218365162
	.long	2357120316
	.long	3420649082
	.long	1333104599
	.long	740744509
	.long	3657120927
	.long	2141763028
	.long	419273337
	.long	2546330868
	.long	2666209199
	.long	1481489018
	.long	3019274558
	.long	4283526057
	.long	838546674
	.long	797694440
	.long	1037451103
	.long	2962978036
	.long	1743581820
	.long	4272084818
	.long	1677093349
	.long	1595388880
	.long	2074902206
	.long	1630988776
	.long	3487163641
	.long	4249202340
	.long	3354186699
	.long	3190777760
	.long	4149804412
	.long	3261977553
	.long	2679359986
	.long	4203437385
	.long	2413406103
	.long	2086588225
	.long	4004641529
	.long	2228987811
	.long	1063752677
	.long	4111907475
	.long	531844911
	.long	4173176451
	.long	3714315762
	.long	163008326
	.long	2127505355
	.long	3928847655
	.long	1063689823
	.long	4051385607
	.long	3133664229
	.long	326016652
	.long	4255010710
	.long	3562728015
	.long	2127379647
	.long	3807803918
	.long	1972361163
	.long	652033304
	.long	4215054124
	.long	2830488734
	.long	4254759295
	.long	3320640540
	.long	3944722327
	.long	1304066609
	.long	4135140952
	.long	1366010173
	.long	4214551295
	.long	2346313785
	.long	3594477359
	.long	2608133219
	.long	3975314608
	.long	2732020347
	.long	4134135294
	.long	397660275
	.long	2893987423
	.long	921299143
	.long	3655661921
	.long	1169073399
	.long	3973303293
	.long	795320551
	.long	1493007550
	.long	1842598287
	.long	3016356546
	.long	2338146798
	.long	3651639290
	.long	1590641102
	.long	2986015100
	.long	3685196575
	.long	1737745796
	.long	381326301
	.long	3008311285
	.long	3181282204
	.long	1677062904
	.long	3075425855
	.long	3475491593
	.long	762652602
	.long	1721655274
	.long	2067597113
	.long	3354125809
	.long	1855884414
	.long	2656015891
	.long	1525305205
	.long	3443310548
	.long	4135194227
	.long	2413284322
	.long	3711768829
	.long	1017064486
	.long	3050610411
	.long	2591653800
	.long	3975421159
	.long	531601349
	.long	3128570363
	.long	2034128973
	.long	1806253526
	.long	888340305
	.long	3655875023
	.long	1063202699
	.long	1962173430
	.long	4068257947
	.long	3612507052
	.long	1776680610
	.long	3016782751
	.long	2126405399
	.long	3924346860
	.long	3841548598
	.long	2930046808
	.long	3553361221
	.long	1738598206
	.long	4252810799
	.long	3553726425
	.long	3388129901
	.long	1565126321
	.long	2811755147
	.long	3477196413
	.long	4210654302
	.long	2812485555
	.long	2481292507
	.long	3130252643
	.long	1328542998
	.long	2659425531
	.long	4126341309
	.long	1330003814
	.long	667617719
	.long	1965537991
	.long	2657085997
	.long	1023883767
	.long	3957715323
	.long	2660007629
	.long	1335235438
	.long	3931075983
	.long	1019204698
	.long	2047767535
	.long	3620463350
	.long	1025047962
	.long	2670470877
	.long	3567184671
	.long	2038409397
	.long	4095535070
	.long	2945959404
	.long	2050095924
	.long	1045974458
	.long	2839402047
	.long	4076818795
	.long	3896102844
	.long	1596951513
	.long	4100191849
	.long	2091948916
	.long	1383836798
	.long	3858670295
	.long	3497238392
	.long	3193903027
	.long	3905416403
	.long	4183897833
	.long	2767673597
	.long	3422373294
	.long	2699509489
	.long	2092838759
	.long	3515865511
	.long	4072828371
	.long	1240379898
	.long	2549779293
	.long	1104051682
	.long	4185677519
	.long	2736763727
	.long	3850689447
	.long	2480759797
	.long	804591290
	.long	2208103365
	.long	4076387742
	.long	1178560158
	.long	3406411599
	.long	666552299
	.long	1609182581
	.long	121239434
	.long	3857808189
	.long	2357120316
	.long	2517855902
	.long	1333104599
	.long	3218365162
	.long	242478868
	.long	3420649082
	.long	419273337
	.long	740744509
	.long	2666209199
	.long	2141763028
	.long	484957737
	.long	2546330868
	.long	838546674
	.long	1481489018
	.long	1037451103
	.long	4283526057
	.long	969915474
	.long	797694440
	.long	1677093349
	.long	2962978036
	.long	2074902206
	.long	4272084818
	.long	1939830949
	.long	1595388880
	.long	3354186699
	.long	1630988776
	.long	4149804412
	.long	4249202340
	.long	3879661898
	.long	3190777760
	.long	2413406103
	.long	3261977553
	.long	4004641529
	.long	4203437385
	.long	3464356500
	.long	2086588225
	.long	531844911
	.long	2228987811
	.long	3714315762
	.long	4111907475
	.long	2633745705
	.long	4173176451
	.long	1063689823
	.long	163008326
	.long	3133664229
	.long	3928847655
	.long	972524114
	.long	4051385607
	.long	2127379647
	.long	326016652
	.long	1972361163
	.long	3562728015
	.long	1945048229
	.long	3807803918
	.long	4254759295
	.long	652033304
	.long	3944722327
	.long	2830488734
	.long	3890096458
	.long	3320640540
	.long	4214551295
	.long	1304066609
	.long	3594477359
	.long	1366010173
	.long	3485225620
	.long	2346313785
	.long	4134135294
	.long	2608133219
	.long	2893987423
	.long	2732020347
	.long	2675483945
	.long	397660275
	.long	3973303293
	.long	921299143
	.long	1493007550
	.long	1169073399
	.long	1056000594
	.long	795320551
	.long	3651639290
	.long	1842598287
	.long	2986015100
	.long	2338146798
	.long	2112001188
	.long	1590641102
	.long	3008311285
	.long	3685196575
	.long	1677062904
	.long	381326301
	.long	4224002377
	.long	3181282204
	.long	1721655274
	.long	3075425855
	.long	3354125809
	.long	762652602
	.long	4153037458
	.long	2067597113
	.long	3443310548
	.long	1855884414
	.long	2413284322
	.long	1525305205
	.long	4011107621
	.long	4135194227
	.long	2591653800
	.long	3711768829
	.long	531601349
	.long	3050610411
	.long	3727247947
	.long	3975421159
	.long	888340305
	.long	3128570363
	.long	1063202699
	.long	1806253526
	.long	3159528599
	.long	3655875023
	.long	1776680610
	.long	1962173430
	.long	2126405399
	.long	3612507052
	.long	2024089902
	.long	3016782751
	.long	3553361221
	.long	3924346860
	.long	4252810799
	.long	2930046808
	.long	4048179805
	.long	1738598206
	.long	2811755147
	.long	3553726425
	.long	4210654302
	.long	1565126321
	.long	3801392314
	.long	3477196413
	.long	1328542998
	.long	2812485555
	.long	4126341309
	.long	3130252643
	.long	3307817333
	.long	2659425531
	.long	2657085997
	.long	1330003814
	.long	3957715323
	.long	1965537991
	.long	2320667370
	.long	1023883767
	.long	1019204698
	.long	2660007629
	.long	3620463350
	.long	3931075983
	.long	346367444
	.long	2047767535
	.long	2038409397
	.long	1025047962
	.long	2945959404
	.long	3567184671
	.long	692734889
	.long	4095535070
	.long	4076818795
	.long	2050095924
	.long	1596951513
	.long	2839402047
	.long	1385469779
	.long	3896102844
	.long	3858670295
	.long	4100191849
	.long	3193903027
	.long	1383836798
	.long	2770939558
	.long	3497238392
	.long	3422373294
	.long	3905416403
	.long	2092838759
	.long	2767673597
	.long	1246911821
	.long	2699509489
	.long	2549779293
	.long	3515865511
	.long	4185677519
	.long	1240379898
	.long	2493823642
	.long	1104051682
	.long	804591290
	.long	2736763727
	.long	4076387742
	.long	2480759797
	.long	692679989
	.long	2208103365
	.long	1609182581
	.long	1178560158
	.long	3857808189
	.long	666552299
	.long	1385359979
	.long	121239434
	.long	3218365162
	.long	2357120316
	.long	3420649082
	.long	1333104599
	.long	2770719959
	.long	242478868
	.long	2141763028
	.long	419273337
	.long	2546330868
	.long	2666209199
	.long	1246472623
	.long	484957737
	.long	4283526057
	.long	838546674
	.long	797694440
	.long	1037451103
	.long	2492945247
	.long	969915474
	.long	4272084818
	.long	1677093349
	.long	1595388880
	.long	2074902206
	.long	690923199
	.long	1939830949
	.long	4249202340
	.long	3354186699
	.long	3190777760
	.long	4149804412
	.long	1381846399
	.long	3879661898
	.long	4203437385
	.long	2413406103
	.long	2086588225
	.long	4004641529
	.long	2763692798
	.long	3464356500
	.long	4111907475
	.long	531844911
	.long	4173176451
	.long	3714315762
	.long	1232418301
	.long	2633745705
	.long	3928847655
	.long	1063689823
	.long	4051385607
	.long	3133664229
	.long	2464836603
	.long	972524114
	.long	3562728015
	.long	2127379647
	.long	3807803918
	.long	1972361163
	.long	634705910
	.long	1945048229
	.long	2830488734
	.long	4254759295
	.long	3320640540
	.long	3944722327
	.long	1269411821
	.long	3890096458
	.long	1366010173
	.long	4214551295
	.long	2346313785
	.long	3594477359
	.long	2538823642
	.long	3485225620
	.long	2732020347
	.long	4134135294
	.long	397660275
	.long	2893987423
	.long	782679989
	.long	2675483945
	.long	1169073399
	.long	3973303293
	.long	795320551
	.long	1493007550
	.long	1565359979
	.long	1056000594
	.long	2338146798
	.long	3651639290
	.long	1590641102
	.long	2986015100
	.long	3130719959
	.long	2112001188
	.long	381326301
	.long	3008311285
	.long	3181282204
	.long	1677062904
	.long	1966472623
	.long	4224002377
	.long	762652602
	.long	1721655274
	.long	2067597113
	.long	3354125809
	.long	3932945247
	.long	4153037458
	.long	1525305205
	.long	3443310548
	.long	4135194227
	.long	2413284322
	.long	3570923199
	.long	4011107621
	.long	3050610411
	.long	2591653800
	.long	3975421159
	.long	531601349
	.long	2846879102
	.long	3727247947
	.long	1806253526
	.long	888340305
	.long	3655875023
	.long	1063202699
	.long	1398790909
	.long	3159528599
	.long	3612507052
	.long	1776680610
	.long	3016782751
	.long	2126405399
	.long	2797581819
	.long	2024089902
	.long	2930046808
	.long	3553361221
	.long	1738598206
	.long	4252810799
	.long	1300196342
	.long	4048179805
	.long	1565126321
	.long	2811755147
	.long	3477196413
	.long	4210654302
	.long	2600392684
	.long	3801392314
	.long	3130252643
	.long	1328542998
	.long	2659425531
	.long	4126341309
	.long	905818072
	.long	3307817333
	.long	1965537991
	.long	2657085997
	.long	1023883767
	.long	3957715323
	.long	1811636145
	.long	2320667370
	.long	3931075983
	.long	1019204698
	.long	2047767535
	.long	3620463350
	.long	3623272290
	.long	346367444
	.long	3567184671
	.long	2038409397
	.long	4095535070
	.long	2945959404
	.long	2951577284
	.long	692734889
	.long	2839402047
	.long	4076818795
	.long	3896102844
	.long	1596951513
	.long	1608187272
	.long	1385469779
	.long	1383836798
	.long	3858670295
	.long	3497238392
	.long	3193903027
	.long	3216374545
	.long	2770939558
	.long	2767673597
	.long	3422373294
	.long	2699509489
	.long	2092838759
	.long	2137781795
	.long	1246911821
	.long	1240379898
	.long	2549779293
	.long	1104051682
	.long	4185677519
	.long	4275563591
	.long	2493823642
	.long	2480759797
	.long	804591290
	.long	2208103365
	.long	4076387742
	.long	4256159887
	.long	692679989
	.long	666552299
	.long	1609182581
	.long	121239434
	.long	3857808189
	.long	4217352479
	.long	1385359979
	.long	1333104599
	.long	3218365162
	.long	242478868
	.long	3420649082
	.long	4139737663
	.long	2770719959
	.long	2666209199
	.long	2141763028
	.long	484957737
	.long	2546330868
	.long	3984508030
	.long	1246472623
	.long	1037451103
	.long	4283526057
	.long	969915474
	.long	797694440
	.long	3674048764
	.long	2492945247
	.long	2074902206
	.long	4272084818
	.long	1939830949
	.long	1595388880
	.long	3053130232
	.long	690923199
	.long	4149804412
	.long	4249202340
	.long	3879661898
	.long	3190777760
	.long	1811293169
	.long	1381846399
	.long	4004641529
	.long	4203437385
	.long	3464356500
	.long	2086588225
	.long	3622586339
	.long	2763692798
	.long	3714315762
	.long	4111907475
	.long	2633745705
	.long	4173176451
	.long	2950205382
	.long	1232418301
	.long	3133664229
	.long	3928847655
	.long	972524114
	.long	4051385607
	.long	1605443469
	.long	2464836603
	.long	1972361163
	.long	3562728015
	.long	1945048229
	.long	3807803918
	.long	3210886938
	.long	634705910
	.long	3944722327
	.long	2830488734
	.long	3890096458
	.long	3320640540
	.long	2126806581
	.long	1269411821
	.long	3594477359
	.long	1366010173
	.long	3485225620
	.long	2346313785
	.long	4253613162
	.long	2538823642
	.long	2893987423
	.long	2732020347
	.long	2675483945
	.long	397660275
	.long	4212259029
	.long	782679989
	.long	1493007550
	.long	1169073399
	.long	1056000594
	.long	795320551
	.long	4129550763
	.long	1565359979
	.long	2986015100
	.long	2338146798
	.long	2112001188
	.long	1590641102
	.long	3964134231
	.long	3130719959
	.long	1677062904
	.long	381326301
	.long	4224002377
	.long	3181282204
	.long	3633301166
	.long	1966472623
	.long	3354125809
	.long	762652602
	.long	4153037458
	.long	2067597113
	.long	2971635037
	.long	3932945247
	.long	2413284322
	.long	1525305205
	.long	4011107621
	.long	4135194227
	.long	1648302778
	.long	3570923199
	.long	531601349
	.long	3050610411
	.long	3727247947
	.long	3975421159
	.long	3296605556
	.long	2846879102
	.long	1063202699
	.long	1806253526
	.long	3159528599
	.long	3655875023
	.long	2298243816
	.long	1398790909
	.long	2126405399
	.long	3612507052
	.long	2024089902
	.long	3016782751
	.long	301520336
	.long	2797581819
	.long	4252810799
	.long	2930046808
	.long	4048179805
	.long	1738598206
	.long	603040673
	.long	1300196342
	.long	4210654302
	.long	1565126321
	.long	3801392314
	.long	3477196413
	.long	1206081346
	.long	2600392684
	.long	4126341309
	.long	3130252643
	.long	3307817333
	.long	2659425531
	.long	2412162692
	.long	905818072
	.long	3957715323
	.long	1965537991
	.long	2320667370
	.long	1023883767
	.long	529358088
	.long	1811636145
	.long	3620463350
	.long	3931075983
	.long	346367444
	.long	2047767535
	.long	1058716176
	.long	3623272290
	.long	2945959404
	.long	3567184671
	.long	692734889
	.long	4095535070
	.long	2117432353
	.long	2951577284
	.long	1596951513
	.long	2839402047
	.long	1385469779
	.long	3896102844
	.long	4234864706
	.long	1608187272
	.long	3193903027
	.long	1383836798
	.long	2770939558
	.long	3497238392
	.long	4174762117
	.long	3216374545
	.long	2092838759
	.long	2767673597
	.long	1246911821
	.long	2699509489
	.long	4054556938
	.long	2137781795
	.long	4185677519
	.long	1240379898
	.long	2493823642
	.long	1104051682
	.long	3814146581
	.long	4275563591
	.long	4076387742
	.long	2480759797
	.long	692679989
	.long	2208103365
	.long	3333325867
	.long	4256159887
	.long	3857808189
	.long	666552299
	.long	1385359979
	.long	121239434
	.long	2371684438
	.long	4217352479
	.long	3420649082
	.long	1333104599
	.long	2770719959
	.long	242478868
	.long	448401580
	.long	4139737663
	.long	2546330868
	.long	2666209199
	.long	1246472623
	.long	484957737
	.long	896803160
	.long	3984508030
	.type	__svml_dtan_reduction_data_internal,@object
	.size	__svml_dtan_reduction_data_internal,49152
	.space 512, 0x00 	
	.align 32
.L_2il0floatpacket.35:
	.long	0x00000008,0x00000008,0x00000008,0x00000008,0x00000008,0x00000008,0x00000008,0x00000008
	.type	.L_2il0floatpacket.35,@object
	.size	.L_2il0floatpacket.35,32
	.align 8
.L_2il0floatpacket.18:
	.long	0xffffffff,0xffffffff
	.type	.L_2il0floatpacket.18,@object
	.size	.L_2il0floatpacket.18,8
	.align 8
.L_2il0floatpacket.19:
	.long	0x00000000,0x7ff00000
	.type	.L_2il0floatpacket.19,@object
	.size	.L_2il0floatpacket.19,8
	.align 8
.L_2il0floatpacket.20:
	.long	0xffffffff,0x000fffff
	.type	.L_2il0floatpacket.20,@object
	.size	.L_2il0floatpacket.20,8
	.align 8
.L_2il0floatpacket.21:
	.long	0x00000000,0x00100000
	.type	.L_2il0floatpacket.21,@object
	.size	.L_2il0floatpacket.21,8
	.align 8
.L_2il0floatpacket.22:
	.long	0xffffffff,0x00000000
	.type	.L_2il0floatpacket.22,@object
	.size	.L_2il0floatpacket.22,8
	.align 8
.L_2il0floatpacket.23:
	.long	0x00000000,0x80000000
	.type	.L_2il0floatpacket.23,@object
	.size	.L_2il0floatpacket.23,8
	.align 8
.L_2il0floatpacket.24:
	.long	0x00000000,0x3ff00000
	.type	.L_2il0floatpacket.24,@object
	.size	.L_2il0floatpacket.24,8
	.align 8
.L_2il0floatpacket.25:
	.long	0x00000000,0x42a80000
	.type	.L_2il0floatpacket.25,@object
	.size	.L_2il0floatpacket.25,8
	.align 8
.L_2il0floatpacket.26:
	.long	0x000001ff,0x00000000
	.type	.L_2il0floatpacket.26,@object
	.size	.L_2il0floatpacket.26,8
	.align 8
.L_2il0floatpacket.27:
	.long	0x00000000,0x39700000
	.type	.L_2il0floatpacket.27,@object
	.size	.L_2il0floatpacket.27,8
	.align 8
.L_2il0floatpacket.28:
	.long	0x00ffffff,0x00000000
	.type	.L_2il0floatpacket.28,@object
	.size	.L_2il0floatpacket.28,8
	.align 8
.L_2il0floatpacket.29:
	.long	0x00000000,0x3cb00000
	.type	.L_2il0floatpacket.29,@object
	.size	.L_2il0floatpacket.29,8
	.align 8
.L_2il0floatpacket.30:
	.long	0x00000fff,0x00000000
	.type	.L_2il0floatpacket.30,@object
	.size	.L_2il0floatpacket.30,8
	.align 8
.L_2il0floatpacket.31:
	.long	0x54442d18,0x401921fb
	.type	.L_2il0floatpacket.31,@object
	.size	.L_2il0floatpacket.31,8
	.align 8
.L_2il0floatpacket.32:
	.long	0x33145c07,0x3cb1a626
	.type	.L_2il0floatpacket.32,@object
	.size	.L_2il0floatpacket.32,8
	.align 8
.L_2il0floatpacket.33:
	.long	0xffffffff,0x7fffffff
	.type	.L_2il0floatpacket.33,@object
	.size	.L_2il0floatpacket.33,8
	.align 8
.L_2il0floatpacket.34:
	.long	0x00000000,0x3eb00000
	.type	.L_2il0floatpacket.34,@object
	.size	.L_2il0floatpacket.34,8
	.align 8
.L_2il0floatpacket.36:
	.long	0x54442d18,0x3f8921fb
	.type	.L_2il0floatpacket.36,@object
	.size	.L_2il0floatpacket.36,8
	.align 8
.L_2il0floatpacket.37:
	.long	0x33145c07,0x3c21a626
	.type	.L_2il0floatpacket.37,@object
	.size	.L_2il0floatpacket.37,8
	.align 8
__dtan_la_Tab:
	.long	0
	.long	0
	.long	0
	.long	2146435072
	.type	__dtan_la_Tab,@object
	.size	__dtan_la_Tab,16

      .section        .note.GNU-stack,"",@progbits
