/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t4_t6_decode_states.h - state tables for T.4/T.6 FAX image decoding
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Legitimate runs of zero bits which are the tail end of one code
   plus the start of the next code do not exceed 10 bits. */

/* Finite state machine state codes */
enum
{
    S_Null      = 0,
    S_Pass      = 1,
    S_Horiz     = 2,
    S_Vert      = 3,
    S_Ext       = 4,
    S_TermW     = 5,
    S_TermB     = 6,
    S_MakeUpW   = 7,
    S_MakeUpB   = 8,
    S_MakeUp    = 9,
    S_EOL       = 10
};

/*! T.4 finite state machine state table entry */
typedef struct
{
    /*! State */
    uint8_t state;
    /*! Width of code in bits */
    uint8_t width;
    /*! Run length in bits */
    int16_t param;
} t4_table_entry_t;

static const t4_table_entry_t t4_2d_table[128] =
{
    {S_Null,       1,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       6,   -2},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       7,   -3},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       6,    2},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Ext,        7,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       6,   -2},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       7,    3},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Vert,       6,    2},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0},
    {S_Pass,       4,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,   -1},
    {S_Vert,       1,    0},
    {S_Horiz,      3,    0},
    {S_Vert,       1,    0},
    {S_Vert,       3,    1},
    {S_Vert,       1,    0}
};

static const t4_table_entry_t t4_1d_white_table[4096] =
{
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1792},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1856},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2112},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2368},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 1984},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1920},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2240},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2496},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_EOL,       12,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1792},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1856},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2176},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2432},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2048},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    11, 1920},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1472},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1216},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  960},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  704},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2304},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  832},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1600},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1344},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1088},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_Null,       1,    0},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   39},
    {S_TermW,      6,   16},
    {S_MakeUpW,    8,  576},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   55},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   45},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   53},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  448},
    {S_TermW,      4,    6},
    {S_TermW,      8,   35},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   51},
    {S_TermW,      6,   15},
    {S_TermW,      8,   63},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1536},
    {S_TermW,      4,    5},
    {S_TermW,      8,   43},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1280},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   29},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   33},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   49},
    {S_TermW,      6,   14},
    {S_TermW,      8,   61},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   47},
    {S_TermW,      4,    3},
    {S_TermW,      8,   59},
    {S_TermW,      4,    5},
    {S_TermW,      8,   41},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1024},
    {S_TermW,      4,    6},
    {S_TermW,      8,   31},
    {S_TermW,      5,    8},
    {S_TermW,      8,   57},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    9,  768},
    {S_TermW,      4,    6},
    {S_TermW,      8,   37},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  320},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_MakeUp,    12, 2560},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      7,   20},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   24},
    {S_TermW,      6,   14},
    {S_TermW,      7,   28},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   23},
    {S_TermW,      4,    3},
    {S_TermW,      7,   27},
    {S_TermW,      4,    5},
    {S_TermW,      8,   40},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9,  896},
    {S_TermW,      4,    6},
    {S_TermW,      7,   19},
    {S_TermW,      5,    8},
    {S_TermW,      8,   56},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   46},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      8,   54},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  512},
    {S_TermW,      4,    6},
    {S_TermW,      8,   36},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   52},
    {S_TermW,      6,   15},
    {S_TermW,      8,    0},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_MakeUpW,    9, 1728},
    {S_TermW,      4,    5},
    {S_TermW,      8,   44},
    {S_TermW,      6,   17},
    {S_MakeUpW,    9, 1408},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   30},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      6,   12},
    {S_TermW,      5,    9},
    {S_MakeUpW,    6, 1664},
    {S_TermW,      4,    6},
    {S_TermW,      8,   34},
    {S_MakeUpW,    5,  128},
    {S_TermW,      8,   50},
    {S_TermW,      6,   14},
    {S_TermW,      8,   62},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      8,   48},
    {S_TermW,      4,    3},
    {S_TermW,      8,   60},
    {S_TermW,      4,    5},
    {S_TermW,      8,   42},
    {S_TermW,      6,   16},
    {S_MakeUpW,    9, 1152},
    {S_TermW,      4,    6},
    {S_TermW,      8,   32},
    {S_TermW,      5,    8},
    {S_TermW,      8,   58},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      7,   22},
    {S_TermW,      4,    3},
    {S_TermW,      5,   11},
    {S_TermW,      4,    5},
    {S_TermW,      7,   26},
    {S_TermW,      5,    9},
    {S_MakeUpW,    8,  640},
    {S_TermW,      4,    6},
    {S_TermW,      8,   38},
    {S_MakeUpW,    5,  128},
    {S_TermW,      7,   25},
    {S_TermW,      6,   15},
    {S_MakeUpW,    8,  384},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7},
    {S_TermW,      6,   13},
    {S_TermW,      4,    3},
    {S_TermW,      7,   18},
    {S_TermW,      4,    5},
    {S_TermW,      7,   21},
    {S_TermW,      6,   17},
    {S_MakeUpW,    7,  256},
    {S_TermW,      4,    6},
    {S_TermW,      6,    1},
    {S_TermW,      5,    8},
    {S_MakeUpW,    6,  192},
    {S_MakeUpW,    5,   64},
    {S_TermW,      5,   10},
    {S_TermW,      4,    4},
    {S_TermW,      4,    2},
    {S_TermW,      4,    7}
};

static const t4_table_entry_t t4_1d_black_table[8192] =
{
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1792},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   23},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   20},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   25},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  128},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   56},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   30},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1856},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   57},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   21},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   54},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   52},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   48},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2112},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   44},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   36},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  384},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   28},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   60},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   40},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2368},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 1984},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   50},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   34},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1664},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   26},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1408},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   32},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1920},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   61},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   42},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1024},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  768},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   62},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2240},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   46},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   38},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  512},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   19},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   24},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   22},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2496},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_EOL,       12,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1792},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   23},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   20},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   25},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  192},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1280},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   31},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1856},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   58},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   21},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  896},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  640},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   49},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2176},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   45},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   37},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  448},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   29},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1536},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   41},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2432},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2048},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   51},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   35},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  320},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   27},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   59},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   33},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1920},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  256},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   43},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1152},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   55},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   63},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2304},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   47},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   39},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   53},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   19},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   24},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   22},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2560},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1792},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   23},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   20},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   25},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  128},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   56},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   30},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1856},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   57},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   21},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   54},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   52},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   48},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2112},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   44},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   36},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  384},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   28},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   60},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   40},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2368},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 1984},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   50},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   34},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1728},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   26},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1472},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   32},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1920},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   61},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   42},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1088},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  832},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   62},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2240},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   46},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   38},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  576},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   19},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   24},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   22},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2496},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_EOL,       12,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1792},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   23},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   20},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   25},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  192},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1344},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   31},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1856},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   58},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   21},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  960},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13,  704},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   49},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2176},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   45},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   37},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  448},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   29},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1600},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   41},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2432},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   18},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   17},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2048},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   51},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   35},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  320},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   27},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   59},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   33},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    11, 1920},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   12,  256},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   43},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   13, 1216},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      9,   15},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   55},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   63},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2304},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   47},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   39},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     12,   53},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_Null,       1,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   13},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   19},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   24},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     11,   22},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUp,    12, 2560},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   10},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,   16},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,     10,    0},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_MakeUpB,   10,   64},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    9},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   11},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      8,   14},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      6,    8},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      7,   12},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    6},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2},
    {S_TermB,      5,    7},
    {S_TermB,      2,    3},
    {S_TermB,      3,    1},
    {S_TermB,      2,    2},
    {S_TermB,      4,    5},
    {S_TermB,      2,    3},
    {S_TermB,      3,    4},
    {S_TermB,      2,    2}
};

/*- End of file ------------------------------------------------------------*/
