% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/drive_upload.R
\name{drive_upload}
\alias{drive_upload}
\title{Upload into a new Drive file}
\usage{
drive_upload(
  media,
  path = NULL,
  name = NULL,
  type = NULL,
  ...,
  overwrite = NA,
  verbose = deprecated()
)
}
\arguments{
\item{media}{Character, path to the local file to upload.}

\item{path}{Specifies target destination for the new file on Google
Drive. Can be an actual path (character), a file id marked with
\code{\link[=as_id]{as_id()}}, or a \code{\link{dribble}}.

If \code{path} is a shortcut to a folder, it is automatically resolved to its
target folder.

If \code{path} is given as a path (as opposed to a \code{dribble} or an id), it is
best to explicitly indicate if it's a folder by including a trailing
slash, since it cannot always be worked out from the context of the call.
By default, the file is created in the current
user's "My Drive" root folder.}

\item{name}{Character, new file name if not specified as part of
\code{path}. This will force \code{path} to be interpreted as a folder, even if it
is character and lacks a trailing slash. Defaults to the file's local name.}

\item{type}{Character. If \code{type = NULL}, a MIME type is automatically
determined from the file extension, if possible. If the source file is of a
suitable type, you can request conversion to Google Doc, Sheet or Slides by
setting \code{type} to \code{document}, \code{spreadsheet}, or \code{presentation},
respectively. All non-\code{NULL} values for \code{type} are pre-processed with
\code{\link[=drive_mime_type]{drive_mime_type()}}.}

\item{...}{Named parameters to pass along to the Drive API. Has \link[rlang:dyn-dots]{dynamic dots} semantics. You can affect the metadata of the target file by
specifying properties of the Files resource via \code{...}. Read the "Request
body" section of the Drive API docs for the associated endpoint to learn
about relevant parameters.}

\item{overwrite}{Logical, indicating whether to check for a pre-existing file
at the targetted "filepath". The quotes around "filepath" refer to the fact
that Drive does not impose a 1-to-1 relationship between filepaths and files,
like a typical file system; read more about that in \code{\link[=drive_get]{drive_get()}}.
\itemize{
\item \code{NA} (default): Just do the operation, even if it results in multiple
files with the same filepath.
\item \code{TRUE}: Check for a pre-existing file at the filepath. If there is
zero or one, move a pre-existing file to the trash, then carry on. Note
that the new file does not inherit any properties from the old one, such
as sharing or publishing settings. It will have a new file ID. An error is
thrown if two or more pre-existing files are found.
\item \code{FALSE}: Error if there is any pre-existing file at the filepath.
}

Note that existence checks, based on filepath, are expensive operations, i.e.
they require additional API calls.}

\item{verbose}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} This logical argument to
individual googledrive functions is deprecated. To globally suppress
googledrive messaging, use \code{options(googledrive_quiet = TRUE)} (the default
behaviour is to emit informational messages). To suppress messaging in a
more limited way, use the helpers \code{\link[=local_drive_quiet]{local_drive_quiet()}} or
\code{\link[=with_drive_quiet]{with_drive_quiet()}}.}
}
\value{
An object of class \code{\link{dribble}}, a tibble with one row per file.
}
\description{
Uploads a local file into a new Drive file. To update the content or metadata
of an existing Drive file, use \code{\link[=drive_update]{drive_update()}}. To upload or update,
depending on whether the Drive file already exists, see \code{\link[=drive_put]{drive_put()}}.
}
\examples{
\dontshow{if (drive_has_token()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# upload a csv file
chicken_csv <- drive_example_local("chicken.csv") \%>\%
  drive_upload("chicken-upload.csv")

# or convert it to a Google Sheet
chicken_sheet <- drive_example_local("chicken.csv") \%>\%
  drive_upload(
    name = "chicken-sheet-upload.csv",
    type = "spreadsheet"
  )

# check out the new Sheet!
drive_browse(chicken_sheet)

# clean-up
drive_find("chicken.*upload") \%>\% drive_rm()

# Upload a file and, at the same time, star it
chicken <- drive_example_local("chicken.jpg") \%>\%
  drive_upload(starred = "true")

# Is is really starred? YES
purrr::pluck(chicken, "drive_resource", 1, "starred")

# Clean up
drive_rm(chicken)

# `overwrite = FALSE` errors if something already exists at target filepath
# THIS WILL ERROR!
drive_create("name-squatter")
drive_example_local("chicken.jpg") \%>\%
  drive_upload(
    name = "name-squatter",
    overwrite = FALSE
  )

# `overwrite = TRUE` moves the existing item to trash, then proceeds
chicken <- drive_example_local("chicken.jpg") \%>\%
  drive_upload(
    name = "name-squatter",
    overwrite = TRUE
  )

# Clean up
drive_rm(chicken)

\dontrun{
# Upload to a shared drive:
#   * Shared drives are only available if your account is associated with a
#     Google Workspace
#   * The shared drive (or shared-drive-hosted folder) MUST be captured as a
#     dribble first and provided via `path`
sd <- shared_drive_get("Marketing")
drive_upload("fascinating.csv", path = sd)
}
\dontshow{\}) # examplesIf}
}
\seealso{
Wraps the \code{files.create} endpoint:
\itemize{
\item \url{https://developers.google.com/drive/api/v3/reference/files/create}
}

MIME types that can be converted to native Google formats:
\itemize{
\item \url{https://developers.google.com/drive/api/v3/manage-uploads#import_to_google_docs_types}
}
}
