//=======================================================================
// basepkg.h
//-----------------------------------------------------------------------
// This file is part of the package paco
// Copyright (C) 2004-2009 David Rosal
// For more information visit http://paco.sourceforge.net
//=======================================================================

#ifndef LIBPACO_BASE_PKG_H
#define LIBPACO_BASE_PKG_H

#include "common.h"
#include "file.h"
#include <iosfwd>
#include <vector>

namespace Paco
{

class BasePkg : public std::vector<File*>
{
	public:

	// Types and consts
	//-----------------
	static int const NO_FILES			= 0;
	static int const INSTALLED_FILES	= (1 << 0);
	static int const MISSING_FILES		= (1 << 1);
	static int const ALL_FILES			= (INSTALLED_FILES | MISSING_FILES);

	enum ConstructorError { };

	// Inspectors
	//-----------
	std::string const& name() const		{ return mName; }
	std::string const& log() const		{ return mLog; }
	std::string const& confOpts() const	{ return mConfOpts; }
	int date() const					{ return mDate; }
	long sizeInst() const				{ return mSizeInst; }
	long sizeMiss() const				{ return mSizeMiss; }
	long filesInst() const				{ return mFilesInst; }
	long filesMiss() const				{ return mFilesMiss; }
	
	// Public member functions
	//------------------------
	void sort(SortType type = SORT_NAME, bool reverse = false);
	void getFiles(int type = ALL_FILES);
	bool hasFile(File*);

	bool hasFile(std::string const& path)
	{
		File file(path, 0);
		return hasFile(&file);
	}
	
	// Test whether our package shares the @file with any package in @pkgs
	template <typename T>	// T = {Paco,Gpaco}::PkgSet
	bool shares(File* file, T& pkgs)
	{
		for (typename T::iterator p = pkgs.begin(); p != pkgs.end(); ++p) {
			if ((*p)->name() != mName && (*p)->hasFile(file))
				return true;
		}
		return false;
	}

	// Public static member functions
	//-------------------------------
	static bool updateLog(std::string const& log);
	static bool getSize(long&, std::string const&, std::string const& = "");


	protected:

	BasePkg(std::string const&);
	virtual ~BasePkg();

	// Protected member data
	//----------------------
	std::string const mName;
	std::string const mLog;
	int mDate;
	long mSizeInst;
	long mSizeMiss;
	long mFilesInst;
	long mFilesMiss;
	SortType mSortType;
	bool mSortReverse;
	std::string mConfOpts;

	// This nested class is used to sort the files of the package
	class Sorter
	{
		public:
		Sorter(SortType type = SORT_NAME);
		bool operator()(File* left, File* right) const;

		private:
		bool (Sorter::*mSortFunc)(File* , File*) const;
		bool sortByName(File* left, File* right) const;
		bool sortBySize(File* left, File* right) const;

	};	// class Pkg::Sorter

	private:

	// Prevent copies
	BasePkg(BasePkg const&);
	BasePkg& operator=(BasePkg const&);

};	// class Pkg

}	// namespace Paco


#endif  // LIBPACO_BASE_PKG_H
