/***************************************************************************
                          jdxtypes.h  -  description
                             -------------------
    begin                : Sun Jun 6 2004
    copyright            : (C) 2000-2014 by Thies H. Jochimsen
    email                : thies@jochimsen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef JDXTYPES_H
#define JDXTYPES_H


#include <odinpara/jdxbase.h>


/**
  * @addtogroup jcampdx
  * @{
  */

//////////////////////////////////////////////////////////////////

/**
  *
  *  JCAMP-DX class for representing strings
  */
class JDXstring : public STD_string, public virtual JcampDxClass  {

 public:

/**
  *  Default constructor
  */
  JDXstring () : STD_string() {}

/**
  *  Constructor with the following arguments:
  * - ss:            Initial value for the string
  * - name:          The label of the JCAMP-DX parameter
  * - userParameter: Whether this is a user defined JCAMP-DX parameter
  * - mode:          Mode for ASCII representation of strings or arrays of strings
  * - parameter_mode: Mode for GUI accesibility of the parameter
  * - parx_equivalent: Equivalent parameter in PARX to which this parameter will be assigned
  */
  JDXstring (const STD_string& ss,const STD_string& name="",bool userParameter=true,
             compatMode mode=notBroken,parameterMode parameter_mode=edit,
             const STD_string& parx_equivalent="");


/**
  *  Constructs a string of length i with c as their initial content
  */
  JDXstring (int i,const char c=' ') : STD_string(i,c) {}

/**
  *  Copy constructor from a C-string
  */
  JDXstring (const char *charptr) : STD_string(charptr) {}

/**
  *  Copy constructor
  */
  JDXstring (const JDXstring& str) {JDXstring::operator = (str);}

/**
  *  Assigns the string ss to the parameter
  */
  JDXstring& operator = (const STD_string& ss) {STD_string::operator = (ss); return *this;}

/**
  *  Assigns the C-string charptr to the parameter
  */
  JDXstring& operator = (const char *charptr) {STD_string::operator = (charptr); return *this;}

/**
  *  Copy assignment
  */
  JDXstring& operator = (const JDXstring& ss);

/**
  *  Final overrider for stream output
  */
  friend STD_ostream& operator << (STD_ostream& s,const JDXstring& t) {return s << STD_string(t);}


  // overwriting virtual functions from JcampDxClass
  bool parsevalstring (const STD_string& parstring);
  STD_string printvalstring() const;
  STD_string get_parx_code(parxCodeType type, const ParxEquiv& equiv) const;
  ParxEquiv get_parx_equiv() const {parx_equiv.type="char"; return parx_equiv;}
  const char* get_typeInfo() const {return "string";}
  JcampDxClass* create_copy() const {return new JDXstring(*this);}
  STD_string* cast(STD_string*) {return this;}

 private:
  mutable ParxEquiv parx_equiv;

};



//////////////////////////////////////////////////////////////////

/**
  *
  *  JCAMP-DX class for representing Boolean values
  */
class JDXbool : public virtual JcampDxClass  {

 public:

/**
  * Default constructor
  */
  JDXbool () : val(false) {}

/**
  *  Constructor with the following arguments:
  * - flag:          Initial value for the Boolean value
  * - name:          The label of the JCAMP-DX parameter
  * - userParameter: Whether this is a user defined JCAMP-DX parameter
  * - mode:          Mode for ASCII representation of strings or arrays of strings
  * - parameter_mode: Mode for GUI accesibility of the parameter
  * - parx_equivalent: Equivalent parameter in PARX to which this parameter will be assigned
  */
  JDXbool(bool flag, const STD_string& name="",
          bool userParameter=true, compatMode mode=notBroken,
          parameterMode parameter_mode=edit,
          const STD_string& parx_equivalent="");

/**
  * Copy constructor
  */
  JDXbool(const JDXbool& jb) {JDXbool::operator = (jb);}

/**
  * Assigns the value of flag to the parameter
  */
  JDXbool& operator = (bool flag) {val=flag; return *this;}

/**
  * Assigns the value of s to the parameter, s may contain "yes" or "true"
  * (upper- or lowercase) to indicate a value of true, otherwise false
  */
  JDXbool& operator = (const STD_string& s) {parsevalstring(s); return *this;}

/**
  * Copy assignment
  */
  JDXbool& operator = (const JDXbool& jb);

/**
  * type conversion operator of the current value of the parameter
  */
  operator bool () const {return val;}

  // overwriting virtual functions from JcampDxClass
  bool parsevalstring (const STD_string& parstring);
  STD_string printvalstring() const;
  STD_string get_parx_code(parxCodeType type, const ParxEquiv& equiv) const;
  ParxEquiv get_parx_equiv() const {parx_equiv.type="YesNo"; return parx_equiv;}
  const char* get_typeInfo() const {return "bool";}
  JcampDxClass* create_copy() const {return new JDXbool(*this);}
  bool* cast(bool*) {return &val;}

 private:
  bool val;
  mutable ParxEquiv parx_equiv;
};


//////////////////////////////////////////////////////////////////

/**
  *
  *  JCAMP-DX class to represent enumerations
  */
class JDXenum : public virtual JcampDxClass  {

 public:

/**
  * Default constructor
  */
  JDXenum() {actual=entries.end();}

/**
  *  Constructor with the following arguments:
  * - first_entry:   Initial value for the first entry of the enumeration
  * - name:          The label of the JCAMP-DX parameter
  * - userParameter: Whether this is a user defined JCAMP-DX parameter
  * - mode:          Mode for ASCII representation of strings or arrays of strings
  * - parameter_mode: Mode for GUI accesibility of the parameter
  * - parx_equivalent: Equivalent parameter in PARX to which this parameter will be assigned
  */
  JDXenum(const STD_string& first_entry, const STD_string& name="",
          bool userParameter=true, compatMode mode=notBroken,
          parameterMode parameter_mode=edit, const STD_string& parx_equivalent="");

/**
  * Copy constructor
  */
  JDXenum(const JDXenum& je) {JDXenum::operator = (je);}


/**
  * Sets the current value of the enumeration to the given item
  */
  JDXenum& operator = (const char* item) {set_actual(STD_string(item)); return *this;}

/**
  * Sets the current value of the enumeration to the given item
  */
  JDXenum& operator = (const STD_string& item) {set_actual(item); return *this;}

/**
  * Sets the current value of the enumeration to the given item
  */
  JDXenum& operator = (int item) {set_actual(item); return *this;}

/**
  * Assignment operator
  */
  JDXenum& operator = (const JDXenum& je);

/**
  * Appends an item to the the list of items in the enumeration.
  * If index is non-negative, the item will be inserted at the
  * position indicated by index.
  */
  JDXenum& add_item(const STD_string& item, int index=-1);

/**
  * Sets the current value of the enumeration to the given item
  */
  JDXenum& set_actual(const STD_string& item);

/**
  * Sets the current value of the enumeration to the given item
  */
  JDXenum& set_actual(int index);

/**
  * Clears the list of items in the enumeration
  */
  JDXenum& clear();

/**
  * type conversion operator of the current value of the enumeration
  */
  operator int () const;

/**
  * type conversion operator of the current value of the enumeration
  */
  operator STD_string () const;

/**
  * Compares the current value with s for equality
  */
  bool operator == (const STD_string& s) const {return (operator STD_string ())==s;}

/**
  * Compares the current value with s for equality
  */
  bool operator == (const char* s) const {return (operator STD_string ())==STD_string(s);}

/**
  * Compares the current index with i for equality
  */
  bool operator == (int i) const {return (operator int ())==i;}

/**
  * Compares the current value with s for inequality
  */
  bool operator != (const STD_string& s) const {return (operator STD_string ())!=s;}

/**
  * Compares the current value with s for inequality
  */
  bool operator != (const char* s) const {return (operator STD_string ())!=STD_string(s);}

/**
  * Compares the current index with i for inequality
  */
  bool operator != (int i) const {return (operator int ())!=i;}


/**
  * Returns the number of items in the enumeration
  */
  unsigned int n_items() const {return entries.size();}

/**
  * Returns the item at the position 'index' in the label-index map
  */
  const STD_string& get_item(unsigned int index) const;

/**
  * Returns the current position in the label-index map
  */
  unsigned int get_item_index() const;

/**
  * Sets the current position 'index' in the label-index map
  */
  JDXenum& set_item_index(unsigned int index);


  // overwriting virtual functions from JcampDxClass
  bool parsevalstring (const STD_string& parstring);
  STD_string printvalstring() const;
  svector get_alternatives() const;
  STD_string get_parx_code(parxCodeType type, const ParxEquiv& equiv) const;
  ParxEquiv get_parx_equiv() const;
  const char* get_typeInfo() const {return "enum";}
  JcampDxClass* create_copy() const {return new JDXenum(*this);}
  JDXenum* cast(JDXenum*) {return this;}

 private:
  STD_map<int,STD_string> entries;
  STD_map<int,STD_string>::const_iterator actual;
  mutable ParxEquiv parx_equiv;

  STD_string parxtype_cache;

};

//////////////////////////////////////////////////////////////////

/**
  *
  *  JCAMP-DX class to trigger actions from the GUI
  */
class JDXaction : public virtual JcampDxClass  {

 public:

/**
  * Default constructor
  */
  JDXaction() : state(false) {set_filemode(exclude);}

/**
  *  Constructor with the following arguments:
  * - init_state:    Initial value for the action
  * - name:          The label of the JCAMP-DX parameter
  * - userParameter: Whether this is a user defined JCAMP-DX parameter
  * - mode:          Mode for ASCII representation of strings or arrays of strings
  * - parameter_mode: Mode for GUI accesibility of the parameter
  */
  JDXaction(bool init_state, const STD_string& name="",
            bool userParameter=true, compatMode mode=notBroken,
            parameterMode parameter_mode=edit);

/**
  * Copy constructor
  */
  JDXaction(const JDXaction& ja) {JDXaction::operator = (ja);}

/**
  * Copy assignment
  */
  JDXaction& operator = (const JDXaction& ja);

/**
  * Returns whether an action should be triggered and resets the action flag
  */
  operator bool () const;

/**
  * After calling this function, the next type conversion to bool will return true
  */
  JDXaction& trigger_action() {state=true; return *this;}
  
  // overwriting virtual functions from JcampDxClass
  bool parsevalstring (const STD_string& parstring);
  STD_string printvalstring() const;
  const char* get_typeInfo() const {return "action";}
  JcampDxClass* create_copy() const {return new JDXaction(*this);}
  JDXaction* cast(JDXaction*) {return this;}

 private:
  mutable bool state;
};




//////////////////////////////////////////////////////////////////


/**
  *
  * JCAMP-DX class for representing file names. Besides using JDXfileName
  * as a convenient way within the UI to display/retrieve/browse file names,
  * it can be used to analyze file paths, for example in the following way:
  *   \verbatim
  STD_string myfile="/somedir/anotherdir/file.txt";

  STD_string myfile_base=JDXfileName(myfile).get_basename();
  STD_cout << "myfile_base=" << myfile_base << STD_endl;

  STD_string myfile_base_nosuffix=JDXfileName(myfile).get_basename_nosuffix();
  STD_cout << "myfile_base_nosuffix=" << myfile_base_nosuffix << STD_endl;

  STD_string myfile_dir= JDXfileName(myfile).get_dirname();
  STD_cout << "myfile_dir=" << myfile_dir << STD_endl;
      \endverbatim
  *
  */
class JDXfileName : public JDXstring {

public:

/**
  * Default constructor
  */
  JDXfileName () {common_init();}

/**
  *  Constructor with the following arguments:
  * - filename:      Initial value for the file name
  * - name:          The label of the JCAMP-DX parameter
  * - userParameter: Whether this is a user defined JCAMP-DX parameter
  * - mode:          Mode for ASCII representation of strings or arrays of strings
  * - parameter_mode: Mode for GUI accesibility of the parameter
  */
  JDXfileName (const STD_string& filename, const STD_string& name="",
               bool userParameter=true, compatMode mode=notBroken,
               parameterMode parameter_mode=edit);

/**
  * Copy constructor
  */
  JDXfileName(const JDXfileName& jf);

/**
  * Assignment from a string
  */
  JDXfileName& operator = (const STD_string& filename);

/**
  * Assignment operator
  */
  JDXfileName& operator = (const JDXfileName& jf);

/**
  * Returns 'true' only if the file/directory exists
  */
  bool exists() const;

/**
  * Returns the filename without preceeding directories
  */
  STD_string get_basename() const {return basename_cache;}

/**
  * Returns the filename without preceeding directories and without
  * a previously specified suffix (file extension)
  */
  STD_string get_basename_nosuffix() const;

/**
  * Returns the directory of the filename
  */
  STD_string get_dirname() const {return dirname_cache;}

/**
  * Returns the suffix (file extension) of the filename
  */
  STD_string get_suffix() const {return suffix_cache;}

/**
  * Sets/overwrites the suffix (file extension) of the filename
  */
  JDXfileName& set_suffix(const STD_string& suff) {suffix_cache=suff; return *this;}

/**
  * Returns the default location of the filename
  */
  STD_string get_defaultdir() const {return defaultdir;}

/**
  * Sets the default location of the filename
  */
  JDXfileName& set_defaultdir(const STD_string& defdir);


/**
  * Returns whether used exclusively for directory names
  */
  bool is_dir() const {return dir;}

/**
  * Specifies whether used exclusively for directory names
  */
  JDXfileName& set_dir(bool flag) {dir=flag; return *this;}


  // overwriting virtual functions from JcampDxClass
  bool parsevalstring (const STD_string& parstring);
  const char* get_typeInfo() const {return "fileName";}
  JcampDxClass* create_copy() const {return new JDXfileName(*this);}
  JDXfileName* cast(JDXfileName*) {return this;}

private:

  static void normalize(const STD_string& fname, bool dir, STD_string& result, STD_string& result_dirname, STD_string& result_basename, STD_string& result_suffix);

  void common_init() {dir=false;}

  STD_string defaultdir;
  STD_string dirname_cache;
  STD_string basename_cache;
  STD_string suffix_cache;

  bool dir;

};

//////////////////////////////////////////////////////////////////

/**
  *
  *  JCAMP-DX class for representing mathematic formulas
  */
class JDXformula : public JDXstring {

public:

/**
  * Default constructor
  */
  JDXformula () : JDXstring() {}

/**
  *  Constructor with the following arguments:
  * - formula:      Initial value for the formula
  * - name:          The label of the JCAMP-DX parameter
  * - userParameter: Whether this is a user defined JCAMP-DX parameter
  * - mode:          Mode for ASCII representation of strings or arrays of strings
  * - parameter_mode: Mode for GUI accesibility of the parameter
  */
  JDXformula (const STD_string& formula, const STD_string& name="",
              bool userParameter=true, compatMode mode=notBroken,
              parameterMode parameter_mode=edit);

/**
  * Copy constructor
  */
  JDXformula (const JDXformula& jf) {JDXformula::operator = (jf);}

/**
  * Assignment from a formula string
  */
  JDXformula& operator = (const STD_string& formula) {JDXstring::operator = (formula); return *this;}

/**
  * Copy assignment
  */
  JDXformula& operator = (const JDXformula& jf);

/**
  * Sets a string describing the formulas syntax
  */
  JDXformula& set_syntax(const STD_string& syn) {syntax=syn; return *this;}

/**
  * Returns a string describing the formulas syntax
  */
  STD_string get_syntax() const {return syntax;}

  // overwriting virtual functions from JcampDxClass
  const char* get_typeInfo() const {return "formula";}
  JcampDxClass* create_copy() const {return new JDXformula(*this);}
  JDXformula* cast(JDXformula*) {return this;}

private:
  STD_string syntax;
};



/** @}
  */

#endif

