/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache;

import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.ConfigurationException;
import org.jboss.cache.factories.ComponentFactory;
import org.jboss.cache.factories.ComponentRegistry;
import org.jboss.cache.factories.XmlConfigurationParser;
import org.jboss.cache.invocation.CacheInvocationDelegate;

import java.io.InputStream;

/**
 * Default implementation of the {@link org.jboss.cache.CacheFactory} interface.
 * <p/>
 * This is a special instance of a {@link ComponentFactory} which contains bootstrap information for the
 * {@link org.jboss.cache.factories.ComponentRegistry}.
 * <p/>
 * In JBoss Cache 2.0.x, this was a singleton and you had to use {@link #getInstance()} to obtain an instance.  From
 * JBoss Cache 2.1.x onwards, this is no longer a singleton and you can use the default no-arg constructor to obtain
 * a reference.  {@link #getInstance()} has been deprecated and modified to return a new instance of this class.
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 * @see org.jboss.cache.factories.ComponentFactory
 */
public class DefaultCacheFactory<K, V> extends ComponentFactory implements CacheFactory<K, V>
{
   private ClassLoader defaultClassLoader;

   /**
    * Note - this method used to return a singleton instance, and since 2.1.0 returns a new instance.  The method is
    * deprecated and you should use the no-arg constructor to create a new instance of this factory.
    *
    * @return a NEW instance of this class.
    */
   @SuppressWarnings("unchecked")
   @Deprecated
   public static <K, V> CacheFactory<K, V> getInstance()
   {
      return new DefaultCacheFactory();
   }

   public Cache<K, V> createCache() throws ConfigurationException
   {
      return createCache(true);
   }

   public Cache<K, V> createCache(boolean start) throws ConfigurationException
   {
      return createCache(new Configuration(), start);
   }

   public Cache<K, V> createCache(String configFileName) throws ConfigurationException
   {
      return createCache(configFileName, true);
   }

   public Cache<K, V> createCache(String configFileName, boolean start) throws ConfigurationException
   {
      XmlConfigurationParser parser = new XmlConfigurationParser();
      Configuration c = parser.parseFile(configFileName);
      return createCache(c, start);
   }

   /**
    * This implementation clones the configuration passed in before using it.
    *
    * @param configuration to use
    * @return a cache
    * @throws ConfigurationException if there are problems with the cfg
    */
   public Cache<K, V> createCache(Configuration configuration) throws ConfigurationException
   {
      return createCache(configuration, true);
   }

   /**
    * This implementation clones the configuration passed in before using it.
    *
    * @param configuration to use
    * @param start         whether to start the cache
    * @return a cache
    * @throws ConfigurationException if there are problems with the cfg
    */
   public Cache<K, V> createCache(Configuration configuration, boolean start) throws ConfigurationException
   {
      try
      {
         CacheSPI<K, V> cache = createAndWire(configuration);
         if (start) cache.start();
         return cache;
      }
      catch (ConfigurationException ce)
      {
         throw ce;
      }
      catch (RuntimeException re)
      {
         throw re;
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   protected CacheSPI<K, V> createAndWire(Configuration configuration) throws Exception
   {
      CacheSPI<K, V> spi = new CacheInvocationDelegate<K, V>();
      bootstrap(spi, configuration);
      return spi;
   }

   /**
    * Bootstraps this factory with a Configuration and a ComponentRegistry.
    */
   private void bootstrap(CacheSPI spi, Configuration configuration)
   {
      // injection bootstrap stuff
      componentRegistry = new ComponentRegistry(configuration, spi);
      componentRegistry.registerDefaultClassLoader(defaultClassLoader);
      this.configuration = configuration;

      componentRegistry.registerComponent(spi, CacheSPI.class);
   }

   /**
    * Allows users to specify a default class loader to use for both the construction and running of the cache.
    *
    * @param loader class loader to use as a default.
    */
   public void setDefaultClassLoader(ClassLoader loader)
   {
      this.defaultClassLoader = loader;
   }

   public Cache<K, V> createCache(InputStream is) throws ConfigurationException
   {
      XmlConfigurationParser parser = new XmlConfigurationParser();
      Configuration c = parser.parseStream(is);
      return createCache(c);
   }

   public Cache<K, V> createCache(InputStream is, boolean start) throws ConfigurationException
   {
      XmlConfigurationParser parser = new XmlConfigurationParser();
      Configuration c = parser.parseStream(is);
      return createCache(c, start);
   }

   @Override
   protected <T> T construct(Class<T> componentType)
   {
      throw new UnsupportedOperationException("Should never be invoked - this is a bootstrap factory.");
   }
}
