/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quarter@users.sourceforge.net> */

#include "include/support.h"
#include "include/paritycommand.h"
#include "include/message.h"

class LWCommandBreak: public LWCommandSegment {

private:
LWSymbol *matchedSymbol;
	
public:

LWCommandBreak(): matchedSymbol(NULL) {};
		
LWValue *execute1 (LWContext *context, guint argc, LWValue *args[])
{
	(void) context;
	(void) argc;
	(void) args;

	enum BreakMode {
		LW_BREAK_FOR,
		LW_BREAK_REPEAT,
		LW_BREAK_WHILE
	} bm;

	if (!g_ascii_strcasecmp (matchedSymbol->getName(), "to") ||
			!g_ascii_strcasecmp (matchedSymbol->getName(), "downto"))
				bm = LW_BREAK_FOR;
	else
	if (!g_ascii_strcasecmp (matchedSymbol->getName(), "repeat"))
		bm = LW_BREAK_REPEAT;
	else 
	if (!g_ascii_strcasecmp (matchedSymbol->getName(), "while"))
		bm = LW_BREAK_WHILE;
	else
		g_return_val_if_reached (NULL);

	for (GNode *n = context->instrPtr->parent; !G_NODE_IS_ROOT (n); n = n->parent) {
		LWPiece *piece = (LWPiece *) n->data;

		if (n->prev != NULL && !g_ascii_strcasecmp (piece->getSymbol()->getName(), "begin"))
			piece = (LWPiece *) n->prev->data;		
		
		if (matchedSymbol == piece->getSymbol()) {

			switch (bm) {

			case LW_BREAK_FOR:
				g_queue_pop_head(context->stack);
				context->instrPtr = n->next;
				break;

			case LW_BREAK_REPEAT:
				context->instrPtr = n->next->next;
				break;

			case LW_BREAK_WHILE:
				context->instrPtr = n->next;

			}
		
			//When null, set correct command to execute, otherwise interpeter
			//will get parent of current command (break), and that is probably
			//not we want...
			if (context->instrPtr == NULL)
				context->instrPtr =  n->parent;
			
			return NULL;
		}
	}
	
	g_return_val_if_reached (NULL);
}

const gchar *getName ()
{
	return "break";
}	

gchar *getHint ()
{
	return _("Break the loop");
}

void checkArgc (guint n)
{
	if (n != 0) 
		throw new LWMessage (LW_ERROR_WrongNumberOfArguments);
}

gboolean matchPrevCondition (LWCommand *cmd)
{
	return (cmd->isLoop() || !g_ascii_strcasecmp (cmd->getName(), "repeat")) ? TRUE : FALSE;
}

gboolean matchWith (LWCommand *cmd)
{
	if (cmd == NULL && matchedSymbol == NULL)
			throw new LWMessage (LW_ERROR_BreakNotInALoop);

	if (matchedSymbol == NULL) {
		matchedSymbol = (LWSymbol *) cmd;	
		match();
	}
	
	return FALSE;
}

void reset1 ()
{
	matchedSymbol = NULL;
	LWCommandSegment::reset1();
}

gboolean isSegment ()
{
	return (FALSE == isAlreadyMatched()) ? TRUE : FALSE;
}

};

LWSymbol *new_LWCommandBreak()
{
	return new LWCommandBreak();
}
